//Bodged by Simon Jansen. Provided as is.
//Versioning:
#define VERSION "2025-05-28_Stable"

#include "arduino_secrets.h"
#include "I2Cdev.h"
#include "MPU6050_6Axis_MotionApps612.h"
#include "Wire.h"
#include <ESP8266WiFi.h>    //For ESP8266
#include <ESP8266mDNS.h>    //For OTA
#include <WiFiUdp.h>        //For OTA
#include <Ticker.h>         //For MQTT
#include <AsyncMqttClient.h>//For MQTT
#include <ArduinoOTA.h>     //For OTA
#include <ArduinoJson.h>
#include <Extras.h>
Extras SketchExtras;
#include <Roomba632.h>
Roomba632 roomba;

//upload to
//Testplatform: 80e74a
//Roomba:       84ed23
//Uncomment to use all parameters for testbed
//#define TEST
uint32_t ESPFreeHeap;
uint8_t ESPHeapFragmentation;

//IMU configuration
// class default I2C address is 0x68
MPU6050 mpu;

#define INTERRUPT_PIN 15  // use pin 2 on Arduino Uno & most boards
// MPU control/status vars
bool dmpReady = false;  // set true if DMP init was successful
uint8_t mpuIntStatus;   // holds actual interrupt status byte from MPU
uint8_t devStatus;      // return status after each device operation (0 = success, !0 = error)
uint16_t packetSize;    // expected DMP packet size (default is 42 bytes)
uint16_t fifoCount;     // count of all bytes currently in FIFO
uint8_t fifoBuffer[64]; // FIFO storage buffer

// orientation/motion vars
Quaternion q;           // [w, x, y, z]         quaternion container
VectorInt16 aa;         // [x, y, z]            accel sensor measurements
VectorInt16 gy;         // [x, y, z]            gyro sensor measurements
VectorInt16 aaReal;     // [x, y, z]            gravity-free accel sensor measurements
VectorInt16 aaWorld;    // [x, y, z]            world-frame accel sensor measurements
VectorFloat gravity;    // [x, y, z]            gravity vector
float euler[3];         // [psi, theta, phi]    Euler angle container
float ypr[3];           // [yaw, pitch, roll]   yaw/pitch/roll container and gravity vector
float realYaw, yawOffset;
//INTERRUPT DETECTION ROUTINE
volatile bool mpuInterrupt = false;     // indicates whether MPU interrupt pin has gone high
void IRAM_ATTR dmpDataReady() {
  mpuInterrupt = true;
}
//INITIAL SETUP IMU
#define SDA_PIN 2
#define SCL_PIN 14
const int16_t I2C_MASTER = 0x42;

//OTA configuration
const char * sketchpass = SECRET_SKETCHPASS;
#ifdef TEST
  const char* host = "TEST"; //80e74a
#else
  const char* host = "Roomba632"; //84ed23
#endif

// MQTT configuration
#define HANAME "homeassistant"
#define DEVICENAME "Roomba632"
#ifdef TEST
  #define DEVICEID "XX"
#else
  #define DEVICEID "01"
#endif
#define MANUFACTURER "iRobot"
#define MODEL "632"

#define MQTT_SUB_TOPIC "homeassistant/device/roomba/set"
#define MQTT_PUB_TOPIC "homeassistant/device/roomba/state"
#define MQTT_STREAM_TOPIC "homeassistant/device/roomba/stream"
#define MQTT_POSE_TOPIC "homeassistant/device/roomba/pose"
#define MQTT_RAW_TOPIC "homeassistant/device/roomba/raw"

#define MQTT_PUB_MESSAGE "MQTT online "
#define MQTT_KEEPALIVE 15 //Default is 15s

String mqtt_client_id = host;  //This text is concatenated with ChipId to get unique client_id
//unsigned long lastReconnectAttempt = 0;

unsigned long streamTimer = 0;
unsigned long positionTimer = 0;
unsigned long rawTimer = 0;
unsigned long debugTimer = 0;
unsigned long debugLoops = 0;
unsigned long debugSum = 0;
unsigned int debugCounter = 0;
unsigned long debugVal = 0;

#ifdef TEST
  #define DOCKEDSTREAMINTERVAL 1000 //1 sec
  #define UNDOCKEDSTREAMINTERVAL 500 //500 ms
  #define POSINTERVAL 5000 //5 sec
  #define RAWINTERVAL 500 //500 ms
#else
  #define DOCKEDSTREAMINTERVAL 60000 //1mins
  #define UNDOCKEDSTREAMINTERVAL 2000 // 1 sec
  #define POSINTERVAL 503 //500 ms
  #define RAWINTERVAL 101 //100 ms
#endif

/*
  RMHalt,   //3 scriptable
  RMDriveF, //5 scriptable verb + noun steps of 25mm
  RMDriveR, //7 scriptable verb + noun steps of 25mm
  RMTurnL,  //9 scriptable verb + noun full circle = 255
  RMTurnR,  //11 scriptable verb + noun full circle = 255
  RMWait,   //13 scriptable verb + noun ticks of 128ms
*/
const uint8_t Square[] PROGMEM = {
  RMDriveF, 20, //50cm
  RMTurnL, 64,  //quarter turn L
  RMDriveF, 20, //50cm
  RMTurnL, 64,  //quarter turn L
  RMDriveF, 20, //50cm
  RMTurnL, 64,  //quarter turn L
  RMDriveF, 20, //50cm
  RMTurnL, 64,  //quarter turn L
  RMDriveF, 20, //50cm
  RMTurnR, 64,  //quarter turn R
  RMDriveF, 20, //50cm
  RMTurnR, 64,  //quarter turn R
  RMDriveF, 20, //50cm
  RMTurnR, 64,  //quarter turn R
  RMDriveF, 20, //50cm
  RMTurnR, 64,  //quarter turn R
  RMDriveF, 20, //50cm
  RMHalt        //Stop
};

const uint8_t Empty[] PROGMEM = {
  //RMDriveRawBL, 50, 50, 32, //Velocity: reverse 150, Arc: Left 500mm, Time 32*128ms=4sec
  RMDriveR, 40,
  RMTurnL, 64,
  RMDriveF, 16,
  RMTurnL, 128,
  RMHalt
};

const uint8_t CallibrationRun[] PROGMEM = {
  //RMDriveRawBL, 50, 50, 32, //Velocity: reverse 150, Arc: Left 500mm, Time 32*128ms=4sec
  RMDriveR, 40,
  RMTurnL, 128,
  RMTurnL, 128,
  RMTurnR, 128,
  RMTurnR, 128,
  RMDriveF, 16,
  RMHalt
};

AsyncMqttClient mqttClient;
Ticker mqttReconnectTimer;

WiFiEventHandler wifiConnectHandler;
WiFiEventHandler wifiDisconnectHandler;
Ticker wifiReconnectTimer;

//Connecting to Wi-Fi...
void connectToWifi() {
  WiFi.begin(SECRET_WIFI_SSID, SECRET_WIFI_PASSWORD);
}
//Connected to Wi-Fi
void onWifiConnect(const WiFiEventStationModeGotIP& event) {
  //connectToMqtt();
  mqttReconnectTimer.once(2, connectToMqtt); //Wait 2 seconds, then attempt to connect to MQTT-broker once)
}
//Disconnected from Wi-Fi
void onWifiDisconnect(const WiFiEventStationModeDisconnected& event) {
  mqttReconnectTimer.detach(); // ensure we don't reconnect to MQTT while reconnecting to Wi-Fi
  wifiReconnectTimer.once(2, connectToWifi); //Wait 2 seconds, then attempt to connect to WiFi once)
}

void setup() {
  // join I2C bus (I2Cdev library doesn't do this automatically)
  Wire.begin(SDA_PIN, SCL_PIN, I2C_MASTER);
  Wire.setClock(100000); // 100kHz I2C clock
  mpu.initialize();
  pinMode(INTERRUPT_PIN, INPUT);
  devStatus = mpu.dmpInitialize();
  // supply your own gyro offsets here, scaled for min sensitivity
  #ifdef TEST
    mpu.setXGyroOffset(164); //testbed:164
    mpu.setYGyroOffset(-37); //testbed:-37
    mpu.setZGyroOffset(-11); //testbed:-11
    mpu.setXAccelOffset(-2886); //testbed:-2886
    mpu.setYAccelOffset(-1903); //testbed:-1903
    mpu.setZAccelOffset(872); //testbed:872
  #else
    mpu.setXGyroOffset(83); //Roomba:83
    mpu.setYGyroOffset(3); //Roomba:3
    mpu.setZGyroOffset(-8); //Roomba:-8
    mpu.setXAccelOffset(-273); //Roomba:-273
    mpu.setYAccelOffset(-2933); //Roomba:-2933
    mpu.setZAccelOffset(5274); //Roomba:5274
  #endif
  // make sure it worked (returns 0 if so)
  if (devStatus == 0) {
    // Calibration Time: generate offsets and calibrate our MPU6050
    mpu.CalibrateAccel(7);
    mpu.CalibrateGyro(7);
    //mpu.PrintActiveOffsets();
    // turn on the DMP, now that it's ready
    mpu.setDMPEnabled(true);

    // enable Arduino interrupt detection
    attachInterrupt(digitalPinToInterrupt(INTERRUPT_PIN), dmpDataReady, RISING);
    mpuIntStatus = mpu.getIntStatus();

    // set our DMP Ready flag so the main loop() function knows it's okay to use it
    //Serial.println(F("DMP ready! Waiting for first interrupt..."));
    dmpReady = true;

    // get expected DMP packet size for later comparison
    packetSize = mpu.dmpGetFIFOPacketSize();
  } else {
    // ERROR!
    // 1 = initial memory load failed
    // 2 = DMP configuration updates failed
    // (if it's going to break, usually the code will be 1)
    //Serial.print(F("DMP Initialization failed (code "));
    //Serial.print(devStatus);
    //Serial.println(F(")"));
  }
  wifiConnectHandler = WiFi.onStationModeGotIP(onWifiConnect);
  wifiDisconnectHandler = WiFi.onStationModeDisconnected(onWifiDisconnect);
  #ifdef TEST
    ArduinoOTA.setHostname("Roomba_TEST");
  #else
    ArduinoOTA.setHostname("Roomba632_01");
  #endif
  ArduinoOTA.onStart([]() { //Make sure device is in a safe mode
    //store params to EEPROM?
  });
  ArduinoOTA.onEnd([]() { //Device will reboot

  });
  ArduinoOTA.onError([](ota_error_t error) {
    (void)error;
    ESP.restart();
  });
  ArduinoOTA.begin();
  mqttClient.onConnect(onMqttConnect);
  mqttClient.onDisconnect(onMqttDisconnect);
  mqttClient.onSubscribe(onMqttSubscribe);
  mqttClient.onUnsubscribe(onMqttUnsubscribe);
  mqttClient.onMessage(onMqttMessage);
  mqttClient.onPublish(onMqttPublish);
  mqttClient.setServer(SECRET_MQTT_HOST, SECRET_MQTT_PORT);
  mqttClient.setCredentials(SECRET_MQTT_USER, SECRET_MQTT_PASSWORD);
  //mqttClient.setKeepAlive(MQTT_KEEPALIVE);

  connectToWifi();
  // It takes a few seconds to connect to WiFi and MQTT
  //mqttClient.publish(MQTT_PUB_TOPIC, 0, false, "ESP8266-Reset");
  roomba.setCallbackEndRun(onEndRun);
  roomba.setCallbackStartRun(onStartRun);
  roomba.setCallbackStateChange(onRoombaStateChange);
  roomba.setCallbackError(onRoombaError);
}
void IMUHandler() {
  if (!dmpReady) return;
  // read a packet from FIFO
  if (mpuInterrupt) {
    mpuInterrupt = false; //Clear interrupt flag
    if (mpu.dmpGetCurrentFIFOPacket(fifoBuffer)) { // Get the Latest packet
      mpu.dmpGetQuaternion(&q, fifoBuffer);
      mpu.dmpGetGravity(&gravity, &q);
      mpu.dmpGetYawPitchRoll(ypr, &q, &gravity);
      realYaw = ypr[0] - yawOffset;
      if (realYaw > 6.28){
        realYaw -= 6.28;
      }
      else if(realYaw < -6.28){
        realYaw += 6.28;
      }
      roomba.AuxYaw = realYaw;
      //roomba.AuxYaw = ypr[0];
      
      mpu.dmpGetQuaternion(&q, fifoBuffer);
      mpu.dmpGetGravity(&gravity, &q);
      mpu.dmpGetAccel(&aa, fifoBuffer);
      mpu.dmpGetLinearAccel(&aaReal, &aa, &gravity);
      roomba.AuxXacc = aaReal.x;
      roomba.AuxYacc = aaReal.y;
    }
  }
}

void loop() {
  ArduinoOTA.handle();
  IMUHandler();
  roomba.handler();
  SketchExtras.Update();
  
  if (roomba.roombaState < roombaCleaning) {
    if ((millis() - streamTimer) > DOCKEDSTREAMINTERVAL) {
      streamTimer =  millis();
      MQTT_UpdateSensors();
    }
  }
  else {
    if ((millis() - streamTimer) > UNDOCKEDSTREAMINTERVAL) {
      streamTimer =  millis();
      MQTT_UpdateSensors();
    }
    if ((millis() - positionTimer) > POSINTERVAL) {
      positionTimer =  millis();
      MQTT_UpdatePositionLog();
    }
    if ((millis() - rawTimer) > RAWINTERVAL) {
      rawTimer =  millis();
      MQTT_UpdateRawLog();
    }
  }
  //Filtering of LoopsPerSec to avoid snapshots when displaying every minute vs every second
  if ((millis() - debugTimer) > 1000) {
    debugTimer =  millis();
    debugCounter ++;
    if (debugCounter ==1){
      debugSum = SketchExtras.LoopsPerSec;
    }
    else if(debugCounter > 16){
      debugCounter = 16;
      debugSum = debugSum - (debugSum/debugCounter) + SketchExtras.LoopsPerSec;  
    }
    else{
      debugSum =+ SketchExtras.LoopsPerSec;
    }
    debugLoops = (debugSum/debugCounter);
  }
}
