// Roomba632.cpp
// By Simon Jansen 2022-08-21 "Remember the pandemic?"
// Changlog:
// Latest 2025-05-05: 
//	- Added callback for start cleaning;
//	- Reset Roomba on error in comm;
//	- Added callback for critical error / cry for help;
//	- Check OI-mode before starting OI or upgrading mode;
// 	- Improve cleaning functions without mode-set;
//	- More sensordata in datastream;
//	- Improve state machine for reading charging message;

#include "Roomba632.h"

Roomba632::Roomba632(){
	//Publicaly set variables
	//music song to be played
	//driving instructions

	//Privately set public variables
	chargingState = chargingStateNotCharging;
	batteryVoltage = 15000;
	current = 0;
	batteryTemp = 20;
	batteryCharge = 0;
	batteryCapacity = 0;
	docked = true;
	OImode = Off; //enum for states
	encoderLeft = 0;
	encoderRight = 0;
	bumpRight = false;
	bumpLeft = false;
	wheelDropRight = false;
	wheelDropLeft = false;
	cliffLeft = 0;
	cliffFrontLeft = 0;
	cliffFrontRight = 0;
	cliffRight = 0;
	sideBrushOvercurrent = false;
	mainBrushOvercurrent = false;
	rightWheelOvercurrent = false;
	leftWheelOvercurrent = false;
	ltBumperLeft = false;
	ltBumperFrontLeft = false;
	ltBumperCenterLeft = false;
	ltBumperCenterRight = false;
	ltBumperFrontRight = false;
	ltBumperRight = false;

	angle = 0; //calculated from encoders
	distance = 0; //calculated from encoders
	debugVal = 99; //resetvalue
	Xcoo = 0;
	Ycoo = 0;
	theta = 0;
	AuxYaw = 0;
	AuxXacc = 0;
	AuxYacc = 0;


	//Privately set public states. Only set and change states from within state machine.
	roombaState = roombaHome;//roombaIdle; //enum for states
	dataState = dataIdle;
	p_undockedTimer = millis();
	//p_odometryTimer = millis();
	//p_requestStartDatastreamFlag = false;
	//int receivingDatastreamState = datastreamIdle; //enum for states
	//int sendingCommands = sendingCommandsIdle; //enum for states

	//Privately set public flags
	b_songPlaying = false;
	b_dirtDetected = false;
	//sensorTripped = false; //for bump cliff and wheel drop sensors

	//Private flags and variables
	//p_chargingFlag = false;
	p_cleanFlag = false;
	p_dockFlag = false;
	p_spotFlag = false;
	p_stopFlag = false;
	p_startScriptFlag = false;
	p_playMusicFlag = false;
	p_pressToPause = false;
	//p_driveFlag = false;

	p_encodersReady = false;

	//Private counters and timers
	//p_musicTimer = 0; //could be static?

	//Initialize serial connection to Roomba
	Serial.begin(115200);
}
//public functions for change in state set a flag (and vairables) as a 'request' whitch is handled in order of prirority
void Roomba632::stop(){
	SendStopCommand();
	p_stopFlag = true;
}
void Roomba632::clean(){
	p_cleanFlag = true;
	//Xcoo = 0;
	//Ycoo = 0;
	//theta = 0;
}
void Roomba632::spot(){
	p_spotFlag = true;
}
void Roomba632::dock(){
	p_dockFlag = true;
}
void Roomba632::playScript(const uint8_t *script){
	//memcpy(script,script,sizeof(script));
	p_scriptPointer = script;
	p_startScriptFlag = true;
	//Xcoo = 0;
	//Ycoo = 0;
	//theta = 0;
}
void Roomba632::playMusic(const uint8_t *song, size_t songNumberOfBytes){
	//set music to be played
	p_songPointer = song;
	p_songNumberOfBytes = songNumberOfBytes;
	p_playMusicFlag = true;
}
/*
void Roomba632::drive(arguments){
	//set driving variables
	p_driveFlag = true;
}
*/

void Roomba632::setCallbackEndRun(void (*callbackFuncEndRun)()){
	_callbackFuncEndRun = callbackFuncEndRun;
}
void Roomba632::EventEndRun(){
	if (_callbackFuncEndRun){
		_callbackFuncEndRun();
	}
}
void Roomba632::setCallbackStartRun(void (*callbackFuncStartRun)()){
	_callbackFuncStartRun = callbackFuncStartRun;
}
void Roomba632::EventStartRun(){
	if (_callbackFuncStartRun){
		_callbackFuncStartRun();
	}
}
void Roomba632::setCallbackStateChange(void (*callbackFuncStateChange)()){
	_callbackFuncStateChange = callbackFuncStateChange;
}
void Roomba632::EventStateChange(){
	if (_callbackFuncStateChange){
		_callbackFuncStateChange();
	}
}
void Roomba632::setCallbackError(void (*callbackFuncError)()){
	_callbackFuncError = callbackFuncError;
}
void Roomba632::EventError(){
	if (_callbackFuncError){
		_callbackFuncError();
	}
}
void Roomba632::handler(){
	//state machine for high level states
	switch (roombaState){
		case roombaHome: //Charging
			//Check flags in appropriate order of prirority
			if(p_cleanFlag){
				Xcoo = 0;
				Ycoo = 0;
				theta = 0;
				SendCleanCommand();
			}
			else if(p_startScriptFlag){
				Xcoo = 0;
				Ycoo = 0;
				theta = 0;
				roombaState = roombaPlayingScript;
				EventStateChange();
			}
			else if(p_playMusicFlag){
				p_playMusicFlag=false;
				//SendPlayMusicCommand();
			}	
			if (dataState < chargingMessageWaitingForHeader){
				roombaState = roombaIdle;
				p_pressToPause = false;
				EventStateChange();
			}
			break;
		case roombaIdle: //IDLE: not charging, driving, playing music or cleaning
			p_stopFlag = false; //Clear stop request flag if set
			//Keep alive
			//Check flags in appropriate order of prirority
			if(p_cleanFlag){
				SendCleanCommand();
			}
			else if(p_spotFlag){
				SendSpotCommand();
			}
			else if(p_dockFlag){
				SendDockCommand();
			}
			else if(p_playMusicFlag){
				SendPlayMusicCommand();
			}	
			else if(p_startScriptFlag){
				roombaState = roombaPlayingScript;
				EventStateChange();
			}
			if((millis() - p_undockedTimer) > DOCKEDDEBOUNCETIME){
				if(docked){
					roombaState = roombaHome;
					p_pressToPause = false;
					EventStateChange();
				}
			}
			break;
		case roombaPlayingScript:
			if(ScriptHandler() == RMIdle){
				roombaState = roombaIdle;
				p_pressToPause = false;
				EventStateChange();
			}
			UpdateOdometry();
			break;
		case roombaCleaning:
			p_cleanFlag = false; //clear request flag
			if(p_stopFlag){
				SendStopCommand();
			}
			else if(p_dockFlag){
				SendDockCommand();
			}
			else if(p_spotFlag){
				SendSpotCommand();
			}
			else if(p_playMusicFlag){
				SendPlayMusicCommand();
			}
			if((millis() - p_undockedTimer) > DOCKEDDEBOUNCETIME){
				if(docked){
					roombaState = roombaHome;
					p_pressToPause = false;
					EventStateChange();
					EventEndRun();
				}
			}
			UpdateOdometry();
			break;
		case roombaSpotCleaning:
			p_spotFlag = false; //clear request flag
			if(p_stopFlag){
				SendStopCommand();
			}
			else if(p_cleanFlag){
				SendCleanCommand();
			}
			else if(p_dockFlag){
				SendDockCommand();
			}
			else if(p_playMusicFlag){
				SendPlayMusicCommand();
			}	
			if((millis() - p_undockedTimer) > DOCKEDDEBOUNCETIME){
				if(docked){
					roombaState = roombaHome;
					p_pressToPause = false;
					EventStateChange();
					EventEndRun();
				}
			}
			UpdateOdometry();
			break;
		case roombaDocking:
			p_dockFlag = false; //clear request flag
			if(p_stopFlag){
				SendStopCommand();
			}
			else if(p_cleanFlag){
				SendCleanCommand();
			}
			else if(p_spotFlag){
				SendSpotCommand();
			}
			else if(p_playMusicFlag){
				SendPlayMusicCommand();
			}	
			if((millis() - p_undockedTimer) > DOCKEDDEBOUNCETIME){
				if(docked){
					roombaState = roombaHome;
					p_pressToPause = false;
					EventStateChange();
					EventEndRun();
				}
			}
			UpdateOdometry();
			break;
		case roombaDriving:
			p_driveFlag = false;
			roombaState = roombaIdle;
			p_pressToPause = false;
			EventStateChange();
			UpdateOdometry();
			break;
		case roombaMusic:
			p_playMusicFlag = false;
			b_songPlaying = true;
			//time music or poll state
			static int songPointerIndex;
			static uint8_t maxIndex, i;
			static int musicState = musicInit;
			static unsigned long commandTimerMusic, musicTimer, songTime;
			switch (musicState){
				case musicInit:
					songPointerIndex = 0;
					musicState = musicWriteSongHeader;
					commandTimerMusic = millis();
        			break;
				case musicWriteSongHeader:
					if ((millis() - commandTimerMusic) > COMMANDTIMEOUT){
						Serial.write(OCSong);
						Serial.write(1);
						songTime = 0;
						if ((p_songNumberOfBytes-songPointerIndex)<MAXSONGLENGTH){
							maxIndex = (p_songNumberOfBytes-songPointerIndex);
						}
						else{
							maxIndex = MAXSONGLENGTH;
						}
						Serial.write(maxIndex/2); //number of notes
						musicState = musicWriteNotes;
						i = 0;
						commandTimerMusic = millis();
					}
					break;
				case musicWriteNotes:
					if ((millis() - commandTimerMusic) > COMMANDTIMEOUT){
						if(i < maxIndex){
							Serial.write(pgm_read_byte(p_songPointer + songPointerIndex));
							if(songPointerIndex % 2 != 0){
								songTime += pgm_read_byte(p_songPointer + songPointerIndex);
								if (pgm_read_byte(p_songPointer + songPointerIndex) != 0){
									songTime -= 2;
								}
							}
							songPointerIndex++;
							i++;
						}
						else{
							musicState = musicStartSong;
						}
						commandTimerMusic = millis();
					}
					break;
				case musicStartSong:
					if ((millis() - commandTimerMusic) > COMMANDTIMEOUT){
						musicTimer = millis();
						songTime = (1000 * songTime)/64;
						musicState = musicPlaySong;
						Serial.write(OCPlaySong);
						Serial.write(1);
					}
					break;
				case musicPlaySong:
					if((millis() - musicTimer) > songTime){
						if(songPointerIndex >= p_songNumberOfBytes){
							musicState = musicInit;
							roombaState = roombaIdle;
							p_pressToPause = false;
							EventStateChange();
							b_songPlaying = false;
							SendStopCommand();
						}
						else {
							musicState = musicWriteSongHeader; //next song
						}
					}
					//waiting while song plays
					break;
			}
			break;
	}
	//State machine for getting data from Roomba in docked and undocked state.
	static unsigned long messagetimeouttimer, commandTimer;
	static char* pos;
	static char incomming, buffer[7];
	static char i, j, errorCounter, resetCounter;
	static char chargingHeaderBuffer[10], chargingMessageBuffer[80];
	static byte databuffer[44]; // 19 packetID's + 25 databytes
    static byte checksum;  
	byte bumpAndDropByte, wheelOvercurrentByte, lightBumpersByte;
	//debugVal = dataState;
	switch (dataState){
		//Idle: try to start datastream
		case dataIdle:
			//try to start datastream
			dataState = datastreamStart;
			break;
		//Handle serial charging message
		//Example:
		//"bat:   min 16  sec 35  mV 16198  mA 1102  tenths-deg-C 223  mAH 2496  state 5"
		case chargingMessageWaitingForHeader:
		
			if(Serial.available()){
				incomming = Serial.read();
				if(incomming =='b'){
					dataState = chargingMessageReadingHeader;
					j=0;
				}
			}
			//timeout?
			else if ((millis() - messagetimeouttimer) > CHARGINGMESSAGETIMEOUT){
   				//start datastream to figure out what it's up to
				dataState = datastreamStart;
			}
		
			break;
		case chargingMessageReadingHeader:
			if (Serial.available()){
				chargingHeaderBuffer[j] = Serial.read();
				j++;
				if (j==10){
					if (strncmp(chargingHeaderBuffer, "at:   min ",10)==0){ 
						dataState = chargingMessageReadingMessage;
		                //messagetimeouttimer = millis(); //reset timer
						OImode = Off; //Docked
					}
					else{
						dataState = chargingMessageWaitingForHeader;
					}
					j = 0;
				}
			}
			//timeout?
			else if ((millis() - messagetimeouttimer) > CHARGINGMESSAGETIMEOUT){
   				//start datastream to figure out what it's up to
				dataState = datastreamStart;
			}
			break;
		case chargingMessageReadingMessage:
			if(Serial.available()){
				chargingMessageBuffer[j] = Serial.read();
				if (j==79){ //Message should have ended by now
					j=0;
					dataState = chargingMessageWaitingForHeader;
				}
				if (chargingMessageBuffer[j] == '\n'){ //End of message
					pos = strtok(chargingMessageBuffer, " ");
					j = 1;				
					while (pos != NULL){
						if (j== 5){ //Voltage
							strcpy(buffer, pos);
							batteryVoltage = atol(buffer);
						}
						if (j== 7){ //Current
							strcpy(buffer, pos);
							current = atol(buffer);
						}
						if (j== 9){ //Battery temperature
							strcpy(buffer, pos);
							batteryTemp = atol(buffer)/10; //Gaat nog niet goed. Is in tienden! Dus delen door 10
						}
						if (j== 11){
							strcpy(buffer, pos);
							batteryCharge = atol(buffer);
						}
						if (j== 13){
							strcpy(buffer, pos);
							chargingState = atoi(buffer);
						}
						j++;
						pos = strtok(NULL, " ");
					}
					//state = roombaHome;
					//charging = true;
					//docked = true;
					//p_requestStopDatastreamFlag = true;	
					messagetimeouttimer = millis(); //reset timer
					dataState = chargingMessageWaitingForHeader;
					j = 0;
				}
				j++;
			}
			//timeout?
			else if ((millis() - messagetimeouttimer) > CHARGINGMESSAGETIMEOUT){
   				//start datastream to figure out what it's up to
				dataState = datastreamStart;
			}
			break;
		//Handle datastream
		case datastreamStart:
			if(OImode == Off){
				Serial.write(OCstart); //Roomba will change to passive mode
				OImode = Passive;
			}
			//maybe clear data?
			commandTimer = millis();		
			dataState = datastreamStarting;
			break;
 		case datastreamStarting:
            if ((millis() - commandTimer) > COMMANDTIMEOUT){
                //setup datastream
                Serial.write(OCDatastream); //start stream
                Serial.write(19); //19 packets
                Serial.write(7);  //1- Bumps & wheel drops U 1
                Serial.write(9);  //2- Clif left U 1
                Serial.write(10); //3- Clif front left U 1
                Serial.write(11); //4- Clif front right U 1
                Serial.write(12); //5- Clif right U 1
                Serial.write(14); //6- Wheel overcurrents U 1
                Serial.write(15); //7- dirt detect U 1
                Serial.write(21); //8- Charging state U 1
                Serial.write(22); //9- voltage U 2
                Serial.write(23); //10- current S 2
                Serial.write(24); //11- Battery temp S 1
                Serial.write(25); //12- Battery charge U 2
                Serial.write(26); //13- Battery capacity U 2
                Serial.write(34); //14- Charging sources available U 1
                Serial.write(35); //15- OI mode U 1
                Serial.write(37); //16- Song playing? U 1
                Serial.write(43); //17- Wheel encoder counts left S 2
                Serial.write(44); //18-Wheel encoder counts right S 2
                Serial.write(45); //19- Light bumper U 1
				//Number of packets 19, with 25 databytes 
                //serial write startDatastream
                dataState = datastreamWaitingForHeader;
                //Set timer
                messagetimeouttimer = millis();
            }
            break;
        case datastreamWaitingForHeader: 
			//disentangle datastream and do checksum for valid data. 
            if (Serial.available()){
				byte serialByte= Serial.read();
				if(serialByte == 19){ //Header for datastream (save to variable to do multiple checks for charging message etc)
                    dataState = datastreamWaitingForNbytes;
                    messagetimeouttimer = millis(); //reset timer
                }     
            }
			//timeout
			else if ((millis() - messagetimeouttimer) > CHARGINGMESSAGETIMEOUT){
				messagetimeouttimer = millis(); //reset timer
				errorCounter++;
				dataState = datastreamStart;
				if (errorCounter == DATASTREAMMAXERROR){
					dataState = dataReset;
				}
			}
            break;    
        case datastreamWaitingForNbytes:
            if (Serial.available()){
			    if(Serial.read() == 44){ //number of databytes & packet ID's 25+19
                    dataState = datastreamReadingData;
                    i = 0;
                    checksum = 63; //19 + 44 (number of bytes) starting value of checksum
                    messagetimeouttimer = millis(); //reset timer
                }    
                else { //number of bytes should imediately follow header. Else, header was just a rando 19 value
					//Go back to finding the next header
					dataState = datastreamWaitingForHeader;
                }
            }    
            //timeout
            else if ((millis() - messagetimeouttimer) > CHARGINGMESSAGETIMEOUT){
                messagetimeouttimer = millis(); //reset timer
                errorCounter++;
				dataState = datastreamStart;
                if (errorCounter == DATASTREAMMAXERROR){
					dataState = dataReset;
                }
            }
            break;
        case datastreamReadingData:
            if(Serial.available()){
                databuffer[i] = Serial.read();
                checksum += databuffer[i];
                if(i==43){ //number of bytes -1
                    dataState = datastreamCheckingData;
                    messagetimeouttimer = millis(); //Reset timer
                }
                i++;
            }
            //timeout
            else if ((millis() - messagetimeouttimer) > CHARGINGMESSAGETIMEOUT){
                messagetimeouttimer = millis(); //reset timer
                errorCounter++;
				dataState = datastreamStart;
                if (errorCounter == DATASTREAMMAXERROR){
 					dataState = dataReset;
               }
            }
            break;
        case datastreamCheckingData:
            if(Serial.available()){
                checksum += Serial.read(); //Addition to checksum should give 256 on rollover of byte to 0
                if (checksum == 0){
                    //checksum passed
					bumpAndDropByte = databuffer[1];
					bumpRight = bitRead(bumpAndDropByte, 0); 
					bumpLeft = bitRead(bumpAndDropByte, 1);
					wheelDropRight = bitRead(bumpAndDropByte, 2);
					wheelDropLeft = bitRead(bumpAndDropByte, 3);
					
					cliffLeft = databuffer[3];
					cliffFrontLeft = databuffer[5];
					cliffFrontRight = databuffer[7];
					cliffRight = databuffer[9];
                    
					wheelOvercurrentByte = databuffer[11];
					sideBrushOvercurrent = bitRead(wheelOvercurrentByte, 0);
					mainBrushOvercurrent = bitRead(wheelOvercurrentByte, 2);
					rightWheelOvercurrent = bitRead(wheelOvercurrentByte, 3);
					leftWheelOvercurrent = bitRead(wheelOvercurrentByte, 4);
					
					dirtDetect = databuffer[13];
                    chargingState = databuffer[15];
					batteryVoltage = (databuffer[17] << 8) | databuffer[18];
                    current = (databuffer[20] << 8) | databuffer[21];
                    batteryTemp = databuffer[23];
                    batteryCharge = (databuffer[25] << 8) | databuffer[26];
                    batteryCapacity = (databuffer[28] << 8) | databuffer[29];
                    docked = bitRead(databuffer[31],1);
					OImode = databuffer[33];
                    songPlaying = databuffer[35];
                    encoderLeft = (databuffer[37] << 8) | databuffer[38];
                    encoderRight = (databuffer[40] << 8) | databuffer[41];
					
					lightBumpersByte = databuffer[43];
					ltBumperLeft = bitRead(lightBumpersByte, 0);
					ltBumperFrontLeft = bitRead(lightBumpersByte, 1);
					ltBumperCenterLeft = bitRead(lightBumpersByte, 2);
					ltBumperCenterRight = bitRead(lightBumpersByte, 3);
					ltBumperFrontRight = bitRead(lightBumpersByte, 4);
					ltBumperRight = bitRead(lightBumpersByte, 5);
					//debugVal = lightBumpersByte;
					
                    p_encodersReady = true;
					errorCounter = 0;
					if(docked){
						dataState = datastreamStopping;
					}
					else{
			            dataState = datastreamWaitingForHeader;
					}
                }
                else{
                    //checksum failed
                    errorCounter++;
					dataState = datastreamWaitingForHeader;
                    if (errorCounter == DATASTREAMMAXERROR){
 						dataState = datastreamStopping;
                   }
                }
            }
            //timeout
            else if ((millis() - messagetimeouttimer) > CHARGINGMESSAGETIMEOUT){
                //dataState = datastreamWaitingForHeader;
                errorCounter++;
				dataState = datastreamStart;
                if (errorCounter == DATASTREAMMAXERROR){
 					dataState = dataReset;
                }
            }
            break;
        case datastreamStopping:
            //handle last message
            //serial write stop datastream
            Serial.write(OCPauseDatastream);
            Serial.write(0);
            Serial.write(OCStop);
			OImode = Off;
 			dataState = chargingMessageWaitingForHeader;
			messagetimeouttimer = millis(); //reset timer
            errorCounter = 0; 
			break;
        case dataReset:
            //Send command to reset Roomba
            Serial.write(OCReset);
			OImode = Off;
 			dataState = dataResetMessage;
			commandTimer = millis(); //reset timer
			messagetimeouttimer = millis(); //Reset timer
			i=0;
			resetCounter++;
			break;
        case dataResetMessage:
            if ((millis() - commandTimer) > COMMANDTIMEOUT){
				if(Serial.available()){
					messagetimeouttimer = millis(); //Reset timer
					//databuffer[i] = Serial.read(); //Maybe read into buffer
					Serial.read();	//Read serial to empty buffer of resetmessage
					//bl-start                                       
					//STR730                                                                       
					//bootloader id: #x470B575B 67CCBFFF                                           
					//bootloader info rev: #xF000                                                 
					//bootloader rev: #x0001                                                     
					//2007-05-14-1715-L   
					
					if(i==123){
						dataState = datastreamStart;
						messagetimeouttimer = millis(); //Reset timer
						resetCounter = 0;
						errorCounter = 0;
					}
					i++;
				}
				//timeout
				else if ((millis() - messagetimeouttimer) > REBOOTTIMEOUT){
					messagetimeouttimer = millis(); //reset timer
					//errorCounter++;
					dataState = dataReset;
					if (resetCounter == DATASTREAMMAXERROR){
						resetCounter = 0;
						errorCounter = 0;
						//onErrorCallback?
						EventError();
						dataState = datastreamStopping;
				   }
				}
			}
			break;
 	}
}
void Roomba632::SendStopCommand(){
	static bool commandstate = false;
	static unsigned long commandTimer;
	if (!commandstate){
		Serial.write(OCStop);
		OImode = Off;
		commandstate = true;
		commandTimer = millis();
	}
	else{		
		if ((millis() - commandTimer) > COMMANDTIMEOUT){
			commandstate = false;
			roombaState = roombaIdle;	
			OImode = Off;
			p_pressToPause = false;
			EventStateChange();
		}
	}
}
void Roomba632::SendCleanCommand(){
	static bool commandstate = false;
	static unsigned long commandTimer;
	if (!commandstate){
		if(OImode == Off){
			Serial.write(OCstart);
			OImode = Passive;
		}
		commandstate = true;
		commandTimer = millis();
	}
	else{
		if ((millis() - commandTimer) > CLEANCOMMANDSTIMEOUT){
			Serial.write(OCClean);
			if(p_pressToPause){
				p_pressToPause = false;
			}
			else{
				roombaState = roombaCleaning;
				p_pressToPause = true;
				EventStartRun();
				EventStateChange();
				p_undockedTimer = millis();
			}	
			commandstate = false; //reset state
		}
	}
}
void Roomba632::SendDockCommand(){
	static bool commandstate = false;
	static unsigned long commandTimer;
	if (!commandstate){
		if(OImode == Off){
			Serial.write(OCstart);
			OImode = Passive;
		}
		commandstate = true;
		commandTimer = millis();
	}
	else{
		if ((millis() - commandTimer) > CLEANCOMMANDSTIMEOUT){
			Serial.write(OCDock);
			if(p_pressToPause){
				p_pressToPause = false;
			}
			else{
				roombaState = roombaDocking;
				p_pressToPause = true;
				EventStateChange();
				p_undockedTimer = millis();
			}
			commandstate = false; //reset state
		}
	}
}
void Roomba632::SendSpotCommand(){
	static bool commandstate = false;
	static unsigned long commandTimer;
	if (!commandstate){
		if(OImode == Off){
			Serial.write(OCstart);
			OImode = Passive;
		}
		commandstate = true;
		commandTimer = millis();
	}
	else{
		if ((millis() - commandTimer) > CLEANCOMMANDSTIMEOUT){
			Serial.write(OCSpot);
			if(p_pressToPause){
				p_pressToPause = false;
			}
			else{
				roombaState = roombaSpotCleaning;
				p_pressToPause = true;
				EventStateChange();
			}
			commandstate = false; //reset state
		}
	}
}
void Roomba632::SendPlayMusicCommand(){
	static int commandstate = startIO;
	static unsigned long commandTimer;
	switch (commandstate){
		case startIO:
			if(OImode == Off){
				Serial.write(OCstart);
				OImode = Passive;
			}
			commandstate = setMode;
			commandTimer = millis();
			break;
		case setMode:
			if ((millis() - commandTimer) > COMMANDMODETIMEOUT){
				if(OImode < Safe){
					Serial.write(OCSafemode);
					OImode = Safe;
				}
				commandstate = startIO; //reset state
				roombaState = roombaMusic; //sending music commands is handled in the music state
				EventStateChange();
			}
			break;		
	}
}
int Roomba632::ScriptHandler(){
	int16_t velocity, radius;
	uint8_t scriptDistance, scriptAngle, scriptTime;
	static uint8_t scriptRepeat;
	static int scriptState = RMIdle, p_programCounter; //program counter for scripts;
	static unsigned long scriptTimer, waitingTime;
	switch (scriptState){
		case RMIdle:	//0
			if (p_startScriptFlag){
				p_startScriptFlag = false;
				scriptState = RMStart;
			}
			break;
		case RMStart:	//1
			if(OImode == Off){
				Serial.write(OCstart);
				OImode = Passive;
				scriptState = RMStarting;
			}
			else{
				p_programCounter= 0;	//Reset script
				scriptState = pgm_read_byte(p_scriptPointer + p_programCounter);
			}
			scriptTimer = millis();
			break;
		case RMStarting://2
			if ((millis() - scriptTimer) > COMMANDSTARTTIMEOUT){
				if (OImode < Safe){
					Serial.write(OCSafemode);
					OImode = Safe;
				}
				p_programCounter= 0;	//Reset script
				scriptState = pgm_read_byte(p_scriptPointer + p_programCounter);
				scriptTimer = millis();
				EventStartRun();
			}
			break;
		case RMHalt:	//3 scriptable
			if ((millis() - scriptTimer) > COMMANDMODETIMEOUT){
				Serial.write(OCStop);
				OImode = Off;
				scriptTimer = millis();
				scriptState = RMHalting;
			}
			break;
		case RMHalting:	//4
			if((millis() - scriptTimer) > COMMANDSTARTTIMEOUT){
				p_programCounter = 0;
				scriptState = RMIdle;
			}
			break;
		case RMDriveF:	//5 scriptable verb + noun
			if ((millis() - scriptTimer) > COMMANDMODETIMEOUT){
				scriptDistance = pgm_read_byte(p_scriptPointer + p_programCounter +1); //Distance in steps of 25mm (gives maximum of 6.375m)
				waitingTime = scriptDistance * DISTANCESCALAR; 
				velocity = 250; //mm/s
				radius = 32767; //Drive straight
				SendDriveCommand(velocity, radius);
				scriptTimer = millis();
				scriptState = RMDrivingF;
			}
			break;
		case RMDrivingF://6
			if((millis() - scriptTimer) > waitingTime){
				p_programCounter = p_programCounter + 2;
				scriptState = pgm_read_byte(p_scriptPointer + p_programCounter);
				scriptTimer = millis();
			}
			break;
		case RMDriveR:	//7 scriptable verb + noun
			if ((millis() - scriptTimer) > COMMANDMODETIMEOUT){
				scriptDistance = pgm_read_byte(p_scriptPointer + p_programCounter + 1); //Distance in steps of 25mm (gives maximum of 6.375m)
				waitingTime = scriptDistance * DISTANCESCALAR; 
				velocity = -250; //mm/s
				radius = 32767; //Drive straight
				SendDriveCommand(velocity, radius);
				scriptTimer = millis();
				scriptState = RMDrivingR;
			}
			break;
		case RMDrivingR://8
			if((millis() - scriptTimer) > waitingTime){
				p_programCounter = p_programCounter + 2;
				scriptState = pgm_read_byte(p_scriptPointer + p_programCounter);
				scriptTimer = millis();
			}
			break;
		case RMTurnL:	//9 scriptable verb + noun
			if ((millis() - scriptTimer) > COMMANDMODETIMEOUT){
				scriptAngle = pgm_read_byte(p_scriptPointer + p_programCounter +1); //Angle in 256 steps for full circle (quarter turn = 64)
				waitingTime = scriptAngle * TURNINGSPEED;//quarter turn should give 1570ms
				velocity = 256; //mm/s 
				radius = 256; //radius in mm's
				SendDriveCommand(velocity, radius);
				scriptTimer = millis();
				scriptState = RMTurningL;
			}
			break;
		case RMTurningL://10
			if((millis() - scriptTimer) > waitingTime){
				p_programCounter = p_programCounter + 2;
				scriptState = pgm_read_byte(p_scriptPointer + p_programCounter);
				scriptTimer = millis();
			}
			break;
		case RMTurnR:	//11 scriptable verb + noun
			if ((millis() - scriptTimer) > COMMANDMODETIMEOUT){
				scriptAngle = pgm_read_byte(p_scriptPointer + p_programCounter +1); //Angle in 256 steps for full circle (quarter turn = 64)
				waitingTime = scriptAngle * (TURNINGSPEED);//quarter turn should give 1570ms
				velocity = 256 - TURNINGOFFSET; //mm/s 
				radius = -256; //radius in mm's
				SendDriveCommand(velocity, radius);
				scriptTimer = millis();
				scriptState = RMTurningR;
			}
			break;
		case RMTurningR://12
			if((millis() - scriptTimer) > waitingTime){
				p_programCounter = p_programCounter + 2;
				scriptState = pgm_read_byte(p_scriptPointer + p_programCounter);
				scriptTimer = millis();
			}
			break;
		case RMDriveRawFL:	//13 scriptable verb + 3 nouns: velocity, radius & time
			if ((millis() - scriptTimer) > COMMANDMODETIMEOUT){
				velocity = pgm_read_byte(p_scriptPointer + p_programCounter + 1) * 2; //mm/s 
				radius = pgm_read_byte(p_scriptPointer + p_programCounter +2) * 8; //radius in mm's
				waitingTime = pgm_read_byte(p_scriptPointer + p_programCounter +3) * 128;//Distance measured in time
				SendDriveCommand(velocity, radius);
				scriptTimer = millis();
				scriptState = RMDrivingRawFL;
			}
			break;
		case RMDrivingRawFL://14
			if((millis() - scriptTimer) > waitingTime){
				p_programCounter = p_programCounter + 4;
				scriptState = pgm_read_byte(p_scriptPointer + p_programCounter);
				scriptTimer = millis();
			}
			break;
		case RMDriveRawFR:	//15 scriptable verb + 3 nouns: velocity, radius & time
			if ((millis() - scriptTimer) > COMMANDMODETIMEOUT){
				velocity = pgm_read_byte(p_scriptPointer + p_programCounter +1) * 2; //mm/s 
				radius = 0 - (pgm_read_byte(p_scriptPointer + p_programCounter +2) * 8); //radius in mm's
				waitingTime = pgm_read_byte(p_scriptPointer + p_programCounter +3) * 128;//Distance measured in time
				SendDriveCommand(velocity, radius);
				scriptTimer = millis();
				scriptState = RMDrivingRawFR;
			}
			break;
		case RMDrivingRawFR://16
			if((millis() - scriptTimer) > waitingTime){
				p_programCounter = p_programCounter + 4;
				scriptState = pgm_read_byte(p_scriptPointer + p_programCounter);
				scriptTimer = millis();
			}
			break;
		case RMDriveRawBL:	//17 scriptable verb + 3 nouns: velocity, radius & time
			if ((millis() - scriptTimer) > COMMANDMODETIMEOUT){
				velocity = 0 - (pgm_read_byte(p_scriptPointer + p_programCounter +1) * 2); //mm/s 
				radius = pgm_read_byte(p_scriptPointer + p_programCounter +2) * 8; //radius in mm's
				waitingTime = pgm_read_byte(p_scriptPointer + p_programCounter +3) * 128;//Distance measured in time
				SendDriveCommand(velocity, radius);
				scriptTimer = millis();
				scriptState = RMDrivingRawBL;
			}
			break;
		case RMDrivingRawBL://18
			if((millis() - scriptTimer) > waitingTime){
				p_programCounter = p_programCounter + 4;
				scriptState = pgm_read_byte(p_scriptPointer + p_programCounter);
				scriptTimer = millis();
			}
			break;
		case RMDriveRawBR:	//19 scriptable verb + 3 nouns: velocity, radius & time
			if ((millis() - scriptTimer) > COMMANDMODETIMEOUT){
				velocity = 0 - (pgm_read_byte(p_scriptPointer + p_programCounter +1) * 2); //mm/s 
				radius = 0 - (pgm_read_byte(p_scriptPointer + p_programCounter +2) * 8); //radius in mm's
				waitingTime = pgm_read_byte(p_scriptPointer + p_programCounter +3) * 128;//Distance measured in time
				SendDriveCommand(velocity, radius);
				scriptTimer = millis();
				scriptState = RMDrivingRawBR;
			}
			break;
		case RMDrivingRawBR://20
			if((millis() - scriptTimer) > waitingTime){
				p_programCounter = p_programCounter + 4;
				scriptState = pgm_read_byte(p_scriptPointer + p_programCounter);
				scriptTimer = millis();
			}
			break;
		case RMWait:	//21 scriptable verb + noun
			if ((millis() - scriptTimer) > COMMANDMODETIMEOUT){
				SendDriveCommand(0, 0);
				scriptTime = pgm_read_byte(p_scriptPointer + p_programCounter +1); //Wating time in steps of 128ms for maximum wating time of ~32,5 seconds
				waitingTime = scriptTime * 128;
				scriptState = RMWaiting;
				scriptTimer = millis();
			}
			break;
		case RMWaiting:	//22
			if((millis() - scriptTimer) > waitingTime){
				p_programCounter = p_programCounter + 2;
				scriptState = pgm_read_byte(p_scriptPointer + p_programCounter);
				scriptTimer = millis();
			}
			break;
		case RMGoto: //23 Will go on forever!
			p_programCounter = pgm_read_byte(p_scriptPointer + p_programCounter + 1);
			scriptState = pgm_read_byte(p_scriptPointer + p_programCounter);
			scriptTimer = millis();
			break;
		case RMRepeat: //24 repeat for total of numbered runs
			if (scriptRepeat == 0){
				scriptRepeat = pgm_read_byte(p_scriptPointer + p_programCounter + 1);
				scriptState = pgm_read_byte(p_scriptPointer);
			}
			else if(scriptRepeat != 1){
				scriptRepeat--;
				scriptState = pgm_read_byte(p_scriptPointer);
			}
			else if(scriptRepeat == 1){
				scriptState = RMHalt;
			}
			p_programCounter = 0;
			scriptTimer = millis();
			break;
		default:
			//Reset
			scriptState = RMHalt;
			break;
	}
	return scriptState;
}
void Roomba632::SendDriveCommand(int16_t velocity, int16_t radius){
	Serial.write(OCDrive);
	Serial.write(velocity >> 8);
	Serial.write(velocity);
	Serial.write(radius >> 8);
	Serial.write(radius);
}
/*
void Roomba632::_evilLights(){
}
*/

/*
unsigned long Roomba632::_songLength(song as argument){
	return songlength;
}
*/
void Roomba632::UpdateOdometry(){
	if(p_encodersReady){
		p_encodersReady = false;
		//p_odometryTimer = millis();
		static int16_t encoderRPosPrev = 0;
		static int16_t encoderLPosPrev = 0;
		int16_t encoderRDelta = encoderRight - encoderRPosPrev;
		int16_t encoderLDelta = encoderLeft - encoderLPosPrev;
		float SR = DISTANCEPERCOUNTR * (encoderRDelta);
		float SL = DISTANCEPERCOUNTL * (encoderLDelta);
		float meanDistance = (SL + SR)/2;
		encoderRPosPrev = encoderRight;
		encoderLPosPrev = encoderLeft;
		//theta += atan((SR - SL) / WHEELDISTANCE);
		theta += (SR - SL) / WHEELDISTANCE;
		if(theta > 3.14){
			theta -= 6.28;
		}
		else if(theta < -3.14){
			theta += 6.28;
		}
		//Use motor decoder values
		//Xcoo += meanDistance * cos(theta);           
		//Ycoo += meanDistance * sin(theta);
		
		//Use external gyro values
		Xcoo += meanDistance * cos(AuxYaw);           
		Ycoo += meanDistance * sin(AuxYaw);
	}
}
