﻿; PDP-8 Emulator for SymbOS, version 1.0
; Copyright 2025 Daniel E. Gaskell
;
; Permission is hereby granted, free of charge, to any person obtaining a copy
; of this software and associated documentation files (the “Software”), to deal
; in the Software without restriction, including without limitation the rights
; to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
; copies of the Software, and to permit persons to whom the Software is
; furnished to do so, subject to the following conditions:
;
; The above copyright notice and this permission notice shall be included in
; all copies or substantial portions of the Software.
;
; THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
; IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
; AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
; OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
; THE SOFTWARE.

.z80

;=======================================
; HELPER ROUTINES
;=======================================

; sr(): return switch register in HL
.export _sr
_sr:
	ld hl,_buttons+6
	ld a,0
	or (hl) ; bit 11
	rla
	inc hl
	or (hl) ; bit 10
	rla
	inc hl
	or (hl) ; bit 9
	rla
	inc hl
	or (hl) ; bit 8
	ld d,a
	inc hl
	ld a,0
	or (hl)	; bit 7
	rla
	inc hl
	or (hl)	; bit 6
	rla
	inc hl
	or (hl)	; bit 5
	rla
	inc hl
	or (hl)	; bit 4
	rla
	inc hl
	or (hl)	; bit 3
	rla
	inc hl
	or (hl)	; bit 2
	rla
	inc hl
	or (hl)	; bit 1
	rla
	inc hl
	or (hl)	; bit 0
	ld e,a
	ex de,hl
	ret
	
; set_sr(val): set switch register to val
.export _set_sr
_set_sr:
	; get value, preserving stack
	pop hl
	pop de
	push de
	push hl
	; set buttons[]
	ld b,12
	ld hl,_buttons+17
srloop:
	srl d
	rr e
	ld a,0
	rla
	ld (hl),a
	dec hl
	djnz srloop
	ret

; increment PC
incpc:
	ld de,(_pc)
	inc de
	inc de
	res 5,d	; throw away any overflow past 12 bits (actually 13 because pc is x2)
	ld (_pc),de
	ret

;=======================================
; CPU
;=======================================

; do_multiple(): run _do_inst 255 times
mcount: .byte 0
.export _do_multiple
_do_multiple:
	ld a,255
	ld (mcount),a
mloop:
	call _do_inst
	ld hl,mcount
	dec (hl)
	jp nz,mloop
	ret

; do_inst(): polymorphic jump table to remaining opcodes
.export _do_inst
_do_inst:
	; check breakpoint
	ld de,(_pc)		; note that _pc points to the high byte of the pair!
	ld hl,(_breakpoint)
	or a			; FIXME: can this be eliminated somehow?
	sbc hl,de
	jp z,g2halt
	; read instruction
	ld a,(_ifield)	; graft on instruction field
	or d
	ld d,a
	ld hl,_core
	add hl,de
	ld a,(hl)
	ld (_inst_light),a
	add a,a
	add a,a
	ld (jtab+1),a
jtab: .byte 0x18, 0x00 ; jr 0
	jp op_and
	.byte 0
	jp op_and
	.byte 0
	jp op_tad
	.byte 0
	jp op_tad
	.byte 0
	jp op_isz
	.byte 0
	jp op_isz
	.byte 0
	jp op_dca
	.byte 0
	jp op_dca
	.byte 0
	jp op_jms
	.byte 0
	jp op_jms
	.byte 0
	jp op_jmp
	.byte 0
	jp op_jmp
	.byte 0
	jp op_iot
	.byte 0
	jp incpc		; NOP on IOT with high bit of device set, since devices >31 are not supported
	.byte 0
	jp op_opr
	.byte 0
	jp op_opr
	.byte 0

; AND
; acc &= c(addr)
op_and:
	call get_addr
	ld a,(_acc)     ;4
	and (hl)        ;2
	ld (_acc),a     ;4
	ld a,(_acc+1)   ;4
	inc hl          ;2
	and (hl)        ;2
	ld (_acc+1),a   ;4 22 (15b)
	ret

if 0    ;***

ld de,_acc      ;3
ld a,(de)       ;2
and (hl)        ;2
ld (de),a       ;2
inc de          ;2
inc hl          ;2
ld a,(de)       ;2
and (hl)        ;2
ld (de),a       ;2 19 (11b)

endif   ;***

	
; TAD
op_tad:
	call get_addr
	ld e,(hl)
	inc hl
	ld d,(hl)
	ld hl,(_acc)
	add hl,de
	bit 4,h

	jp z,nocarry    ;2/3
	; carry, toggle link and reset relevant bit
	ld a,(_link)    ;4
	xor 1           ;2
	ld (_link),a    ;4
	res 4,h         ;2
	; reset carried bit
nocarry:
	ld (_acc),hl    ;5 19 (16b)
	ret

if 0    ;***

    res 4,h         ;2
    ld (_acc),hl    ;5
    ret z           ;2
    ld hl,_link     ;3
    ld a,(hl)       ;2
    xor 1           ;2
    ld (hl),a       ;2 18 (13b)
    ret

endif   ;***

	
; ISZ
; c(addr)++; if (!c(addr)) pc++;
op_isz:
	call get_addr
	ld (_ma),hl
	ld e,(hl)
	inc hl
	ld d,(hl)
	inc de
	res 4,d	; throw away any overflow past 12 bits
	ld (hl),d
	dec hl
	ld (hl),e
	ld (_mb),de
	ld a,d
	or e
	ret nz
	ld hl,(_pc)
	inc hl
	inc hl
	res 5,h	; throw away any overflow past 12 bits (actually 13 because pc is x2)
	ld (_pc),hl
	ret

; DCA
; c(addr) = acc; acc = 0;
op_dca:
	call get_addr
	ld (_ma),hl
	ld de,(_acc)
	ld (_mb),de
	ld (hl),e
	inc hl
	ld (hl),d
	ld hl,0
	ld (_acc),hl
	ret
	
; JMS
; c(addr) = pc+1; pc = addr+1;
op_jms:
	call get_addr
	ld (_ma),hl
	ld a,(_ibuf)
	ld (_ifield),a
	ld a,0
	ld (_ion_mmu),a
	ld de,(_pc)	; PC is already incremented
	srl d
	rr e
	ld (hl),e
	inc hl
	ld (hl),d
	inc hl
	inc hl
	ld de,_core
	or a
	sbc hl,de
	res 5,h	; throw away any overflow past 12 bits (actually 13 because pc is x2)
	ld (_pc),hl
	srl h
	rr l
	ld (_mb),hl
	ret
	
; JMP
; pc = addr
op_jmp:
	call get_addr
	ld a,(_ibuf)
	ld (_ifield),a
	ld a,0
	ld (_ion_mmu),a
	inc hl	; because we need to point pc to the high byte
	ld de,_core
	or a
	sbc hl,de
	ld (_pc),hl
	ret
	
; IOT
; Communicate with an I/O device
op_iot:
	call incpc
	dec hl
	ld a,(hl)
	rra
	and #0x7C ; now a = device number * 4
	ld (dtab+1),a
	ld a,(hl)
	and 7	  ; now a = device opcode
dtab: .byte 0x18, 0x00 ; jr 0
	jp io_int		; device 0
	.byte 0
	jp io_ptread	; device 1
	.byte 0
	jp io_ptpunch	; device 2
	.byte 0
	jp io_ttyin		; device 3
	.byte 0
	jp io_ttyout	; device 4
	.byte 0
	; jump hooks for devices 5-15 (all RET, note that devices >31 are not currently supported):
	.byte 0xC9, 0, 0, 0 ; device 5
	.byte 0xC9, 0, 0, 0 ; device 6
	.byte 0xC9, 0, 0, 0 ; device 7
	.byte 0xC9, 0, 0, 0 ; device 8
	.byte 0xC9, 0, 0, 0 ; device 9
	.byte 0xC9, 0, 0, 0 ; device 10
	.byte 0xC9, 0, 0, 0 ; device 11
	.byte 0xC9, 0, 0, 0 ; device 12
	.byte 0xC9, 0, 0, 0 ; device 13
	.byte 0xC9, 0, 0, 0 ; device 14
	.byte 0xC9, 0, 0, 0 ; device 15
	; jump hooks for MMU
	jp io_mmu			; device 16
	.byte 0
	jp io_mmu			; device 17
	.byte 0
	jp io_mmu			; device 18
	.byte 0
	jp io_mmu			; device 19
	.byte 0
	jp io_mmu			; device 20
	.byte 0
	jp io_mmu			; device 21
	.byte 0
	jp io_mmu			; device 22
	.byte 0
	jp io_mmu			; device 23
	.byte 0
	; jump hooks for devices 24-31 (all RET, note that devices >31 are not currently supported):
	.byte 0xC9, 0, 0, 0 ; device 24
	.byte 0xC9, 0, 0, 0 ; device 25
	.byte 0xC9, 0, 0, 0 ; device 26
	.byte 0xC9, 0, 0, 0 ; device 27
	.byte 0xC9, 0, 0, 0 ; device 28
	.byte 0xC9, 0, 0, 0 ; device 29
	.byte 0xC9, 0, 0, 0 ; device 30
	.byte 0xC9, 0, 0, 0 ; device 31
	
; OPR
; Microcoded instructions
mq_saved: .word 0

op_opr:
	call incpc
	ld a,(hl)
	dec hl
	ld c,(hl)
	and 1
	jp z,opr_group1
	bit 0,c
	jp z,opr_group2
opr_group3:
	ld de,(_mq)
	ld (mq_saved),de	; save MQ for swapping later
	bit 7,c
	jp z,g3nobit7
	; bit 7 = clear acc
	ld de,0
	ld (_acc),de
g3nobit7:
	bit 4,c
	jp z,g3nobit4
	; bit 4 = load acc into mq and clear acc
	ld de,(_acc)
	ld (_mq),de
	ld de,0
	ld (_acc),de
g3nobit4:
	bit 6,c
	ret z
	; bit 6 = or acc with mq (works on saved copy so MQA MQL swaps MQ with ACC)
	ld de,(mq_saved)
	ld a,(_acc)
	or d
	ld (_acc),a
	ld a,(_acc+1)
	or e
	ld (_acc+1),a
	ret
	
opr_group2:
	bit 6,c
	jp z,g2nobit6
	; bit 6 = skip when acc < 0
	ld a,(_acc+1)
	and 8
	jp nz,tryskip
g2nobit6:
	bit 5,c
	jp z,g2nobit5
	; bit 5 = skip when acc == 0
	ld de,(_acc)
	ld a,d
	or e
	jp z,tryskip
g2nobit5:
	bit 4,c
	jp z,g2nobit4
	; bit 4 = skip when link != 0
	ld a,(_link)
	or a
	jp nz,tryskip
g2nobit4:
	; bit 3 with no skip = skip
	bit 3,c
	call nz,incpc
	jr g2nobit3
tryskip:
	; bit 3 with skip = no skip
	bit 3,c
	call z,incpc
g2nobit3:
	; bit 7 = clear acc
	bit 7,c
	jr z,g2nobit7
	ld de,0
	ld (_acc),de
g2nobit7:
	; bit 2 = read switch register into acc
	bit 2,c
	jr z,g2nobit2
	push bc
	call _sr
	ld (_acc),hl
	pop bc
g2nobit2:
	; bit 1 = halt
	bit 1,c
	ret z
g2halt:
	ld a,1
	ld (_halt),a
	ld (mcount),a
	ret
	
opr_group1:
	bit 7,c
	jp z,g1nobit7
	; bit 7 = clear acc
	ld de,0
	ld (_acc),de
g1nobit7:
	bit 6,c
	jp z,g1nobit6
	; bit 6 = clear link
	ld a,0
	ld (_link),a
g1nobit6:
	bit 5,c
	jp z,g1nobit5
	; bit 5 = complement acc
	ld a,(_acc+1)
	xor #0x0F
	ld (_acc+1),a
	ld a,(_acc)
	cpl
	ld (_acc),a
g1nobit5:
	bit 4,c
	jp z,g1nobit4
	; bit 4 = complement link
	ld a,(_link)
	xor 1
	ld (_link),a
g1nobit4:
	bit 0,c
	jp z,g1nobit0
	; bit 0 = acc++, carrying into link
	ld hl,(_acc)
	inc hl
	bit 4,h
	jp z,noinccarry
	ld a,(_link)
	xor 1
	ld (_link),a
	res 4,h
noinccarry:
	ld (_acc),hl
g1nobit0:
	bit 3,c
	jp z,g1norr
	; bit 3 = rotate right
	call rar
	bit 1,c
	jp z,g1norr	; rotate twice if bsw set
	call rar
g1norr:
	bit 2,c
	jp z,g1norl
	; bit 2 = rotate left
	call ral
	bit 1,c
	jp z,g1norl	; rotate twice if bsw set
	jp ral		; ret from ral returns from this too (can't be BSW by definition)
g1norl:
	bit 1,c
	ret z
	bit 3,c
	ret nz
	; bit 1 but not bits 2 or 3 (2 already excluded by above ret) = byte swap
    ld hl,(_acc)
    ld a,l
    and #0x3C
	rrca
	rrca	; bits 2-5 now at 0-3
    ld c,a	; save high result -> c
    ld a,l
    and #0xC0
    rlca
	rlca	; bits 6-7 now at 0-1
    ld b,a	; save partial result -> b
	ld a,l
	and #0x03
	rrca
	rrca	; bits 0-1 now at 6-7
	ld d,a	; save partial result -> d
    ld a,h
    and #0x0F
    rlca
    rlca	; bits 0-3 now at 2-5
    or b	; combine with partial result
	or d	; combine with partial result
    ld l,a  ; save low result -> l
    ld h,c  ; save low result -> h
    ld (_acc),hl
    ret
	
rar:
	; rotate acc and link right as a 13-bit register
	ld de,(_acc)
	ld a,(_link)
	rlca
	rlca
	rlca
	rlca
	or d
	rra
	ld d,a
	rr e
	ld a,0
	adc a,a	; carry -> link
	ld (_acc),de
	ld (_link),a
	ret
	
ral:
	; rotate acc and link left as a 13-bit register
	ld de,(_acc)
	ld a,(_link)
	rrca	; link -> carry
	rl e
	rl d
	ld a,0
	bit 4,d
	jp z,rallink0
	inc a
rallink0:
	res 4,d
	ld (_acc),de
	ld (_link),a
	ret

; decode HL = address of opcode -> HL = absolute memory address
; destroys AF, C, DE
get_addr:
	ld c,(hl)	; save first byte to check indirection bit later
	dec hl
	ld a,(hl)
	sla a		; need to multiply lower 7 bits by 2 and check bit 7, so do both at the same time here
	jp nc,zeropage
	; PC page - use highest 5 bits from PC
	ld hl,(_pc)
	ld l,a
	jp indirection
zeropage:
	; zero page - set highest 5 bits to zero
	ld h,0
	ld l,a
indirection:
	; check if indirection is necessary
	bit 0,c
	jp z,graftif
	; check if auto-indexing is necessary (address 8-15) - note that A = L here
	sub 16
	cp 15
	jp nc,noindexing
	ld a,h
	or a
	jp nz,noindexing
	; convert to absolute memory address (already 2x by this point)
	ld de,_core
	add hl,de
	; handle auto-indexing
	ld e,(hl)
	inc hl
	ld d,(hl)
	inc de
	res 4,d	; throw away any overflow past 12 bits
	ld (hl),d
	dec hl
	ld (hl),e
	ld hl,_core
	add hl,de
	add hl,de
	; graft on data field
	ld a,(_dfield)
	or h
	ld h,a
	call incpc
	ret
noindexing:
	; convert to absolute memory address (already 2x by this point)
	ld de,_core
	add hl,de
	; handle indirection without auto-indexing
	ld e,(hl)
	inc hl
	ld d,(hl)
	ld hl,_core
	add hl,de
	add hl,de
	; graft on data field
	ld a,(_dfield)
	or h
	ld h,a
	jp incpc
graftif:
	; convert to absolute memory address (already 2x by this point)
	ld de,_core
	add hl,de
	; graft on instruction field
	ld a,(_ifield)
	or h
	ld h,a
	jp incpc
	
;=======================================
; INTERRUPT MANAGEMENT
;=======================================
io_int:
	cp 0
	jp z,intskon
	cp 1
	jp z,intion
	cp 2
	jp z,intiof
	cp 3
	jp z,intsrq
	cp 4
	jp z,intgtf
	cp 5
	jp z,intrtf
	cp 7
	jp z,_clearflags
	ret
intskon:
	ld a,(_ion)
	or a
	ret z
	xor a
	ld (_ion),a
	jp incpc
intion:
	ld a,3
	ld (_ion),a
	ld a,1
	ld (mcount),a	; immediately end multi-instruction block, in case an interrupt is pending
	ret
intiof:
	ld a,0
	ld (_ion),a
	ret
intsrq:
	ld a,(_irq)
	or a
	ret z
	jp incpc
intgtf:
	ld a,(_ion)
	rrca
	ld de,(_dfield)
	srl e
	srl e
	srl e
	srl e
	srl e
	or e
	ld de,(_ifield)
	srl e
	srl e
	or e
	ld (_acc),a
	ld a,(_irq)
	neg	; sets carry if a != 0
	ld a,0
	rl a
	sla a
	ld de,(_link)
	sla e
	sla e
	sla e
	or e
	ld (_acc+1),a
	ret
intrtf:
	ld hl,_acc
	ld a,(hl)
	and 7
	rlca
	rlca
	rlca
	rlca
	rlca
	ld (_dfield),a
	ld a,(hl)
	and 56
	rlca
	rlca
	ld (_ifield),a
	ld a,(hl)
	rlca
	and 1
	ld (_ion),a
	ld hl,_acc+1
	ld a,(hl)
	and 8
	rrca
	rrca
	rrca
	ld (_link),a
	ret

;=======================================
; PAPER TAPE READER
;=======================================
io_ptread:
	ld b,a
	and 7
	jp nz,ptrsf
ptrpe:
	ld a,(_pt_rflag)	; set IRQ = RFLAG
	or a
	ret z
	ld a,(_irq)
	or 1
	ld (_irq),a
	ret
ptrsf:
	bit 0,b
	jp z,ptrrb
	ld a,(_pt_rflag)
	or a
	call nz,incpc
ptrrb:
	bit 1,b
	jp z,ptrfc
	ld a,(_pt_rbuf)
	ld hl,_acc
	or (hl)
	ld (_acc),a
	jp ptclear
ptrfc:
	bit 2,b
	ret z
	; fall through
ptclear:
	ld a,0
	ld (_pt_rflag),a
	ld a,1
	ld (_pt_rgo),a
	ld (mcount),a		; immediately end multi-instruction block to speed up i/o
	ld a,(_irq)
	and #0xFE
	ld (_irq),a
	ret
	
;=======================================
; PAPER TAPE PUNCH
;=======================================
io_ptpunch:
	ld b,a
	and 7
	jp nz,ptpsf
ptpce:
	ld a,(_pt_pflag)	; set IRQ = PFLAG
	or a
	ret z
	ld a,(_irq)
	or 2
	ld (_irq),a
	ret
ptpsf:
	bit 0,b
	jp z,ptpcf
	ld a,(_pt_pflag)
	or a
	call nz,incpc
ptpcf:
	bit 1,b
	jp z,ptppc
	ld a,0
	ld (_pt_pflag),a
	ld a,(_irq)
	and #0xFD
	ld (_irq),a
ptppc:
	bit 2,b
	ret z
	ld a,(_acc)
	ld (_pt_pbuf),a
	ld a,1
	ld (_pt_pgo),a
	ld (mcount),a		; immediately end multi-instruction block to speed up i/o
	ret

;=======================================
; TELETYPE INPUT
;=======================================
io_ttyin:
	ld c,a
	and 7
	jp nz,ttyksf
ttykcf:
	ld (_tty_kflag),a	; note that A = 0 from AND above
	ld a,(_irq)
	and #0xFB
	ld (_irq),a
	ret
ttyksf:
	bit 0,c
	jp z,ttykcc
	ld a,(_tty_kflag)
	or a
	call nz,incpc
ttykcc:
	bit 1,c
	jp z,ttykrs
	ld a,0
	ld (_acc),a
	ld (_acc+1),a
	ld (_tty_kflag),a
	ld a,1
	ld (mcount),a		; immediately end multi-instruction block to speed up i/o
	ld a,(_irq)
	and #0xFB
	ld (_irq),a
ttykrs:
	bit 2,c
	ret z
	ld a,(_tty_kbuf)
	ld hl,_acc
	or (hl)
	ld (_acc),a
	ret

;=======================================
; TELETYPE OUTPUT
;=======================================
io_ttyout:
	ld b,a
	and 7
	jp nz,ttytsk
ttytfl:
	ld a,1
	ld (_tty_tflag),a
	ld a,(_irq)
	or #0x08
	ld (_irq),a
	ret
ttytsk:
	cp 5
	jp nz,ttytsf
	ld a,(_tty_tflag)
	ld de,(_tty_kflag)
	or e
	ret z
	jp incpc
ttytsf:
	bit 0,b
	jp z,ttytcf
	ld a,(_tty_tflag)
	or a
	call nz,incpc
ttytcf:
	bit 1,b
	jp z,ttytpc
	ld a,0
	ld (_tty_tflag),a
	ld a,(_irq)
	and #0xF7
	ld (_irq),a
ttytpc:
	bit 2,b
	ret z
	ld a,(_acc)
	ld (_tty_tbuf),a
	ld a,1
	ld (_tty_tgo),a
	ld (mcount),a		; immediately end multi-instruction block to speed up i/o
	ret
	
;=======================================
; MMU
;=======================================
io_mmu:
	ld c,(hl)
mmucdf:
	bit 0,c
	jp z,mmucif
	ld a,c
	and 56
	add a,a
	add a,a
	and 32	; limits to 8k, can be changed if more memory added
	ld (_dfield),a
mmucif:
	bit 1,c
	jp z,mmurdf
	ld a,c
	and 56
	add a,a
	add a,a
	and 32	; limits to 8k, can be changed if more memory added
	ld (_ibuf),a
	ld a,1
	ld (_ion_mmu),a
mmurdf:
	bit 2,c
	ret z	; not 62x4, so we can return now
	bit 3,c
	jp z,mmurif
	bit 4,c
	jp nz,mmurib
	ld a,(_dfield)
	sla a
	ld hl,_acc
	or (hl)
	ld (hl),a
	ret
mmurib:
	ld a,(_intbuf)
	ld hl,_acc
	or (hl)
	ld (hl),a
	ret
mmurif:
	bit 4,c
	jp mmurmf
	ld a,(_ifield)
	sla a
	ld hl,_acc
	or (hl)
	ld (hl),a
	ret
mmurmf:
	ld a,(_intbuf)
	and 7
	add a,a
	add a,a
	add a,a
	add a,a
	add a,a
	ld (_dfield),a
	ld a,(_intbuf)
	and 56
	add a,a
	add a,a
	add a,a
	ld (_ibuf),a
	ld a,1
	ld (_ion_mmu),a
	ret
	
