/*

DESCRIPTION:    plc include file for PVC project.

COPYRIGHT:      (c) 2025, Brian Cornell, all rights reserved.

 TARGET MCU:    PIC16F1776

HW REVISION:    241209

VERSION:        3.0

MODIFICATION HISTORY:

     6 Feb 25, BWC, created file.


*/



/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * NECESSARY INCLUDES.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * GENERAL PLC CONSTANTS. Convention:  PLC_<TYPE>_<INFO>
 * Types:
 *  + AD=Address of a network node
 *  + AL=ALias
 *  + AR=ARray address
 *  + BL=boolean value
 *  + BP=bit position within a data object
 *  + CC=Calibration Constant
 *  + CD=command data
 *  + CF=configuration value
 *  + CH=character
 *  + DC=Duty Cycle
 *  + DF=Default value for a configurable parameter
 *  + EN=ENable bit
 *  + IN=INitializing value
 *  + LM=LiMiting operational value
 *  + OF=OFfset in an array, list, etc.
 *  + MK=bit MasK
 *  + MP=Peripheral port bit MaPping
 *  + QY=quantity (count) of something
 *  + RT=RaTio or scaling factor
 *  + RN=function return value
 *  + SL=SeLection value
 *  + SZ=size of an object or data type
 *  + TM=TiMing
 *  + VL=VoLtage
 * 
 * <INFO> is user defined and should be descriptive.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define PLC_SZ_PAA              16              //Size of MAC address in ASCII


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * PLC REGISTER ADDRESSES.
 * 
 * Naming convention is PLC_OF_RNN, where:
 *  + PLC=all PLC related constants begin with this.
 *  + OF=fixed designator for registor OFfset
 *  + RNN=letter R and NN is 2-digit mnemonic for register name.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define PLC_OF_RIE              0x00            //Interrupt enable
#define PLC_OF_RLL              0x01            //Local address 8-bit or LSB
#define PLC_OF_RLM              0x02            //Local address MSB for 16-bit
#define PLC_OF_RLG              0x03            //8-bit group address
#define PLC_OF_RGM              0x04            //Group membership
#define PLC_OF_RMD              0x05            //PLC mode
#define PLC_OF_RTL              0x06            //Transmit message length
#define PLC_OF_RTC              0x07            //Transmit config
#define PLC_OF_RDA              0x08            //Destination address
#define PLC_OF_RCD              0x10            //Transmit command ID
#define PLC_OF_RTD              0x11            //Transmit buffer (31 bytes)
#define PLC_OF_RTN              0x30            //BIU noise threshold
#define PLC_OF_RMC              0x31            //Modem config
#define PLC_OF_RTG              0x32            //Transmit gain
#define PLC_OF_RRG              0x33            //Receiver gain
#define PLC_OF_RRI              0x40            //Receive message info
#define PLC_OF_RRS              0x41            //Remote node source address
#define PLC_OF_RID              0x49            //Receive command ID
#define PLC_OF_RRD              0x4A            //Receive data
#define PLC_OF_RIS              0x69            //Interrupt status
#define PLC_OF_RPA              0x6A            //Physical address (8 bytes)
#define PLC_OF_RFW              0x72            //PLC version number


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * PLC REGISTER BIT CONFIGURATION CONSTANTS.
 * 
 * Naming convention is PLC_RNN_BBB, where:
 *  + PLC=all PLC related constants begin with this.
 *  + RNN=letter R and NN is 2-digit mnemonic for register name.
 *  + BBB=mnemonic representing register bit function.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
//Interrupt enable register 0x00
#define PLC_RIE_ICL             0x7F            //Acknowledge (clear) interrupt
#define PLC_RIE_IPL             0x40            //Interrupt pin active low
#define PLC_RIE_IPH     (0xFF ^ PLC_RIE_IPL)    //Interrupt pin active high
#define PLC_RIE_EBT             0x20            //Enable BIU timeout interrupt
#define PLC_RIE_ENK             0x10            //Enable no ACK interrupt
#define PLC_RIE_ENR             0x08            //Enable no response interrupt
#define PLC_RIE_ERD             0x04            //Enable Rx packet dropped int.
#define PLC_RIE_ERA             0x02            //Enable Rx data available int.
#define PLC_RIE_ETS             0x01            //Enable Tx data sent interrupt

//PLC mode register 0x05
#define PLC_RMD_ETX             0x80            //Enable transmit
#define PLC_RMD_DTX     (0xFF ^ PLC_RMD_ETX)    //Disable transmit
#define PLC_RMD_ERX             0x40            //Enable receive
#define PLC_RMD_DRX     (0xFF ^ PLC_RMD_ERX)    //Disable receive
#define PLC_RMD_LCK             0x20            //Lock configuration
#define PLC_RMD_UCK     (0xFF ^ PLC_RMD_LCK)    //Unlock configuration
#define PLC_RMD_DBU             0x10            //Disable Band-In-Use
#define PLC_RMD_EBU     (0xFF ^ PLC_RMD_DBU)    //Enable Band-In-Use
#define PLC_RMD_ERO             0x08            //Enable Rx buffer overwrite
#define PLC_RMD_DRO     (0xFF ^ PLC_RMD_ERO)    //Disable Rx buffer overwrite
#define PLC_RMD_EXA             0x04            //Use 16-bit addressing
#define PLC_RMD_DXA     (0xFF ^ PLC_RMD_EXA)    //Use 8-bit addressing
#define PLC_RMD_EPR             0x02            //Enable promiscuous mode
#define PLC_RMD_DPR     (0xFF ^ PLC_RMD_EPR)    //Disable promiscuous mode
#define PLC_RMD_ICR             0x01            //Ignore CRC check
#define PLC_RMD_DCR     (0xFF ^ PLC_RMD_ICR)    //Drop Rx if CRC check fails

//Transmit message length register 0x06
#define PLC_RTL_SND             0x80            //Transmit packet

//Transmit config register 0x07
#define PLC_RTC_SAT             0x80            //Source address physical
#define PLC_RTC_SAL     (0xFF ^ PLC_RTC_SAT)    //Source address logical
#define PLC_RTC_DTP             0x40            //Destination address physical
#define PLC_RTC_DTG             0x20            //Destination address group
//Destination address logical
#define PLC_RTC_DTL     (0xFF ^ (PLC_RTC_DTP | PLC_RTC_DTG))
#define PLC_RTC_ACK             0x10            //Require acknowledgement
#define PLC_RTC_UCK     (0xFF ^ PLC_RTC_ACK)    //No acknowledgement
#define PLC_RTC_RY0             0x00            //Retry=0
#define PLC_RTC_RY1             0x01            //Retry=1
#define PLC_RTC_RY2             0x02            //Retry=2
#define PLC_RTC_RY3             0x03            //Retry=3
#define PLC_RTC_RY4             0x04            //Retry=4
#define PLC_RTC_RY5             0x05            //Retry=5
#define PLC_RTC_RY6             0x06            //Retry=6
#define PLC_RTC_RY7             0x07            //Retry=7
#define PLC_RTC_RY8             0x08            //Retry=8
#define PLC_RTC_RY9             0x09            //Retry=9
#define PLC_RTC_RYA             0x0A            //Retry=10
#define PLC_RTC_RYB             0x0B            //Retry=11
#define PLC_RTC_RYC             0x0C            //Retry=12
#define PLC_RTC_RYD             0x0D            //Retry=13
#define PLC_RTC_RYE             0x0E            //Retry=14
#define PLC_RTC_RYF             0x0F            //Retry=15

//Command ID register 0x10 (values 0x10-0x2F reserved, 0x30+ are user defined)
//Use these constants for both the TX_CommandID & RC_CommandID registers
#define PLC_RCD_INV             0x00            //Invalid
#define PLC_RCD_RTE             0x01            //Sets remote Tx enable
#define PLC_RCD_RST             0x02            //Reset the remote node config
#define PLC_RCD_REA             0x03            //Remote node extended addr.
#define PLC_RCD_LAD             0x04            //Set remote node logical addr.
#define PLC_RCD_RLA             0x05            //Get remote node logical addr.
#define PLC_RCD_RPA             0x06            //Get remote node physical addr.
#define PLC_RCD_GRS             0x07            //Get remote node mode register
#define PLC_RCD_GRV             0x08            //Get remote node version
#define PLC_RCD_SRD             0x09            //Send data to remote node
#define PLC_RCD_RRD             0x0A            //Request data from remote node
#define PLC_RCD_TRD             0x0B            //Tx response to remote node
#define PLC_RCD_BIU             0x0C            //Set remote BIU functionality
#define PLC_RCD_SRT             0x0D            //Set remote threshold value
#define PLC_RCD_SGM             0x0E            //Set remote group membership
#define PLC_RCD_GGM             0x0F            //Get remote group membership

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *                          PVC SPECIFIC COMMANDS
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define PLC_RCD_RBT             0x30            //Reset converter
#define PLC_RCD_STR             0x31            //Start converter
#define PLC_RCD_STP             0x32            //Stop converter
#define PLC_RCD_TEL             0x33            //Get telemetry data
#define PLC_RCD_SMV             0x34            //Set MPP voltage
#define PLC_RCD_SOV             0x35            //Set output voltage
#define PLC_RCD_GSV             0x36            //Get set voltages
#define PLC_RCD_GFT             0x37            //Get last fault from NVM
#define PLC_RCD_CFT             0x38            //Clear fault from NVM
#define PLC_RCD_RNV             0x39            //Reset NVM to defaults
#define PLC_RCD_CCA             0x3A            //Set charge controller logical\
                                                address
#define PLC_RCD_MOD             0x3B            //Set operating mode
#define PLC_RCD_VOI             0x3C            //Vout increment/decrement
#define PLC_RCD_GOP             0x3D            //Get operating voltages
#define PLC_RCD_GCA             0x3E            //Get charge algorithm values
#define PLC_RCD_SCA             0x3F            //Set charge algorithm values
#define PLC_RCD_GRL             0x40            //Get FW/HW release numbers
#define PLC_RCD_GPG             0x41            //Get PLC Tx & Rx gain values
#define PLC_RCD_SPG             0x42            //Set PLC Tx & Rx gain values

//Threshold noise register 0x30
#define PLC_RTN_EBU             0x40            //Enable BIU auto-set threshold
#define PLC_RTN_DBU     (0xFF ^ PLC_RTN_EBU)    //Disable BIU auto-set threshold
#define PLC_RTN_T99             0x07            //BIU threshold 99 dBuVrms
#define PLC_RTN_T96             0x06            //BIU threshold 96 dBuVrms
#define PLC_RTN_T93             0x05            //BIU threshold 93 dBuVrms
#define PLC_RTN_T90             0x04            //BIU threshold 90 dBuVrms
#define PLC_RTN_T87             0x03            //BIU threshold 87 dBuVrms
#define PLC_RTN_T80             0x02            //BIU threshold 80 dBuVrms
#define PLC_RTN_T75             0x01            //BIU threshold 75 dBuVrms
#define PLC_RTN_T70             0x00            //BIU threshold 70 dBuVrms

//Modem config register 0x31
#define PLC_RMC_D25             0x60            //25mS transmit delay
#define PLC_RMC_D19             0x40            //19mS transmit delay
#define PLC_RMC_D23             0x20            //23mS transmit delay
#define PLC_RMC_D07             0x9F            //7mS transmit delay
#define PLC_RMC_FBL             0x08            //133.3kHz/130.4kHz
#define PLC_RMC_FBS             0xF7            //133.3kHz/131.8kHz
#define PLC_RMC_24B             0x03            //Modem speed 2400bps
#define PLC_RMC_18B             0x02            //Modem speed 1800bps
#define PLC_RMC_12B             0x01            //Modem speed 1200bps
#define PLC_RMC_06B             0xFC            //Modem speed 600bps

//Transmiter gain 0x32
#define PLC_RTG_35V             0x0E            //3.5 Vp-p
#define PLC_RTG_30V             0x0D            //3.0 Vp-p
#define PLC_RTG_22V             0x0C            //2.25 Vp-p
#define PLC_RTG_15V             0x0B            //1.55 Vp-p
#define PLC_RTG_12V             0x0A            //1.25 Vp-p
#define PLC_RTG_09V             0x09            //900 mVp-p
#define PLC_RTG_06V             0x08            //660 mVp-p
#define PLC_RTG_04V             0x07            //480 mVp-p
#define PLC_RTG_03V             0x06            //360 mVp-p
#define PLC_RTG_02V             0x05            //250 mVp-p
#define PLC_RTG_01V             0x04            //180 mVp-p
#define PLC_RTG_012             0x03            //125 mVp-p
#define PLC_RTG_010             0x02            //100 mVp-p
#define PLC_RTG_007             0x01            //75 mVp-p
#define PLC_RTG_005             0xF0            //55 mVp-p

//Receiver gain 0x33
#define PLC_RRG_01V             0x07            //125 uVrms
#define PLC_RRG_02V             0x06            //250 uVrms
#define PLC_RRG_03V             0x05            //350 uVrms
#define PLC_RRG_06V             0x04            //600 uVrms
#define PLC_RRG_12V             0x03            //1.25 mVrms
#define PLC_RRG_25V             0x02            //2.5 mVrms
#define PLC_RRG_50V             0x01            //5 mVrms
#define PLC_RRG_50D             PLC_RRG_50V     //Default - 5 mVrms

//Receive message information register 0x40
#define PLC_RRI_NPR             0x80            //New packet received
#define PLC_RRI_CPR             0x7F            //Clear new packet bit
#define PLC_RRI_DAT             0x40            //Dest. addr. group address
#define PLC_RRI_SAT             0x20            //Source addr. physical address
#define PLC_RRI_RXS             0x1F            //Mask for retrieving msg size
#define PLC_RRI_CLR             0x00            //Clear register

//Interrupt status register 0x69
#define PLC_RIS_STS             0x80            //Status values have changed
#define PLC_RIS_RSV             0x40            //Reserved (do not use)
#define PLC_RIS_BSY             0x20            //BUI timeout
#define PLC_RIS_NAK             0x10            //No ACK received
#define PLC_RIS_TNR             0x08            //No response received
#define PLC_RIS_RPD             0x04            //Rx packet dropped
#define PLC_RIS_NDA             0x02            //Rx data available
#define PLC_RIS_TDS             0x01            //Tx data sent successfully
#define PLC_RIS_CLR             0x00            //Clear - no status


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * I2C/PLC PARAMETERS & COMMUNICATION BUFFER CONSTANTS.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define PLC_AD_I2C              0x02            //I2C address of PLC controller
#define PLC_TM_BUF              550             //I2C bus free time, uS
#define PLC_AD_BRC              0               //PLC broadcast address
#define PLC_TM_RQT              150             //PLC request timeout * 10mS

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * I2C/PLC DATA BUFFER CONSTANTS.
 * 
 * Data object is defined in OBJECT DEFINITIONS section.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define PLC_SZ_DATA             31              //Size of PLC data packet buffer
#define PLC_OF_WDT              0               //Write data buffer offset
#define PLC_OF_RDT              0               //Read data buffer offset
#define PLC_OF_RXD              0               //Start of data in plc_rxd buf.
#define PLC_OF_LAD              0               //Logical addr offset in buffer
#define PLC_SZ_LAD              1               //Size of logical address
#define PLC_SZ_RCF              2               //Single PLC register cfg size
#define PLC_SZ_RCD              3               //Config two consecutive reg.
#define PLC_SZ_RRG              1               //Register read request size
#define PLC_SZ_RWC              0x01            //Register write command size
#define PLC_SZ_PAD              8               //Size of physical address
//Size of I2C size & address objects in object type plci2c
#define PLC_SZ_SZA              2
#define PLC_SZ_WLA              12              //Write remote node logical addr
#define PLC_SZ_CMD              1               //Size of PLC command value
#define PLC_SZ_UI8              1               //Size of uint8_t cmd parameter


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * PLC REGISTER CONFIGURATION CONSTANTS.
 *  + PLC_CF_RIE sets the interrupt polarity & which are active.
 *  + PLC_CF_RMD enables modem tx & rx and related settings.
 *  + PLC_CF_RTN sets the BIU detection threshold.
 *  + PLC_CF_RMC sets the baud rate & FSK deviation.
 *  + PLC_CF_RTG sets the modem tx gain.
 *  + PLC_CF_RRG sets the modem rx gain.
 *  + PLC_CF_RTB sets the transmit config for broadcast communication.
 *  + PLC_CF_RTC sets the transmit config for normal communication.
 *  + PLC_CF_RTP sets the transmit config for remote node logical address.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define PLC_CF_RIE      ((PLC_RIE_ICL&PLC_RIE_IPH)&(PLC_RIE_ERD | PLC_RIE_ERA))
#define PLC_CF_RMD      ((PLC_RMD_ETX | PLC_RMD_ERX | PLC_RMD_DBU)&(PLC_RMD_DRO\
                        &PLC_RMD_DXA&PLC_RMD_UCK&PLC_RMD_DPR&PLC_RMD_DCR))
#define PLC_CF_RTN      (PLC_RTN_T87 & PLC_RTN_DBU)
#define PLC_CF_RMC      (PLC_RMC_D19 | PLC_RMC_FBL | PLC_RMC_24B)
#define PLC_CF_RTG      (PLC_RTG_15V)
#define PLC_CF_RRG      (PLC_RRG_50V)
#define PLC_CF_RTB      (PLC_RTC_RY0 | PLC_RTC_DTG)&(PLC_RTC_SAL & PLC_RTC_UCK)
#define PLC_CF_RTC      (PLC_RTC_RY0 & PLC_RTC_SAL & PLC_RTC_DTL & PLC_RTC_UCK)
#define PLC_CF_RTP      ((PLC_RTC_RY0&PLC_RTC_SAL&PLC_RTC_UCK) | PLC_RTC_DTP)


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * MACROS.
 * + RESETPLC resets the PLC by setting the reset pin high for S_TM_RTC time.
 * + SETINTPLC sets the available interrupts and can also be used to clear 
 *   current interrupts.
 * + WRPLCNACK writes to the PLC controller using the plc object and issues a
 *   break statement if a NACK is returned.
 * + WRPLC writes to the PLC controller using the plc object with no NACK check.
 * + RDPLCNACK reads from the PLC controller using the plc object and issues a
 *   break statement if a NACK is returned.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define RESETPLC() PIN_OD_PRT=TRUE; __delay_us(S_TM_RTC); PIN_OD_PRT=FALSE
#define SETINTPLC() plc.wr.size=PLC_SZ_RCF; plc.wr.i2ca=I2CSETWR(PLC_AD_I2C); \
            plc.wr.reg=PLC_OF_RIE; plc.wr.data[PLC_OF_WDT]=PLC_CF_RIE
#define WRPLCNACK() if (i2c_m_Wd((unsigned char *) &plc) == I2C_NACK) break
#define WRPLC() i2c_m_Wd((unsigned char *) &plc)
#define RDPLCNACK() if (i2c_m_Rd((unsigned char *) &plc) == I2C_NACK) break


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * OBJECT DEFINITIONS.
 *  + plci2c is used to exchange data with the PLC controller over the I2C bus.
 * A union is used to accommodate various read & write setups but should really
 * be reworked to eliminate redundancy and allow the use of the sizeof operator
 * versus constants. This approach was done to balance readability in object/
 * element references.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
typedef struct {
    uint8_t     cgas;                           //Charge algorithm stage
    uint8_t     state;                          //Current state
    uint8_t     sts;                            //Current status code
    uint8_t     ccadr;                          //NVM charge controller addr
    uint8_t     ctrl;                           //NVM control byte
    uint16_t    ttime;                          //Boot cycle count
    uint16_t    inv;                            //Current Vin
    uint16_t    outv;                           //Current Vout
    uint16_t    tp1;                            //Current phase 1 temp
    uint16_t    tp2;                            //Current phase 2 temp
    uint16_t    dcp1;                           //Current phase 1 duty cycle
    uint16_t    dcp2;                           //Current phase 2 duty cycle
} tlmtry;

typedef struct {
    uint8_t     reg;                            //PLC register
    uint8_t     cfg;                            //Tx config
    uint8_t     lga[PLC_SZ_PAD];                //Logical address to send to
    uint8_t     cmd;                            //Command ID
} txcfg;

//Charger telemetry receive
typedef struct {
    uint8_t     lga;                            //Logical address
    tlmtry      ty;
} chargert;

//Charger set voltages
typedef struct {
    uint16_t    out;                            //Output voltage
    uint16_t    mpp;                            //MPP voltage
} setv;

//Fault detail from NVM
typedef struct {
    uint8_t     sts;                            //Status code
    uint8_t     stsc;                           //Status iteration count
    uint8_t     r_sts;                          //MCU status register value
    uint8_t     r_pcon;                         //MCU pcon register value
    uint16_t    time;                           //Cycle count when recorded
} fault;

//Charger operating voltages
typedef struct {
    uint16_t    inv;                            //Converter's current Vin
    uint16_t    outv;                           //Converter's current Vout
    uint16_t    cdc;                            //Converter's current DC
} opv;

//Charge algorithm parameters
typedef struct {
    uint16_t    vnom;                           //Battery nominal voltage
    uint16_t    vblk;                           //Bulk charge voltage
    uint16_t    vabs;                           //Absorption voltage
    uint16_t    vflt;                           //Float voltage
    uint16_t    tabs;                           //Absorption time
} chga;

//Charger version numbers
typedef struct {
    uint8_t     vmaj;                           //Firmware major ver
    uint8_t     vmin;                           //Firmware minor ver
    uint16_t    vbld;                           //Firmware build #
    uint16_t    hwr;                            //Hardware release date
} hwfwr;

//Charger PLC Tx & Rx gain values
typedef struct {
    uint8_t     txg;                            //PLC Tx gain
    uint8_t     rxg;                            //PLC Rx gain
} plcg;

//Charger set voltages receive
typedef struct {
    uint8_t     lga;                            //Logical address
    setv        sv;
} chargerv;

//Charger fault detail receive
typedef struct {
    uint8_t     lga;                            //Logical address
    fault       ft;
} chargerf;

//Charger operating voltages receive
typedef struct {
    uint8_t     lga;                            //Logical address
    opv         ov;
} chargero;

//Charge algorithm parameters receive
typedef struct {
    uint8_t     lga;                            //Logical address
    chga        ca;
} chargera;

//Charger version numbers receive
typedef struct {
    uint8_t     lga;                            //Logical address
    hwfwr       rl;
} chargerr;

//Charger PLC Tx & Rx gain values receive
typedef struct {
    uint8_t     lga;                            //Logical address
    plcg        pg;
} chargerg;

typedef union {
    //Read data from PLC
    struct {
        uint8_t     size;                       //Size of I2C data
        uint8_t     i2ca;                       //I2C node address to read
        uint8_t     data[PLC_SZ_DATA];          //Data
    } rd;
    //Read received command & data from PLC (Rx cmd register precedes data)
    struct {
        uint8_t     size;                       //Size of I2C data
        uint8_t     i2ca;                       //I2C node address to read
        uint8_t     lga[PLC_SZ_PAD];            //PLC node address
        uint8_t     cmd;                        //PLC command
        uint8_t     data[PLC_SZ_DATA];          //PLC packet payload
    } rdc;
    //Write data to PLC
    struct {
        uint8_t     size;                       //Size of I2C data
        uint8_t     i2ca;                       //I2C node address to write
        uint8_t     reg;                        //PLC controller register
        uint8_t     data[PLC_SZ_DATA];          //PLC data
    } wr;
    //Write telemetry send (response) to PLC
    struct {
        uint8_t     size;                       //Size of I2C data
        uint8_t     i2ca;                       //I2C node address to write
        txcfg       txc;                        //PLC Tx config data
        tlmtry      ty;                         //Telemetry data
    } tmy;
    //Write set voltages send (response) to PLC
    struct {
        uint8_t     size;                       //Size of I2C data
        uint8_t     i2ca;                       //I2C node address to write
        txcfg       txc;                        //PLC Tx config data
        setv        sv;                         //Set voltage data
    } gsv;
    //Write fault send (response) to PLC
    struct {
        uint8_t     size;                       //Size of I2C data
        uint8_t     i2ca;                       //I2C node address to write
        txcfg       txc;                        //PLC Tx config data
        fault       ft;                         //Fault data
    } gft;
    struct {
        uint8_t     size;                       //Size of I2C data
        uint8_t     i2ca;                       //I2C node address to write
        txcfg       txc;                        //PLC Tx config data
    } t;
    struct {
        uint8_t     size;                       //Size of I2C data
        uint8_t     i2ca;                       //I2C node address to write
        txcfg       txc;                        //PLC Tx config data
        uint8_t     val;                        //Set command parameter value
    } sc;
    //Write charger operating voltages send (response) to PLC
    struct {
        uint8_t     size;                       //Size of I2C data
        uint8_t     i2ca;                       //I2C node address to write
        txcfg       txc;                        //PLC Tx config data
        opv         ov;                         //Operating voltages
    } gov;
    //Write charge algorithm parameters send (response) to PLC
    struct {
        uint8_t     size;                       //Size of I2C data
        uint8_t     i2ca;                       //I2C node address to write
        txcfg       txc;                        //PLC Tx config data
        chga        ca;                         //Charge algorithm values
    } gca;
    //Write charger version numbers send (response) to PLC
    struct {
        uint8_t     size;                       //Size of I2C data
        uint8_t     i2ca;                       //I2C node address to write
        txcfg       txc;                        //PLC Tx config data
        hwfwr       rl;                         //FW & HW release #s
    } gcr;
    //Write PLC Tx & Rx gain values send (response) to PLC
    struct {
        uint8_t     size;                       //Size of I2C data
        uint8_t     i2ca;                       //I2C node address to write
        txcfg       txc;                        //PLC Tx config data
        plcg        pg;                         //PLC Tx & Rx gain values
    } ggn;
} plci2c;



//End of file