/*

DESCRIPTION:    I2C master functions for PVC charger. This module provides 
                basic I2C read & write functionality and allows reads to be 
                performed in the foreground or background. Only 8-bit I2C
                addresses are supported.

                There are several module objects that must be referenced by
                user code to control the behavior of functions:
 
                + i2c_idle is a read-only bit flag that indicates if the I2C
                    bus is busy. I2C functions should not be called when TRUE.
                    It will be TRUE during background read operations or 
                    between read/write calls if bus RESTARTs are being issued.
                + i2c_rsen is a read/write bit flag to control if an I2C STOP
                    (FALSE) or RESTART (TRUE) should be issued when a read or
                    write is complete.
                + i2c_rmode is a read/write bit flag to control if reads should
                    be performed in the foreground (TRUE) or background (FALSE).
                + i2c_tor is a timing object used in foreground reads to 
                    timeout if the read takes too long, thus preventing a
                    lockup or watchdog timeout. This object should be 
                    decremented by the system timer until zero. The read 
                    function sets this object to the constant I2C_TMR_RTO
                    found in include i2cm.h. This specifies the number of 
                    system timer intervals that the function will wait for the 
                    read to complete before considering the read to have failed
                    (timeout).
 
                The most likely cause of a timeout is the I2C node holding the 
                clock line low (clock stretching) due to lockup or electrical 
                failure.

                Constant I2C_TMR_RTO should be set to a reasonable value 
                considering the system clock speed, I2C speed, peripheral 
                performance, and largest data payload. This must also be 
                evaluated in the context of the criticality of the foreground 
                code for the read and the watchdog timer. Last, consider the
                timing uncertainty if setting this constant to 1 since it is 
                unknown where in the timing interval between system timer 
                events the system is. If in doubt, or timing is a problem, use 
                a background read.
 
                Truth table for i2c_idle & i2c_rsen flags:
                ON ENTRY:
                     idle=T & rsen=F START, idle set F
                     idle=F & rsen=F ASSUME RESTART ISSUED LAST I2C CALL
                     idle=T & rsen=T START, idle set F
                     idle=F & rsen=T ASSUME RESTART ISSUED LAST I2C CALL

                ON EXIT:
                     idle=F & rsen=F STOP, idle set T
                     idle=F & rsen=T RESTART

                PARAMETER FORMAT
                The read/write functions take only an unsigned character 
                pointer to a buffer with the format: SAD.... where:
                + S, offset 0, number of data bytes in the buffer starting 
                    at D.
                + A, offset 1, I2C address of node to read or write.
                + D, offset 2, Starting location of data to read from / write 
                    to slave. The buffer should be sized for the largest 
                    payload (theoretical maximum 255 bytes) plus two bytes for
                    size & address.

                RETURN VALUES
                The read/write functions return the booleans I2C_ACK or
                I2C_NACK to indicate success or failure respectively. The 
                read function will also set the Size byte of the buffer passed
                to it to zero to indicate success. This is the only way to 
                test for completion of a background read.

                DATA VALIDATION & ERROR DETECTION
                These functions do not perform any validation of data read or
                written: user code is responsible for this. The functions will
                return the boolean value I2C_NACK if the slave responds to a
                written byte with a NACK or there is no response to the address.

                In the event that I2C_NACK is returned a STOP is issued and the
                bus released. Flag i2c_idle will be set TRUE, and if i2c_rsen
                was set it will be reset to FALSE. 

                Background read failures can only be detected by using the
                timeout object i2c_tor. After initiating a background read the
                user should test for the timer set to 0 with the size byte 
                remaining non-zero.

                For all errors, the user is responsible for determining the 
                corrective action.

KNOWN BUGS:     none.

COPYRIGHT:      (c) 2022-2024, Brian Cornell, all rights reserved.

DEPENDENCIES:   i2cm.h

 TARGET MCU:    PIC16F1776

VERSION:        3.0

MODIFICATION HISTORY:

    17 Nov 24, BWC, version 3: added RESTART functionality and modified 
    parameter passing for improved efficiency and background read 
    functionality. See description section above.

    7 Jul 24, BWC, adapted for PVC charger communication with the Infineon 
    (Cypress) PLC controller CY8CPLC10. Unlike version 1, this release uses
    fixed timeouts for start, stop, and acknowledge sequences to avoid a 
    controller watchdog reset due to PLC failure. Timeout values need to be
    evaluated for worst case conditions.

    28 Feb 22, BWC, started development based on original i2c master source.
    

*/


#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <pic.h>
#include <xc.h>
#include <binary.h>
#include "i2cm.h"


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * MODULE DATA.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
__bit   i2c_idle,                               //TRUE when I2C bus is idle
        i2c_rmode,                              //TRUE to read in foreground
        i2c_rsen;                               //TRUE to issue restart
uint8_t rsiz,                                   //# bytes to read
        wsiz,                                   //# bytes to write
        i2c_tor;                                //Read/write timeout timer
unsigned char *ptr_o,                           //Set to value of ptr
              *ptr_s;                           //Points to size byte of ptr_o


/* Function i2c_m_cfg. Initializes the MSSP module using constants defined in
 * i2cm.h.
 * 
 * Returns:     none.
 *
 * Parameters:  none.
 *
*/
void i2c_m_cfg()
{
    SSP1CON1=CFG_SSP1_CON_1;
    SSP1STAT=CFG_SSP1_STAT;
    SSP1CON2=CFG_SSP1_CON_2;
    SSP1CON3=CFG_SSP1_CON_3;
    SSP1ADD=CFG_SSP1_ADD;
    i2c_rmode=I2C_RMODED;                   //Set default read mode.
    i2c_rsen=FALSE;                         //START must be issued first.
    i2c_idle=TRUE;                          //I2C bus is idle.

    return;
} //End function i2c_m_cfg


/* Function i2c_m_Wd. Writes data to a slave in foreground.
 * 
 * Returns:     I2C_ACK on success.
 *              I2C_NACK if no response from slave or failure during write.
 *
 * Parameters:  ptr=pointer to buffer using format in Description section of
 *                  of this module.
 * 
 * 
*/
__bit i2c_m_Wd(unsigned char *ptr)
{
    if (*ptr)                                       //Skip if nothing to write
    {
        //if idle set busy status & issue START condition
        if (i2c_idle)                               //Otherwise RESTART issued
        {
            I2CSTART();
        }

        wsiz=(uint8_t) *ptr;
        for (++wsiz, ++ptr; (wsiz); --wsiz, ++ptr)  //Send data, start w/address
        {
            I2C_M_WB(*ptr);                         //Write byte
            if (ACKSTAT == I2C_NACK) break;         //Not found or unresponsive
        }

        //A NACK (I2C failure) resets status & halts communication
        if (wsiz)
        {
            I2CSTOP();
            return((__bit) I2C_NACK);
        }

        if (i2c_rsen)                               //Issue RESTART
        {
            I2CRSTRT();
        }
        else                                        //Issue STOP
        {
            I2CSTOP();
        }
    }

    return((__bit) I2C_ACK);
} //End function i2c_m_Wd


/* Function i2c_m_Rd. Initiates a read. All reads are done in the background 
 * via the ISR function i2c_m_isrRd, but this function can wait for the read
 * to complete or return after issuing the I2C address & enabling the ISR.
 * 
 * The address byte is sent and if ACK is received the ISR is enabled to read 
 * subsequent bytes. A foreground read is emulated if the module object
 * i2c_rmode is set to I2C_RMODEF.
 * 
 * See additional details in the Description section of of this module.
 * 
 * Returns:     I2C_ACK if read started/completed okay or if no bytes to read.
 *                  The Size byte of the buffer will be set to zero.
 *                  The buffer pointed to by ptr contains the data read from
 *                  the slave starting at ptr+2. The address byte remains
 *                  unmodified.
 * 
 *              I2C_NACK if no response (or this is response from slave).
 *                  This is also returned if the receive is not completed in
 *                  the time set in object i2c_tor (e.g., timeout).
 * 
 *                  The address byte in the buffer remains unmodified, but
 *                  the remainder of the buffer may contain partially received
 *                  data. The Size byte of the buffer remains unmodified.
 *              
 * Parameters:  ptr=pointer to buffer using format in Description section of
 *                  of this module.
 * 
 *
*/
__bit i2c_m_Rd(unsigned char *ptr)
{
    //Only start if there are bytes to read
    if (*ptr)
    {
        rsiz=(uint8_t) *ptr;                        //Save # bytes
        ptr_s=ptr;                                  //Save address of size
        ++ptr;                                      //Advance to address byte
        ptr_o=(ptr+1);                              //Start of data
        if (i2c_idle)
        {
            I2CSTART();
        }
        I2C_M_WB(*ptr);                             //Send address byte
        if (ACKSTAT == I2C_NACK)
        {
            I2CSTOP();
            return(I2C_NACK);                       //Fail on address NACK
        }

        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * Setup for Read: SSP1IF should have been cleared by I2C_M_WB.
         +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        SSP1IE=I2C_SET;                             //Enable ISR
        RCEN=I2C_SET;                               //Begin Rx
        i2c_tor=I2C_TMR_RTO;                        //Start timeout counter

        //Emulate a foreground read
        if (i2c_rmode)
        {
            while (rsiz && i2c_tor);                //Wait for idle or timeout
            if (rsiz)
            {
                I2CSTOP();
                return(I2C_NACK);                   //Read timeout
            }
        }
    }

    return(I2C_ACK);                            //Read start or success good
} //End function i2c_m_Rd


/* Function i2c_isr_Rd. Background (ISR) portion of read. The ISR must include 
 * the conditional test to determine if this function needs to be called.
 * 
 * This function clocks data bytes in from the slave until object siz is zero.
 * There is no validation of each byte read into the buffer and no confirmation
 * that the slave is actually responding. Hence, the read will continue until 
 * object siz is decremented to zero.
 * 
 * When siz is zero, a STOP or RESTART condition will be issued on the I2C bus
 * per the value of flag i2c_rsen. The location pointed to by ptr_s is set to
 * zero to indicated that the read has completed.
 * 
 * Returns:     none.
 *
 * Parameters:  none.
 *
*/
void i2c_m_isrRd()
{
    SSP1IE=I2C_CLR;                         //Clear to prevent queuing on ACK
    SSP1IF=I2C_CLR;                         //Clear IF
    *ptr_o=SSP1BUF;                         //Store byte
    if (--rsiz)                             //More to Rx
    {
        ACKDT=I2C_ACK;                      //ACK
        ACKEN=I2C_SET;                      //Send
        __delay_us(I2C_ACKTO);              //Wait
        SSP1IF=I2C_CLR;                     //Clear IF
        ++ptr_o;                            //Next free buffer position
        SSP1IE=I2C_SET;                     //Re-enable interrupt
        RCEN=I2C_SET;                       //Start receive of next byte
    }
    else                                    //All done
    {
        ACKDT=I2C_NACK;                     //End read
        ACKEN=I2C_SET;                      //Send NACK
        __delay_us(I2C_ACKTO);              //Wait
        SSP1IF=I2C_CLR;                     //Clear IF

        *ptr_s=0;                           //Read siz bytes
        if (i2c_rsen)                       //Issue RESTART
        {
            I2CRSTRT();
        }
        else                                //Issue STOP
        {
            I2CSTOP();
        }
    }

    return;
} //End function i2c_m_isrRd



//End of file
