/*

DESCRIPTION:    PIC-specific shared functions:
                    + ReadVoltage
                    + writeFlashRow
                    + eraseFlashRow
                    + nvmChksum

KNOWN BUGS:     

COPYRIGHT:      (c) 2023, Brian Cornell, all rights reserved.

DEPENDENCIES:   see include file listing.

TARGET MCU:     PIC16F1776

VERSION:        2.0

MODIFICATION HISTORY:
 
    11 Jan 23, BWC, created from functions regularly used in projects. It is
    listed as v2 because v1 of the include file exists (without 'v1') and is a
    dependency for prior work.
    

*/


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * COMPILER & CLIB INCLUDES.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <pic.h>
#include <xc.h>

#include <binary.h>                     //folder microchip/code
#include <sizes.h>                      //folder microchip/code

#ifndef _PIC16F1776_MAP
#include <pic16f1776-map-v2.h>          //folder microchip/code
#endif
#ifndef _PIC16F1776_VMV
#include "pic16f1776-vmv-v2.h"          //project folder
#endif

#ifndef _XTAL_FREQ
#include "_xtal_freq.h"                 //project folder
#endif


/* Function ReadVoltage. Reads the value from the analog channel, optionally 
 * performs scaling, and stores in the volts object pointed to by rv. Three
 * Vref ranges can be selected depending on the required accuracy / resolution:
 *      + S_RT_ADC_MV_L (0-2V), 2mV/bit
 *      + S_RT_ADC_MV_H (0-4V), 4mV/bit
 *      + S_RT_ADC_MV_M (0-5V), 4.88mV/bit
 * 
 * The integer, decimal, and VMV components are stored in separate elements of 
 * the object pointed to by rv. All three are stored with the resolution 
 * defined by system constants. See details in the shared code file for the PIC
 * being used.
 * 
 * No error checking is performed and only a single sample is taken. The ADC
 * remains enabled on the channel & Vref from the previous call. Subsequent 
 * calls using the same settings will execute slightly faster (FI_FVR_DLY -
 * S_TM_ADC_ACQ) since settle time for Vref isn't necessary.
 * 
 * S_RT_ADC_MV_M provides the maximum range at the expense of accuracy because
 * the Vref value used (5V) does not evenly divide into the ADC's full-scale 
 * value. The error is approximately 117uV/bit or full scale ~ 119.691mV. 
 * Depending on the scale used the total error could be significant. The 
 * constant S_RT_MV_M_COR compensates for this but cannot fully eliminate the 
 * error. The value should be adjusted to obtain the most accurate values for 
 * the scale of the voltages being measured. The starting value is derived by 
 * dividing the integer resolution (5mV) by the 1 LSB error (0.005 - 0.00488281
 * =0.000117).
 * 
 * An offset can be subtracted from the read ADC value before it is compensated,
 * scaled, etc. This is useful for sensors that have a non-0V output when idle.
 * The offset is subtracted from the read ADC value when greater than the 
 * offset; otherwise the ADC value is zeroed. Note that the value stored in
 * rv->adc is always the read value.
 * 
 * Returns:  none. The following values are populated in object rv:
 *      vmv=composite value, resolution defined by S_RT_VMV_INT & S_RT_VMV_DEC
 *      volt=integer voltage component
 *      mv=millivolts component
 *      adc=unmodified value read from ADC
 *
 * Parameters:  rv=pointer to volts object:
 *                  rv.mvr=S_RT_ADC_MV_L, S_RT_ADC_MV_H, or S_RT_ADC_MV_M.
 *                  rv.adc1v=the complement to rv.mvr and indicates the adc
 *                      value for 1V. Use constants S_QY_ADC_1V_L,
 *                      S_QY_ADC_1V_H, or S_QY_ADC_1V_M.
 *                  rv.channel=analog channel defined in picxxxxx-map.h.
 *                  rv.scale=scale to apply to vmv values.
 *                  rv.ofs=unmodified ADC value to be subtracted from read ADC 
 *                      value (or zero if none).
 *
*/
void ReadVoltage(volts *rv)
{
    static adcresult adc;                       //Retrieves result from ADC reg.
    static uint8_t adcr=S_RT_ADC_MV_U;          //Current ADC range

    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * Connect ADC to channel for acquisition. Select proper Vref+ and delay
     * time.
     +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    ADCON0bits.CHS=rv->channel;                 //Connect to channel
    if (rv->mvr != adcr)
    {
        adcr=rv->mvr;                           //Maybe save time next call...

        //Assign Vref+ to maximum range or...
        if (rv->mvr == S_RT_ADC_MV_M) ADCON1bits.ADPREF=S_MK_ADC_5V;
        else
        {
            ADCON1bits.ADPREF=S_MK_ADC_FVR;     //Set to FVR
            //Set FVR to 2x if low range or 4x for high
            if (rv->mvr == S_RT_ADC_MV_L) FVRCONbits.ADFVR=CFG_FVR_ADC_2X;
            else FVRCONbits.ADFVR=CFG_FVR_ADC_4X;
        }

        __delay_us(FI_FVR_DLY);                 //FVR settle + acquisition time
    }
    else __delay_us(S_TM_ADC_ACQ);              //Acquisition time only

    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * Perform conversion, copy ADC value, and adjust working copy for offset
     +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    ADCON0bits.GO=TRUE;                         //Start conversion
    while (ADCON0bits.GO);                      //Wait for completion
    adc.result[POS_ADC_HB]=ADRESH;
    adc.result[POS_ADC_LB]=ADRESL;
    rv->adc=adc.value;                          //Save raw value 
    if (adc.value < rv->ofs) adc.value=0;       //Can't be less than offset!
    else adc.value-=rv->ofs;                    //Adjust for offset
    if (rv->mvr == S_RT_ADC_MV_M)               //Correction for max range
        adc.value-=(adc.value / S_RT_MV_M_COR);

    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * Apply scaling & populate vmv values.
     +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    if (rv->scale) adc.value*=rv->scale;        //Scale value

    rv->volt=(adc.value / rv->adc1v);
    rv->mv=(((adc.value % rv->adc1v) * rv->mvr) / S_RT_VMV_DEC);
    rv->vmv=((rv->volt * S_RT_VMV_INT) + rv->mv);

    return;
} //End function ReadVoltage


/* Function writeFlashRow.  Writes a row of flash.  A 16-bit checksum for data
 * is first calculated and stored in the last word of the row.  Hence, the user
 * only has (S_SZ_FLASHR - 2) bytes available for use.
 * 
 * This function can also be used to restore default NVM settings by leveraging
 * the 'BOOT & NVM PROTECTION AND MANAGEMENT' documented in function
 * Initialize.  To do so this function should be called with the data parameter
 * set to NULL.  When so the checksum word is cleared so that upon the next
 * boot the NVM defaults will be used since NVM data is considered invalid.
 * 
 * Returns:     none
 *
 * Parameters:  addr=14-bit address of row in flash to write.  Must point to
 *                  start of array.
 *              data=pointer to array of data to write to flash -OR-
 *                   NULL to clear the existing checksum but leave other data
 *                   in the row intact.
 *              
 *
*/
void writeFlashRow(uint16_t addr, unsigned char *data)
{
    uint16_t chksum;
    unsigned char count;

    if (data)                                   //Writing full row
    {
        count = 0;                              //Start at beginning of row
        chksum = nvmChksum(data);               //Calculate & store checksum
        *(data + S_AR_FLCS_L) = (unsigned char) (chksum & S_MK_U16_L);
        *(data + S_AR_FLCS_H) = \
                (unsigned char) ((chksum & S_MK_U16_H) >> S_SZ_BYTE);

        //Must erase row first for clean write
        eraseFlashRow(addr);
    }
    else                                        //Clearing checksum only
    {
        count = S_AR_FLCS_L;                    //Start at checksum offset in row
        chksum = 0;                             //Set checksum to zero
        data = (unsigned char *) &chksum;       //Set data address to checksum
    }

    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * Setup to write data to flash.  Interrupts must be disabled for entire
     * sequence.  Set LWLO to load data to write latches and then to flash on
     * last byte.
     * 
     * Load high address register here & only once.  Update low register for 
     * individual bytes to write within row.
     * 
     * Variable count represents array element to write.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    di();                               //Interrupts disabled for entire sequence.
    WREN = TRUE;                        //Enable write
    CFGS = FALSE;                       //Program memory
    FREE = FALSE;                       //Write data
    LWLO = TRUE;                        //Load data latches
    PMADRH = (unsigned char) (addr >> 8);

    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * Important:  count must be initialized at beginning of function based on 
     * intent.  It controls number of bytes to write and address offset within
     * NVM row.  data must point to full row or checksum only.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    for (; (count < S_SZ_FLASHR); ++count, ++data)
    {
        PMADRL = (unsigned char) (addr + count);
        PMDATL = *data;
        unlockSeq();                    //Unlock for write
    }

    LWLO = FALSE;                                   //Write row
    unlockSeq();                                    //Write flash

    //All done:  disable writes & re-enable interrupts
    WREN = FALSE;
    ei();

    return;
} //End function writeFlashRow


/* Function eraseFlashRow.  Erases a row of program flash memory.  Interrupts
 * are disabled during the erase.
 * 
 * Returns:     none
 *
 * Parameters:  addr=14-bit pointer to starting address for row.
 *
*/
void eraseFlashRow(uint16_t addr)
{
    PMADRL = (unsigned char) addr;
    PMADRH = (unsigned char) (addr >> S_SZ_BYTE);

    //Erase row
    di();                               //Disable interrupts for duration of function
    CFGS = FALSE;                       //Program memory
    FREE = TRUE;                        //Erase row
    WREN = TRUE;                        //Enable erase
    unlockSeq();                        //Unlock for erase
    WREN = FALSE;                       //Disable erase
    ei();                               //Enable interrupts

    return;
} //End function eraseFlashRow


/* Function nvmChksum.  Calculates the checksum for a row of data.  The 'row'
 * is an unsigned char array of length S_SZ_FLASHR since this function is
 * intended to be used data storage in NVM (program flash).
 * 
 * This function is intended to be used in conjunction with writeFlashRow and
 * assumes the checksum resides in the last two bytes of the row.  Hence, when
 * calculating the checksum only (S_SZ_FLASHR - 2) bytes are examined.
 * 
 * The checksum is a simple sum of byte values in the row starting with offset
 * 0.
 * 
 * Returns:     16-bit checksum value
 *
 * Parameters:  data=pointer to data row
 *              
 *
*/
uint16_t nvmChksum(unsigned char *data)
{
    uint16_t sum;
    unsigned char count;

    for (sum = 0, count=0; (count < S_SZ_FLASHC); ++count)
        sum += *(data + count);

    return(sum);
} //End function nvmChksum



//End of file
