/*

DESCRIPTION:  PIC USART service routines.  

IMPORTANT:  Removed background transmit function and supporting data & code
including code block in receive ISR.  Not used in this project and improves
execution efficiency.

KNOWN BUGS:  n/a

COPYRIGHT:  (c) 2017-2023, Brian Cornell, all rights reserved.

DEPENDENCIES:
    main.h - configuration constants for main.c

VERSION:  3.0, release

MODIFICATION HISTORY:

    18 Jan 23, BWC, upated include file logic for use as a shared source across
    multiple projects.

    7 Mar 22, BWC, v3 adds bit flag usar_nl to append NL to usart_wr_ functions.

    7 Aug 20, BWC, commented out __delay_ms(USARTDISDLY) delays in usartSetRx
    because they introduce delays with no need or benefit for this firmware.
    Discovered when troubleshooting data read/display errors in gpibcon software
    while setting up config file.

    3 Aug 20, BWC, changed assignment statements "RxBuf[RxBufPtr] = NULL" to
    "RxBuf[RxBufPtr] = USARTCHRNULL" to correct cast warnings generated by move
    to C99.  USARTCHRNULL is defined in usart.h.

    15 Oct 19, BWC, added function usartWrite.  Writes several bytes via USART
    respecting flow control.  Intended for sending binary data.

    4 Apr 19, BWC, changed buffer termination character from USARTKEYCR to
    USARTKEYNL.
    Added conditional include _xtal_freq.h for project level reference.
 
    19 Apr 17, BWC, updated to stay in sync with model 2.0 code that includes 
                    optional flow control.

    5 apr 17, BWC, released v1.0.

    7 Dec 16, BWC, started development.
    

*/


#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <pic.h>
#include <xc.h>
#include <binary.h>

//Needed for __delay_xx functions.
#ifndef _XTAL_FREQ
#include "_xtal_freq.h"
#endif

//Module include file.
#ifndef _USART
#include <usart-v3.h>
#endif

//User configuration constants.
#ifndef _USARTC
#include "usartc-v3.h"
#endif



/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *                              Module Data
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
unsigned char RxBuf[USARTSZRXBUF],      //Buffer to hold received data
        RxStatus = USARTSTSRXINT,       //Status of USART service module
        RxBufPtr = 0,                   //0-offset index for RxBuf
        RxByte,                         //Storage for byte read from Rx FIFO buffer
        RxEcho = TRUE,                  //Echo received byte; default is on
        usart_txl;                      //Last Tx byte sent
char    wrbuf[USARTWRSZBUF];            //Buffer for usart_wr_xxx functions
__bit   usart_nl;                       //usart_wr functions append NL when TRUE


/*Function usartISR.  Services USART:
 *      +Rx interrupt
 *      +Tx interrupt
 *      +CTS pin (IOC) interrupt
 * 
 * This routine must be called from the user created ISR.  The user must ensure
 * that global & peripheral interrupts are enabled.
 * 
 * Received bytes populate a NULL terminated buffer to be used for interactive 
 * command entry.  Backspace and carriage return are supported for human data 
 * entry.  Buffer is NULL terminated when it reaches USARTMXRXBUF or when 
 * USARTKEYCR is received.
 * 
 * RTS flow control is incorporated to prevent additional data entry after buffer
 * fill is complete or when receive is disabled.  The module variable RxStatus
 * provides status of Rx operations and is also used to control behavior of this
 * function.
 * 
 * Expected behaviors:  data in the FIFO buffer after CR or buffer limit is 
 * ignored (lost).  Once buffer is terminated further data receipt is disabled; 
 * usartSetRx must be called to re-enable.
 * 
 * Servicing of Tx and IOC (for CTS) interrupts are described in their respective
 * code blocks below.  All three handlers are arranged so that each can provide
 * service in a single interrupt if necessary.
 *
 * Returns:  none
 *
 * Parameters:  none
 *
*/
void usartISR()
{
    if (RCIE)
    {
        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * Process data in Rx buffer.  It holds two bytes so for efficiency 
         * read both in one ISR call.  Set RTS to prevent data loss.
         * 
         * Framing errors are ignored but since the error bit is positionally 
         * tied to FIFO register the byte must be read after checking to clear
         * the error bit.
         *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#ifdef USART_PINCTS
        USART_PINRTS = TRUE;
#endif
        while (RCIF)
        {
            if (!(FERR))
            {
                RxByte = RC1REG;            //Get byte from Rx FIFO register
                if (RxByte == USARTKEYBS)   //Delete byte from buffer if there
                {
                    if (RxBufPtr) --RxBufPtr; else continue;
                }
                else if (RxByte == USARTKEYNL)
                {
                    if (RxBufPtr)                   //If data in the buffer...
                    {
                        RxBuf[RxBufPtr] = USARTCHRNULL; //...terminate...
                        RxStatus = USARTSTSRXRDY;   //...set status...
                        RCIE = FALSE;               //...and stop servicing ISR
                    }
                }
                else if (RxBufPtr < USARTMXRXBUF)   //If room add to buffer
                {
                    RxBuf[RxBufPtr++] = RxByte;
                    RxStatus = USARTSTSRXRCV;       //Update status
                }
                else continue;

                /*Try to echo byte but don't wait.  Since we're in ISR we want
                 * to be efficient and not consume processor.  If Tx buffer is 
                 * ready send without respect for flow control; otherwise skip.
                 */
                USARTECHO();
            }
            else RxByte = RC1REG;       //Read & discard to advance FIFO
        }

        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * Now that receive buffer has been processed clear overrun error if set.
         * Don't clear RTS if the buffer is ready to be read.
         *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        if (OERR)
        {
            CREN = DISABLE;             //Disable Rx to clear overrun
            CREN = ENABLE;              //Re-enable Rx
        }
        //Don't undo if buffer ready
#ifdef USART_PINCTS
        if (!(RxStatus & USARTSTSRXRDY)) USART_PINRTS = FALSE;
#endif
    }

    return;
}


/*Function usartSetRx.  Configures USART for receive & transmit.  Also enables / 
 * disables receive to populate buffer.
 *
 * Returns:  none
 *
 * Parameters:  
 *      +RxCmd=ENABLE:  clear buffer & enable reception.  If reception in progress 
 *                      buffer is cleared.
 * 
 *                      Important:  this function must be called with ENABLE during 
 *                      PIC initialization to configure USART and enable receive.  
 *                      The user must enable peripheral & global interrupts to 
 *                      receive data.
 * 
 *      +RxCmd=DISABLE: disables further reception.  If reception was in progress 
 *                      buffer is NULL terminated and contents remain until called 
 *                      with ENABLE.
 * 
 * This function should be used with caution when usartTxBkgd is in use as it
 * can cause data loss.  To prevent this first cancel the current send or wait
 * for it to complete.  Then call usartSetRx.
 *
*/
void usartSetRx(unsigned char RxCmd)
{
    if (RxCmd)                              //ENABLE
    {
        if (RxStatus & USARTSTSRXINT)       //Initialize & enable
        {
            //Set pin configuration
#ifdef USART_PINCTS
#ifdef USART_ANSRTS
            USART_ANSRTS = DIGITAL;
#endif
#ifdef USART_ANSCTS
            USART_ANSCTS = DIGITAL;
#endif
            USART_TRISRTS = OUTPUT;
            USART_TRISCTS = INPUT;
            USART_WPUCTS = DISABLE;
            USART_SLRRTS = MAXSLEW;
#endif
#ifdef USART_ANSRX
            USART_ANSRX = DIGITAL;
#endif
#ifdef USART_ANSTX
            USART_ANSTX = DIGITAL;
#endif
            USART_TRISRX = INPUT;
            USART_TRISTX = OUTPUT;
            USART_WPURX = DISABLE;
            USART_SLRTX = MAXSLEW;

#ifdef USART_PPSPRX
            USART_PPSPRX = USART_PPSVRX;
#endif

#ifdef USART_PPSPTX
            USART_PPSPTX = USART_PPSVTX;    //Connect USART Tx to pin
#endif

            //Configure USART for async operation
            SPEN = ENABLE;                  //Enable serial port
            BRG16 = USART_BRG16;            //Defines use of 16-bit baud rate generator
            SP1BRGH = USART_BRGH;           //Sets baud rate high register
            SP1BRGL = USART_BRGL;           //Sets baud rate low register
            TX1STA = USART_TX1STA;          //Sets transmit status register
            RX9 = DISABLE;                  //8 data bits, no parity
            CREN = ENABLE;                  //Continuous receiver enable
        }
        else if (!(RxStatus & USARTSTSRXENA))   //Enable only; don't disrupt any Tx in progress!
        {
#ifdef USART_PINCTS
            USART_PINRTS = TRUE;            //Flow control on as precaution
#endif
            //__delay_ms(USARTDISDLY);        //Ample time for sender to complete transmission
            CREN = FALSE;                   //Disable receiver
            while (RCIF) RxByte = RC1REG;   //Clear Rx buffer
            CREN = TRUE;                    //Enable Rx - any overrun error is cleared
        }

        //Tasks common to initialization & completing routine enable
        if (!(RxStatus & USARTSTSRXENA))
        {
            RxBufPtr = 0;                   //Reset receive buffer
            RxStatus = USARTSTSRXENA;       //Ready with empty buffer
#ifdef USART_PINCTS
            USART_PINRTS = FALSE;           //Flow control off
#endif
            RCIE = TRUE;                    //Enable processing of received data
        }
    }
    else                                //DISABLE
    {
        if (RxStatus & (USARTSTSRXRCV | USARTSTSRXENA))
        {
#ifdef USART_PINCTS
            USART_PINRTS = TRUE;            //Flow control on
#endif
            //__delay_ms(USARTDISDLY);        //Ample time for sender to complete transmission
            SPEN = DISABLE;                 //Disable serial port to clear errors & Rx buffer
            RxBuf[RxBufPtr] = USARTCHRNULL; //Terminate buffer
            if (RxBufPtr) RxStatus = USARTSTSRXRDY; else RxStatus = USARTSTSRXDIS;
            RCIE = FALSE;                   //Prevent calls to ISR while disabled but...
            SPEN = ENABLE;                  //...enable USART so data can be Tx
        }
    }

    return;
}


/*Function usartStatRx.  Provides status of USART receive buffer.
 *
 * Returns:  value representing status.  See constants in usart.h starting with
 * STSRX.
 *
 * Parameters:  none
 *
*/
unsigned char usartStatRx()
{
    return(RxStatus);
}


/*Function usartSetEcho.  Controls echoing of received characters during data 
 * entry.
 *
 * Returns:  none
 *
 * Parameters:  TRUE for echo on, FALSE for off.
 *
*/
void usartSetEcho(unsigned char Echo)
{
    if (Echo) RxEcho = TRUE; else RxEcho = FALSE;
    return;
}


/*Function usartGetRx.  Returns pointer to NULL terminated receive buffer.  
 * Buffer is ready for read when console sends KEYCR or buffer is full 
 * (USARTMXRXBUF).
 *
 * Returns:  unsigned character pointer to NULL terminated string when receive is 
 * complete.  Otherwise NULL.
 * 
 * Note:  once called the user should not assume that the address contains a 
 * valid NULL-terminated string unless usartStatRx return code is USARTSTSRXRDY.
 * 
 *
 * Parameters:  none
 *
*/
unsigned char * usartGetRx()
{
    if (RxStatus & USARTSTSRXRDY) return(RxBuf); else return(NULL);
}


/*Function putch.  Support for printf output to USART in async mode.  Ensures 
 * that Tx is ready and CTS clear.  To prevent a lock condition (with Tx / CTS) 
 * the routine employs a timeout loop.  USARTTXTO controls the number of loop 
 * iterations and USARTTXWAIT the number of uS to wait in each loop.  If unable 
 * to transmit in that cumulative time the routine returns without attempting to 
 * send.
 * 
 * The user should set both USARTTXTO & USARTTXWAIT to meet the expected 
 * operating environment, required performance, and baud rate.
 * 
 * printf requires a polled implementation run in foreground.  usartTxData should 
 * be used if background transmission and/or high performance (making as much 
 * processor time as possible available for other tasks) is required.
 *
 * Returns:  none
 *
 * Parameters:  Byte=byte to write to Tx buffer.
 *
*/
void putch(char Byte)
{
    static unsigned char tocount;

    for (tocount = USARTTXTO; (tocount); --tocount, __delay_us(USARTTXWAIT))
    {
#ifdef USART_PINCTS
        if (!(USART_PINCTS) && (TXIF))
#else
        if (TXIF)
#endif
        {
            TX1REG = Byte;
            usart_txl=Byte;
            break;
        }
    }
    return;
}


#ifdef _OPT_FCT_WRITE
/*Function usartWrite.  Transmits num bytes of unsigned data.  Like putch it 
 * respects the configured flow control.
 * 
 * Unlike putch it will send several unsigned (binary) bytes of data with one
 * call.  A timeout during send will terminate the transmission.  This is a 
 * foreground function.
 * 
 * If the receiver requires a terminating character, such as newline, it is the
 * caller's responsibility to include in data or send separately.
 * 
 * The user should set both USARTTXTO & USARTTXWAIT to meet the expected 
 * operating environment, required performance, and baud rate.
 * 
 *
 * Returns:  number of bytes sent.
 *
 * Parameters:  data=pointer to data
 *              num=number of bytes to send
 *
*/
uint8_t usartWrite(void *data, uint8_t num)
{
    static uint8_t tocount,
                   sent;

    for (sent = 0; (sent < num); ++sent)
    {
        for (tocount = USARTTXTO; (tocount); --tocount, __delay_us(USARTTXWAIT))
        {
#ifdef USART_PINCTS
            if (!(USART_PINCTS) && (TXIF))
#else
            if (TXIF)
#endif
            {
                TX1REG = (uint8_t) *((uint8_t *) data + sent);
                usart_txl = (uint8_t) *((uint8_t *) data + sent);
                break;
            }
        }
        if (!(tocount)) break;
    }

    if (usart_nl) putch(USARTKEYNL);        //Append NL
    return(sent);
}


/*Function usart_wr_str.  Writes a NULL terminated string to the USART.
 * 
 * Returns:     none
 *
 * Parameters:  str=string to write. Maximum length of 255 (no error checking).
 *
*/
void usart_wr_str(char *str)
{
    usartWrite(str, (uint8_t) strlen(str));
    return;
} //End function usart_wr_str
#endif


/*Function usart_wr_ui16.  Writes a 16-bit integer as ASCII to the USART.  
 * 
 * Returns:     none
 *
 * Parameters:  val=value to write
 *
*/
void usart_wr_ui16(uint16_t val)
{
    uint8_t c=0;                            //Position counter

    //Write value as ASCII to buf
    for (; ((val) && (c < USARTUI16BUF)); ++c)
    {
        wrbuf[c]=(val % 10) + '0';
        val /= (uint16_t) 10;
    }

    if (!(c)) wrbuf[c++]='0';               //Write a zero if val is 0

    for (; (c); --c) putch(wrbuf[c-1]);     //Write number to USART
    if (usart_nl) putch(USARTKEYNL);        //Append NL

    return;
} //End function usart_wr_ui16


/*Function usart_wr_ui24.  Writes a 24-bit integer as ASCII to the USART.  
 * 
 * Returns:     none
 *
 * Parameters:  val=value to write
 *
*/
void usart_wr_ui24(uint24_t val)
{
    uint8_t c=0;                            //Position counter

    //Write value as ASCII to buf
    for (; ((val) && (c < USARTUI24BUF)); ++c)
    {
        wrbuf[c]=(val % 10) + '0';
        val /= (uint24_t) 10;
    }

    if (!(c)) wrbuf[c++]='0';               //Write a zero if val is 0

    for (; (c); --c) putch(wrbuf[c-1]);     //Write number to USART
    if (usart_nl) putch(USARTKEYNL);        //Append NL

    return;
} //End function usart_wr_ui24


/*Function usartGetSz. Returns the current size of the receive buffer.
 * Buffer size should not be considered valid unless function usartStatRx 
 * return code is USARTSTSRXRDY.
 * 
 * Returns:     Receive buffer size not including NULL byte.
 *
 * Parameters:  none
 *
*/
unsigned char usartGetSz()
{
    return(RxBufPtr);
} //End function usartGetSz



//End of file
