/*

DESCRIPTION:    main include file for PVC project.

COPYRIGHT:      (c) 2025, Brian Cornell, all rights reserved.

 TARGET MCU:    PIC16F1776

HW REVISION:    241209

VERSION:        3.0

MODIFICATION HISTORY:

     5 Jan 25, BWC, version 3 adapted from v2.0 branch.

    23 Jun 24, BWC, version 2 adapted from v1.0 branch of PVC code.


*/



/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * Necessary includes. Build.h files contain the build number constant which is
 * maintained by the build.sh scripts in source & common. These build numbers
 * are added together to derive a unique number for the compiled firmware.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#ifndef UINT_MAX
#include <limits.h>
#endif
#include "build.h"
#include "../pvc-common/build.h"


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * Version & build constants.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define VERMAJOR    3
#define VERMINOR    0
#define VERBUILD    (_VBSOURCE+_VBCOMMON)

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * Hardware release date is coded as a 16-bit integer:
 *  + Hi byte=2-digit year,
 *  + Low byte=day number of year (1-365).
 * 
 * The upper 7 bits of the high byte are used for the year, with the lowest bit
 * of high byte and lower byte used for the day number. The epoch year is 2000,
 * so 7 bits accommodates years 2000 thru 2127.
 * 
 * Example: HW release of 4 April 2025=0x325E or 12,894 base 10.
 * 
 * HW release numbers coded for conditional compile:
 * 4 April 25=0x325E or 12,894
 * 20 Aug 25=0x32E8 or 13,032
 * 
 * Each time a new HW release is created that requires a conditional compile it
 * must be defined here and the appropriate error checking added to ensure the
 * build is done correctly.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define _HWRYEAR    25
#define _HWRDNUM    232
#define HWRELEASE   ((_HWRYEAR << 9) | (_HWRDNUM))

//HW release list.
#define _HWR4APR25  0x325E
#define _HWR20AUG25 0x32E8

//HW release constant error checking.
#if (HWRELEASE==_HWR4APR25)
#warning "Compiling for HW release 4 Apr 25."
#else
#if (HWRELEASE==_HWR20AUG25)
#warning "Compiling for HW release 20 Aug 25."
#else
#error "HW release constant not properly defined."
#endif
#endif


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * Conditional compilation.  Uncomment directive to enable / disable features.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
//#define _OPT_LOOP_TIMING                //Enable RUN loop timing via LED pin
#define _OPT_OC_FAULT                   //Enable over-current FAULT shutdown
#define _OPT_SD_FAULT                   //Enable converter shut-down if fault
#define _OPT_RC_FAULT                   //Enable recovery from fault & idle states
#define _OPT_ENA_P1                     //Enable P1 gate drive
#define _OPT_ENA_P2                     //Enable P2 gate drive
//#define _OPT_DIS_NVF                    //Disable NVM updates due to fault
//#define _OPT_DIS_CLOSL                  //Disable plant control closed loop

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * System clock & Watchdog Timer
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define CFG_OSC_C               0x78            //16MHz
#define CFG_WDT_CON             0x10            //Watchdog interval=256mS, off


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * System wide constants & definitions.  Convention:  S_<TYPE>_<INFO>
 * Types:
 *  + AD=Address of a network node
 *  + AL=ALias
 *  + AR=ARray address
 *  + BL=boolean value
 *  + BP=bit position within a data object
 *  + CC=Calibration Constant
 *  + CD=command data
 *  + CF=configuration value
 *  + CH=character
 *  + DC=Duty Cycle
 *  + DF=Default value for a configurable parameter
 *  + EN=ENable bit
 *  + IN=INitializing value
 *  + LM=LiMiting operational value
 *  + OF=OFfset in an array, list, etc.
 *  + MK=bit MasK
 *  + MP=Peripheral port bit MaPping
 *  + QY=quantity (count) of something
 *  + RT=RaTio or scaling factor
 *  + RN=function return value
 *  + SL=SeLection value
 *  + SZ=size of an object or data type
 *  + TM=TiMing
 *  + VL=VoLtage
 * 
 * <INFO> is user defined and should be descriptive.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define S_QY_DC_1US             (_XTAL_FREQ / 1000000)  //Count for 1uS
#define S_TM_MOD_F              80              //Master PWM frequency, 200kHz
#define S_DC_MIN                0               //Minimum DC
#define S_DC_SP                 5               //Single pulse DC
#define S_DC_OFF                0               //No DC (effectively off)
#define S_DC_MAX                50              //Maximum allowable duty cycle
#define S_DC_RUN                S_DC_MAX        //Default running duty cycle
#define S_QY_DC_MAX             S_TM_MOD_F      //Maximum count for gate drive period
#define S_TM_DC_RUN             (((S_TM_MOD_F / S_QY_DC_1US)*2)+2)  //Time for DC capture
#define S_TM_DC_SP              (S_TM_MOD_F / S_QY_DC_1US)  //Time for single puluse
#define S_MK_BUCK_LDA           0x03            //PWM LDA bit mask
#define S_TM_DCDELAY            500             //# uS per DC increment
#define S_TM_RTC                50              //# uS to hold PLC reset high
#define S_TM_INT                1400            //Power-up delay, mS, for PLC
#define S_TM_NTMR               12              //Update NVM hours
#define S_TM_CCTMR              240             //Seconds to update NVM cycle\
                                                counter after boot
#define S_TM_REST               90              //Seconds after recoverable\
                                                fault before restart can be\
                                                attempted
#define S_TM_SYST               10              //System timer period in mS
#define S_TM_TEMP               5               //# seconds to measure temps
#define S_QY_SYST_1S            (S_QY_MS_S / S_TM_SYST) //System timer periods=1s
#define S_BL_TRIS_IN            1               //Pin configured as input
#define S_BL_TRIS_OUT           0               //Pin configured as output
#define S_EN_PWMLD              0x03            //PWMs to load values via PWMLD
#define S_RT_TEMPDR             4               //High temp DC derating factor
#define S_RT_TEMPRD             1               //High temp DC reduction factor
#define S_VL_VOUTINC            1               //Vout DAC increment/decrement,\
                                                equates to 100mV
#define S_TM_CHGA               10              //Charge sampling time, seconds

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * VOLTAGE MEASUREMENT RATIOS
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define S_RT_VIN                25              //Input volts measurement ratio
#define S_RT_VOUT               25              //Output volts measurement ratio
#define S_RT_TEMP               1               //Temperature measurement ratio
#define S_RT_NVV                1               //Vin ratio for NVM write safety


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * NVM CONTROL BIT MAPPED VALUES AT NVM ROW ADDRESS FLASH_AR_CONTRL.
 * 
 * Control bit precedence (enabled):
 *  + Controlled,
 *  + Charge,
 *  + Autostart.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define S_BL_EMANU              0x00                //Manual control
#define S_BL_EAUTO              0x01                //AUTOSTART enabled
#define S_BL_ECTRL              0x02                //CONTROLLED mode enabled
#define S_BL_ECHGA              0x04                //CHARGE algorithm enabled
#define S_BL_ESTART             (S_BL_EAUTO | S_BL_ECTRL | S_BL_ECHGA)

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * NVM DEFAULT VALUES
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define S_DF_VSET               560             //56V voltage
//#define S_DF_VSET               540             //54V output
//#define S_DF_VSET               585             //60V output
#define S_DF_VSET_H             (S_DF_VSET >> S_SZ_BYTE)
#define S_DF_VSET_L             (S_DF_VSET & S_MK_U16_L)
//#define S_DF_VMPP               0               //MPP disabled
//#define S_DF_VMPP               190               //20V
#define S_DF_VMPP               230               //24V
//#define S_DF_VMPP               240             //Default MPP voltage
#define S_DF_VMPP_H             (S_DF_VMPP >> S_SZ_BYTE)
#define S_DF_VMPP_L             (S_DF_VMPP & S_MK_U16_L)
#define S_DF_TTIM               0               //Unit total time (hours)
#define S_DF_TTIM_H             (S_DF_TTIM >> S_SZ_BYTE)
#define S_DF_TTIM_L             (S_DF_TTIM & S_MK_U16_L)
#define S_DF_TSTS               0               //Time of last fault code (hours)
#define S_DF_TSTS_H             (S_DF_TSTS >> S_SZ_BYTE)
#define S_DF_TSTS_L             (S_DF_TSTS & S_MK_U16_L)
#define S_DF_REG_STATUS         0               //Default NVM value
#define S_DF_REG_PCON           0               //Default NVM value
#define S_DF_LCLADR             0x00            //Local PLC node address
#define S_DF_CHGADR             0xFE            //Charge controller node address
#define S_DF_CONTRL             (S_BL_EMANU)    //Control bits
#define S_DF_BIU                PLC_CF_RTN      //PLC BIU noise threshold
#define S_DF_TXG                PLC_CF_RTG      //PLC transmit gain
#define S_DF_RXG                PLC_CF_RRG      //PLC receiver gain
#define S_DF_VNOM               0               //Charge algorithm nominal V
#define S_DF_VNOM_H             (S_DF_VNOM >> S_SZ_BYTE)
#define S_DF_VNOM_L             (S_DF_VNOM & S_MK_U16_L)
#define S_DF_VBLK               0               //Charge algorithm bulk V
#define S_DF_VBLK_H             (S_DF_VBLK >> S_SZ_BYTE)
#define S_DF_VBLK_L             (S_DF_VBLK & S_MK_U16_L)
#define S_DF_VABS               0               //Charge algorithm absorption V
#define S_DF_VABS_H             (S_DF_VABS >> S_SZ_BYTE)
#define S_DF_VABS_L             (S_DF_VABS & S_MK_U16_L)
#define S_DF_VFLT               0               //Charge algorithm float V
#define S_DF_VFLT_H             (S_DF_VFLT >> S_SZ_BYTE)
#define S_DF_VFLT_L             (S_DF_VFLT & S_MK_U16_L)
#define S_DF_TABS               0               //Charge algorithm absorption T
#define S_DF_TABS_H             (S_DF_TABS >> S_SZ_BYTE)
#define S_DF_TABS_L             (S_DF_TABS & S_MK_U16_L)
#define S_DF_STSC               0               //No successive fault counts

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * REGISTER TESTS FOR MCU FAULT
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define S_LM_PCON_EX            0xFA            //PCON bits to examine at boot
#define S_LM_PCON_BT            0x18            //PCON expected value at boot
#define S_LM_STAT_EX            0x18            //STATUS bits to examine at boot
#define S_LM_STAT_BT            0x18            //STATUS expected value at boot

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * FAULT LIMITS, MAXIMUM OPERATING, AND RECOVERY VALUES.
 * These constants define operating limits beyond which the FAULT machine state
 * is set........
 * FLT=
 * MAX=
 * RCV=
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define S_LM_INV_NVM            400             //ADC value for Vin=20V using\
                                                ADC 'low' range. Vin voltage\
                                                divider is 25:1, 20/25=.8/.002=\
                                                400
#define S_LM_FLT_OCC            100             //Over-current count limit

#if (HWRELEASE==_HWR4APR25)
#define S_LM_TMP_DCL            14              //Limit pwr @ VMV< (temp>85C)
#define S_LM_TMP_DCU            15              //Unlim. pwr @ VMV> (temp<83C)
#define S_LM_FLT_TMP            11              //Shutdown when VMV< (temp>92C)
#define S_LM_RCV_TMP            22              //Restart @ VMV>= (temp<=63C)
#endif
#if (HWRELEASE==_HWR20AUG25)
#define S_LM_TMP_DCL            17              //Limit pwr @ VMV< (temp>86C)
#define S_LM_TMP_DCU            18              //Unlim. pwr @ VMV> (temp<85C)
#define S_LM_FLT_TMP            14              //Shutdown @ VMV< (temp>92C)
#define S_LM_RCV_TMP            27              //Restart @ VMV>= (temp<=65C)
#endif

#define S_LM_OVC_CYC            5               //Number of over-current faults\
                                                that can occur in one boot\
                                                cycle before it is treated as\
                                                non-recoverable
#define S_LM_VIN_NVM            400             //Minimum ADC value for\
                                                measuring Vin to ensure safe\
                                                for NVM writes
#define S_LM_KPA_TIM            3600            //Number of seconds that can\
                                                elapse in CONTROLLED mode\
                                                without valid communication\
                                                from the charge controller
#define S_LM_MAX_VSET           (S_DF_VSET+20)  //2V above default Vout
#define S_LM_CLIM               650             //Inductor peak current (13A)
#define S_LM_CLIM_H             (S_LM_CLIM >> S_SZ_BYTE)
#define S_LM_CLIM_L             (S_LM_CLIM & S_MK_U16_L)
#define S_DF_SCOMP              1               //Default slope comp
#define S_LM_HIVOUT             0x14            //High Vout fault (~64V)
#define S_LM_MIN_VNOM           480             //Charge algorithm min. Vnom
#define S_LM_MAX_VBLK           590             //Charge algorithm max. Vblk
#define S_LM_MIN_TABS           60              //Minimum absorption time, secs
#define S_LM_MAX_PTXG           PLC_RTG_22V     //Maximum PLC Tx gain
#define S_LM_MAX_PRXG           PLC_RRG_01V     //Minimum PLC Rx gain
#define S_LM_MIN_PRXG           PLC_RRG_50V     //Maximum PLC Rx gain

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * PERIPHERAL ALIASES.
 * 
 * Implementation specific to help with readability and portability.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define S_AL_DC_P1              PWM5DC          //Phase 1 (master) PWM DC register
#define S_AL_DC_P2              PWM6DC          //Phase 2 PWM DC register
#define S_AL_DCTMR_P1           TMR3            //Timer to measure phase 1 DC
#define S_AL_DCTMR_P2           TMR5            //Timer to measure phase 2 DC
#define S_AL_DCTEN_P1           T3GGO_nDONE     //Enable DC timer for P1
#define S_AL_DCTEN_P2           T5GGO_nDONE     //Enable DC timer for P2
#define S_AL_OCIE_P2            C2IE            //P2 over-current comparator IE
#define S_AL_OCIF_P2            C2IF            //P2 over-current comparator IF
#define S_AL_OCIE_P1            C4IE            //P1 over-current comparator IE
#define S_AL_OCIF_P1            C4IF            //P1 over-current comparator IF
#define S_AL_COGAS_P1           COG1ASD0bits.ASE    //P1 COG auto-shutdown bit
#define S_AL_COGAS_P2           COG2ASD0bits.ASE    //P2 COG auto-shutdown bit
#define S_AL_PRGGO_P1           PRG1CON0bits.GO //P1 PRG GO bit
#define S_AL_PRGGO_P2           PRG2CON0bits.GO //P2 PRG GO bit
#define S_AL_ADCCH_VDD          ADC_CH_DAC_3    //Analog channel to measure Vdd

//System chrono timer
#define S_AL_SYST_CLKCON        T8CLKCON
#define S_AL_SYST_HLT           T8HLT
#define S_AL_SYST_RST           T8RST
#define S_AL_SYST_PR            T8PR
#define S_AL_SYST_CON           T8CON
#define S_AL_SYST_IE            TMR8IE
#define S_AL_SYST_IF            TMR8IF
#define S_AL_SYST_EN            T8ON


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * Port/pin configuration: I/O, analog/digital, slew rate, etc.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
//Port I/O: 1=input, 0=output
#define IO_PORTA                0b01100011
#define IO_PORTB                0b10101101
#define IO_PORTC                0b10011011
//Port Analog/Digital: 1=analog, 0=digital
#define AD_PORTA                0b00100011
#define AD_PORTB                0b00101110
#define AD_PORTC                0b11011000
//Port weak pull-up: 1=enabled, 0=disabled
#define WP_PORTA                0b00000000
#define WP_PORTB                0b00000000
#define WP_PORTC                0b00000000
#define WP_PORTE                0b00001000
//Open drain: 1=open drain, 0=push-pull
#define OD_PORTA                0b00000000
#define OD_PORTB                0b00000000
#define OD_PORTC                0b00000000
//Slew rate: 1=limited, 0=maximum
#define SR_PORTA                0b00000000
#define SR_PORTB                0b00000000
#define SR_PORTC                0b00000000
//Input level: 1=ST, 0=TLL
#define IL_PORTA                0b00000000
#define IL_PORTB                0b00000000
#define IL_PORTC                0b00000000
//Initial latch states: all outputs logic low
#define S_IN_PORTA              0b00000000
#define S_IN_PORTB              0b00000000
#define S_IN_PORTC              0b00100000


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * PPS & DIRECT PIN MAPPING.
 * 
 * Naming convention for PPS output pin constants:
 *  + Output pin register assignment - PPS_P_XXX
 *  + Peripheral mapping assignment - PPS_M_XXX
 *      Where XXX=net name, driving function, etc.
 * 
 *      Set PPS_P_XXX to the output PPS register value (ex. RA1PPS).
 *      Set PPS_M_XXX to the PPS output peripheral constant from the 
 *      appropriate picxxxxx-map.h file.
 * 
 * Naming convention for PPS input pin constants: PPS_I_XXX
 *      Where XXX=net name, driving function, etc.
 *      Set PPS_I_XXX to the PPS input pin constant from the appropriate
 *      picxxxxx-map.h file.
 * 
 * For pins accessed directly or assigned in a peripheral's register (like ADC)
 * use the following convention:  PIN_XT_YYYYYY where
 *          X=(I)nput or (O)utput
 *          T=(A)nalog or (D)igital
 *          YYYYYY=user assigned mnemonic describing function
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
//Timing for P1 COG & PRG, master for P2.
#define PPS_P_P5P1              RA3PPS          //Output on pin 5
#define PPS_M_P5P1              PPS_PWM5_OUT    //Drive is PWM5
#define PPS_I_P1P1              PPSRA3          //PRG1 rise/fall event trigger

//Timing for P2 COG & PRG.
#define PPS_P_P6P2              RA4PPS          //Output on pin 6
#define PPS_M_P6P2              PPS_PWM6_OUT    //Drive is PWM6
#define PPS_I_P2P2              PPSRA4          //PRG2 rise/fall event trigger

//P1 gate drive & timing measurement.
#define PPS_P_C1P1              RC2PPS          //Output on pin 13
#define PPS_M_C1P1              PPS_COG1A_OUT   //Drive is COG1 A
#define PPS_I_T3P1              PPSRC2          //P1 DC measurement

//P2 gate drive & timing measurement.
#define PPS_P_CGP2              RB4PPS          //Output on pin 25
#define PPS_M_CGP2              PPS_COG2A_OUT   //Drive is COG2 A
#define PPS_I_T5P2              PPSRB4          //P2 DC measurement

//SSR gate drive
#define PPS_P_SSR               RA2PPS          //Output on pin 4
#define PPS_M_SSR               PPS_PWM9_OUT    //Drive is PWM9 

//I2C pin mapping for PLC controller: bidirectional so in/outputs are mapped
#define PPS_P_I2C_D_O           RC0PPS          //SDA (data out) maps to pin 15
#define PPS_P_I2C_C_O           RC1PPS          //SCL (clock out) maps to pin 16
#define PPS_M_I2C_D_O           0x22            //Output SDA on selected pin
#define PPS_M_I2C_C_O           0x21            //Output SCL on selected pin
#define PPS_M_I2C_D_I           PPSRC0          //I2C data in
#define PPS_M_I2C_C_I           PPSRC1          //I2C clock in

//Direct digital pin mapping
#define PIN_OD_LED              RC5             //Status LED: 1=fault, 0=run
#define PIN_OD_PRT              RA7             //PLC controller reset
#define PIN_ID_PIR              RA6             //PLC controller IRQ

//Analog pin mapping
#define PIN_IA_INV              ANLCAN16        //ADC sense - input voltage
#define PIN_IA_OUTV             ANLCAN0         //ADC sense - output voltage
#define PIN_IA_TFP1             ANLCAN4         //P1 temperature sense
#define PIN_IA_TFP2             ANLCAN13        //P2 temperature sense

//TRIS pin mapping for peripheral in/output control
#define TRIS_LED                TRISC5          //TRIS bit for status LED


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * MACROS.
 * + SETLOOPCLOSED is called to engage control loop for output regulation.
 * + SETLOOPOPEN is called to disconnect control loop for fixed DC operation.
 * + SSRON is called to enable the converter output solid state relay.
 * + SSROFF is called to disable the converter output solid state relay.
 * + SETSTSF sets the object sts to record a non-recoverable fault condition 
 *   provided it does not hold an existing code. Object flag_nvf is set to 
 *   record the fault event.
 * + SETSTSR sets the object sts to record a recoverable fault condition 
 *   provided it does not hold an existing code. Timing object tmr_rest is set 
 *   to begin recovery timing and flag_nvf is set to record the fault event.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define SETLOOPCLOSED() COG1FIS0=CFG_COG_1_FIS0_C; COG2FIS0=CFG_COG_2_FIS0_C
#define SETLOOPOPEN() COG1FIS0=CFG_COG_1_FIS0_O; COG2FIS0=CFG_COG_2_FIS0_O
#define SSRON() PPS_P_SSR=PPS_M_SSR
#define SSROFF() PPS_P_SSR=PPS_LAT_OUT
#define SETSTSF(s) if (sts==STS_NOF) {sts=s; state=STE_FAULT; flag_nvf=TRUE;}
#define SETSTSR(s) if (sts==STS_NOF) {sts=s; state=STE_RCVRF; flag_nvf=TRUE;\
    tmr_rest=S_TM_REST;}


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * FVR & DAC CONFIGURATION:
 * 1 (10b)=Vref for Vout control loop
 * 2 (10b)=Current limit
 * 5 (10b)=Vref for MPP control loop
 * 3 (5b)=Vdd (measured by ADC)
 * 7 (5b)=Vout over-volt protection
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define CFG_FVR_C               0xCF            //Enabled, CDAFVRx4, ADFVRx4

//DAC1: voltage is set from NVM.
#define CFG_DAC_1_C             0x88            //Enabled, right, FVR, Vss

//DAC2:  voltage is set from NVM.
#define CFG_DAC_2_C             0x88            //Enabled, right, FVR, Vss

//DAC5:  voltage is set from NVM.
#define CFG_DAC_5_C             0x88            //Enabled, right, FVR, Vss

//DAC3: Vdd/4
#define CFG_DAC_3_C             0x80            //Enabled, (+)=Vdd, (-)=Vss
#define CFG_DAC_3_R             0x08            //Vdd / 4

//DAC7: high Vout level is set from NVM.
#define CFG_DAC_7_C             0x88            //Enabled, FVR, Vss


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * ADC & VMV CONFIGURATION.
 * 
 * Reference pic16f1776-map-v2.h for details on setting constants for VMV, 
 * correction, 5V reference.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
//ADC peripheral configuration
#define CFG_ADC_CON_0           (PIN_IA_INV | S_EN_ADC) //Default ch. & enabled
#define CFG_ADC_CON_1           0xD3            //Right, Fosc/16, Vss, FVR
#define CFG_ADC_CON_2           0x00            //No auto conversion


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * OPAMP (OPA) CONFIGURATION:
 * 2=Vout error amp
 * 3=MPP error amp
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define CFG_OPA_2_C             0x80            //Enabled
#define CFG_OPA_2_O             0x00            //Override not used
#define CFG_OPA_2_N             0x00            //OPA2IN0, pin 23
#define CFG_OPA_2_P             0x02            //DAC1_out

#define CFG_OPA_3_C             0x80            //Enabled
#define CFG_OPA_3_O             0x00            //Override not used
#define CFG_OPA_3_N             0x00            //OPA3IN0, pin 18
#define CFG_OPA_3_P             0x02            //DAC5_out


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * COMPARATOR CONFIGURATION:
 * 1=P2 slope compensation
 * 2=P2 current limit
 * 3=P1 slope compensation
 * 4=P1 current limit
 * 5=Vout over-volt fault sense
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define CFG_CM_1_C0             0xAE            //Enabled, HS, ZLF, HYS, async
#define CFG_CM_1_C1             0x00            //No interrupts
#define CFG_CM_1_N              0x02            //C1IN2-, pin 24
#define CFG_CM_1_P              0x04            //PRG2_out

#define CFG_CM_2_C0             0xAE            //Enabled, HS, ZLF, HYS, async
#define CFG_CM_2_C1             0x01            //Negative edge interrupt
#define CFG_CM_2_N              0x02            //C2IN2-, pin 24
#define CFG_CM_2_P              0x06            //DAC2_out

#define CFG_CM_3_C0             0xAE            //Enabled, HS, ZLF, HYS, async
#define CFG_CM_3_C1             0x00            //No interrupts
#define CFG_CM_3_N              0x04            //C3IN4-, pin 14
#define CFG_CM_3_P              0x03            //PRG1_out

#define CFG_CM_4_C0             0xAE            //Enabled, HS, ZLF, HYS, async
#define CFG_CM_4_C1             0x01            //Negative edge interrupt
#define CFG_CM_4_N              0x04            //C4IN4-, pin 14
#define CFG_CM_4_P              0x06            //DAC2_out

#define CFG_CM_5_C0             0xAE            //Enabled, HS, ZLF, HYS, async
#define CFG_CM_5_C1             0x00            //No interrupts
#define CFG_CM_5_N              0x00            //C5IN0-, pin 2
#define CFG_CM_5_P              0x07            //DAC7_out


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * TIMER CONFIGURATION:
 * 2=SSR period for PWM9
 * 3=P1 DC measurement
 * 5=P2 DC measurement
 * SYST=system chrono timing
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define CFG_TMR_2_CLK           0x01            //Fosc/4
#define CFG_TMR_2_CON           0x80            //Enabled, 1:1 pre & post-scale
#define CFG_TMR_2_HLT           0x20            //On sync to clk, free run
#define CFG_TMR_2_RST           0x00            //No external reset sources
#define CFG_TMR_2_PR            7               //500kHz

#define CFG_TMR_3_CON           0x41            //Fosc, 1:1 prescale, sync, on
#define CFG_TMR_3_GCON          0xD0            //Gated by pin, single pulse

#define CFG_TMR_5_CON           0x41            //Fosc, 1:1 prescale, sync, on
#define CFG_TMR_5_GCON          0xD0            //Gated by pin, single pulse

#define CFG_SYST_CLK            0x05            //MFINTOSC
#define CFG_SYST_CON            0x19            //Off, 1:2 prescale, 1:10 post
#define CFG_SYST_HLT            0xA0            //Prescale & bit sync, free run
#define CFG_SYST_RST            0x00            //No external reset sources
#define CFG_SYST_PR             250             //10mS period


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * PWM CONFIGURATION:
 * 5=P1
 * 6=P2
 * 9=SSR drive
 * 
 * Duty cycle is loaded from NVM during initialization.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define CFG_PWM_5_CON           0x80            //Enabled, active high standard
#define CFG_PWM_5_INTE          0x00            //No interrupts
#define CFG_PWM_5_CLK           0x00            //No prescale, Fosc
#define CFG_PWM_5_LDCON         0x80            //Load triggering is immediate
#define CFG_PWM_5_OFCON         0x00            //Independent run mode
#define CFG_PWM_5_PHASE         0x0000          //Phase count
#define CFG_PWM_5_PERIOD        S_TM_MOD_F      //Period (frequency)
#define CFG_PWM_5_OFFSET        ((S_TM_MOD_F/2)+(S_TM_MOD_F%2)) //PWM6 phase
#define CFG_PWM_5_TIMER         0x0000          //Starting value for timer

#define CFG_PWM_6_CON           0x80            //Enabled, active high standard
#define CFG_PWM_6_INTE          0x00            //No interrupts
#define CFG_PWM_6_CLK           0x00            //No prescale, Fosc
#define CFG_PWM_6_LDCON         0x80            //Load triggering is immediate
#define CFG_PWM_6_OFCON         0x20            //Slave to PWM5 with sync start
#define CFG_PWM_6_PHASE         0x0000          //Phase count
#define CFG_PWM_6_PERIOD        S_TM_MOD_F      //Period (frequency)
#define CFG_PWM_6_OFFSET        0x0000          //Offset count
#define CFG_PWM_6_TIMER         0x0000          //Starting value for timer

#define CFG_PWM_9_CON           0x80            //Enabled
#define CFG_PWM_9_DC            15              //Duty cycle, ~ 1uS
#define CFG_PWM_9_DCH           (CFG_PWM_9_DC >> 2)
#define CFG_PWM_9_DCL           (CFG_PWM_9_DC << 6)


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * COG CONFIGURATION:
 * 1=P1
 * 2=P2
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define CFG_COG_1_CON0          0xCC            //Enable, load, Fosc, HB
#define CFG_COG_1_CON1          0x00            //Clk delay chain, active high
//Rising events
#define CFG_COG_1_RIS0          0x00            //None
#define CFG_COG_1_RIS1          0x02            //PWM5
#define CFG_COG_1_RSIM0         0x00            //None
#define CFG_COG_1_RSIM1         0x02            //PWM5
//Falling events
#define CFG_COG_1_FIS0_C        0x18            //Closed loop CM3, CM4
#define CFG_COG_1_FIS0_O        0x10            //Open loop CM4
#define CFG_COG_1_FIS1          0x02            //PWM5
#define CFG_COG_1_FSIM0         0x10            //Hi->low for CM4, low for CM3
#define CFG_COG_1_FSIM1         0x02            //Hi>low for PWM5
//Auto shutdown
#define CFG_COG_1_ASD0          0x80            //Shutdown, inactive states
#define CFG_COG_1_ASD1          0x20            //LC2_out
//Delays
#define CFG_COG_1_DBR           0               //Rising dead band
#define CFG_COG_1_DBF           0               //Falling dead band
#define CFG_COG_1_BLKR          0               //Rising edge blanking
#define CFG_COG_1_BLKF          0               //Falling edge blanking
#define CFG_COG_1_PHR           0               //Rising event phase delay
#define CFG_COG_1_PHF           0               //Falling event phase delay

#define CFG_COG_2_CON0          0xCC            //Enable, load, Fosc, HB
#define CFG_COG_2_CON1          0x00            //Clk delay chain, active high
//Rising events
#define CFG_COG_2_RIS0          0x00            //None
#define CFG_COG_2_RIS1          0x04            //PWM6
#define CFG_COG_2_RSIM0         0x00            //None
#define CFG_COG_2_RSIM1         0x04            //PWM6
//Falling events
#define CFG_COG_2_FIS0_C        0x06            //Closed loop CM1, CM2
#define CFG_COG_2_FIS0_O        0x04            //Open loop CM2
#define CFG_COG_2_FIS1          0x04            //PWM6
#define CFG_COG_2_FSIM0         0x04            //Hi->low for CM2, low for CM1
#define CFG_COG_2_FSIM1         0x04            //Hi>low for PWM6
//Auto shutdown
#define CFG_COG_2_ASD0          0x80            //Shutdown, inactive states
#define CFG_COG_2_ASD1          0x20            //LC2_out
//Delays
#define CFG_COG_2_DBR           0               //Rising dead band
#define CFG_COG_2_DBF           0               //Falling dead band
#define CFG_COG_2_BLKR          0               //Rising edge blanking
#define CFG_COG_2_BLKF          0               //Falling edge blanking
#define CFG_COG_2_PHR           0               //Rising event phase delay
#define CFG_COG_2_PHF           0               //Falling event phase delay


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * PRG CONFIGURATION:
 * 1=P1 slope comp
 * 2=P2 slope comp
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define CFG_PRG_1_CON0          0xB0            //Enabled, edge, slope, off
#define CFG_PRG_1_CON1          0x02            //Fall low, rise high
#define CFG_PRG_1_INS           0x00            //OPA1OUT/PRG1IN0, pin 3
#define CFG_PRG_1_CON2          0x03            //Slope=350mV/uS
#define CFG_PRG_1_RTSS          0x07            //PRG1R
#define CFG_PRG_1_FTSS          0x07            //PRG1F

#define CFG_PRG_2_CON0          0xB0            //Enabled, edge, slope, off
#define CFG_PRG_2_CON1          0x02            //Fall low, rise high
#define CFG_PRG_2_INS           0x01            //OPA1OUT/PRG2IN1, pin 3
#define CFG_PRG_2_CON2          0x03            //Slope=350mV/uS
#define CFG_PRG_2_RTSS          0x07            //PRG2R
#define CFG_PRG_2_FTSS          0x07            //PRG2F


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * CLC CONFIGURATION:
 * 2=Gate connector from comparator C5 output to COG auto-shutdown for Vout
 * over-volt fault
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define CFG_CLC_2_CON           0x82            //Enabled, 4-input AND
#define CFG_CLC_2_POL           0x00            //Output & gates not inverted
#define CFG_CLC_2_SEL_0         0x0C            //Data 1 connected to sync_C5OUT
#define CFG_CLC_2_SEL_1         0x0C            //Data 2 not used
#define CFG_CLC_2_SEL_2         0x0C            //Data 3 not used
#define CFG_CLC_2_SEL_3         0x0C            //Data 4 not used
#define CFG_CLC_2_GLS_0         0x02            //Data 1 gated into g1
#define CFG_CLC_2_GLS_1         0x02            //Data 1 gated into g2
#define CFG_CLC_2_GLS_2         0x02            //Data 1 gated into g3
#define CFG_CLC_2_GLS_3         0x02            //Data 1 gated into g4


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *                          Print & Format Constants
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * OBJECT DEFINITIONS.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * COMMAND CODES. Each class of commands starts with tens digit:
 *  + Get=1,
 *  + Set=2,
 *  + Remote only commands=8,
 *  + System=9.
 * 
 * Commands that require a parameter are denoted in <>. Unless otherwise stated,
 * single parameter commands require a zero-based integer representing the 
 * option in the order listed when separated by a forward-slash (/) or the 
 * converter phase (0 or 1). NVM commands require a zero-based offset address
 * for the byte or word (0-31 or 0-15 respectively).
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define CMDNONCMD           0               //No command
#define CMDGETSTS           10              //Get system status <state/status>
#define CMDGETCNT           11              //Get over-current count <phase>
#define CMDGETDCY           12              //Get duty cycle <phase>
#define CMDGETEMP           13              //Get temps <phase>
#define CMDGETVOL           14              //Get volts <in/out>
#define CMDGETBYT           15              //Get NVM byte <offset>
#define CMDGETWOR           16              //Get NVM word <offset>
#define CMDGETREL           17              //Get firmware release
#define CMDGETPAD           18              //Get PLC MAC address
#define CMDSETCON           20              //Set converter <off/on>
#define CMDSETBYT           21              //Set NVM byte <offset, value>
#define CMDSETWOR           22              //Set NVM word <offset, value>
#define CMDSETSSR           23              //Set SSR <off/on>
#define CMDXPGPLC           90              //Ping PLC <logical address>
#define CMDXRTMCU           91              //Reset converter
#define CMDXNVDFL           92              //Restore NVM defaults

#define CMDSIZSTR           1               //Minimum size of command string


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * STATE MANAGEMENT AND STATUS CODES.
 * 
 * STATE MANAGEMENT refers to identifying the operational state of the unit.
 * A bit-mapped byte is used instead of bit flags as a compromise for code size
 * & efficiency for use in bit operations and when multiple status bits must be
 * operated on simultaneously. Only one bit can be set in the machine state 
 * object 'state', and the possible values are defined by the STE_XXXXX 
 * constants below. Constants for combinations of machine state are also defined
 * for use in firmware control.
 * 
 * The constants STS_XXXX represent the fault status codes used throughout the 
 * control program and recorded in NVM (diagnostics).
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define STE_BOOT            0x01            //Initialization & calibration
#define STE_RSTRT           0x02            //Restart converter
#define STE_STOP            0x04            //Converter off: no automated start
#define STE_IDLE            0x08            //Converter set off in charger mode
#define STE_RUN             0x10            //Converter on 
#define STE_RCVRF           0x20            //Recoverable fault, code in NVM
#define STE_FAULT           0x40            //Unrecoverable fault, code in NVM

//Combination state tests.
#define STE_BTRRT           (STE_BOOT | STE_RSTRT)
#define STE_RNFT            (STE_RUN | STE_FAULT)
#define STE_BRS             (STE_BOOT | STE_RSTRT | STE_STOP)
#define STE_RCRIE           (STE_RCVRF | STE_IDLE)
#define STE_RCRFT           (STE_RCVRF | STE_FAULT)
#define STE_BSIR            (STE_BOOT | STE_STOP | STE_IDLE | STE_RUN)
#define STE_BTRRN           (STE_BOOT | STE_RUN)

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * STATUS CODES.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define STS_NOF             0               //No fault
#define STS_MCU             1               //MCU fault (stack, watchdog, etc.)
#define STS_NVM             2               //Invalid NVM
#define STS_OCL             3               //Over-current limit exceeded
#define STS_HOV             4               //High output voltage
#define STS_NAK             5               //No response from PLC controller
#define STS_TMP             6               //High temperature
#define STS_OCM             7               //Too many over-current events
#define STS_ADR             8               //No logical address in controlled\
                                            mode
#define STS_KPA             9               //No communication from charge\
                                            controller in S_LM_KPA_TIM seconds\
                                            (keep alive timer expired)
#define STS_CGA             10              //Invalid chg. algorithm parameters
#define STS_IVC             11              //Invalid control bit
#define STS_PCG             12              //PLC controller config failed
#define STS_PIS             13              //Fail reading PLC interrupt status
#define STS_PIT             14              //Fail resetting PLC interrupts
#define STS_PRD             15              //Fail retrieving PLC Rx data
#define STS_PCN             16              //Fail clearing PLC Rx buffer
#define STS_PPC             17              //Fail processing PLC command
#define STS_PFR             18              //Failed PLC fault recovery
#define STS_PFP             19              //PLC failed to send ping
#define STS_PFC             20              //PLC fail check for duplicate addr
#define STS_PDA             21              //PLC found duplicate logical addr
#define STS_PFA             22              //PLC fail setting logical address\
                                             post duplicate check in config


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * CHARGE ALGORITHM STATE CODES.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define CGA_STOP            0x00
#define CGA_BLK             0x01
#define CGA_ABS             0x02
#define CGA_FLT             0x04


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * Flash program memory definitions.  For non-volatile storage of configurable
 * parameters.
 * 
 * + Word values must be stored in little endian (L|H) format.
 * + Row offset address for data must match assignment below.
 * + For convention all word values are placed first to respect word address
 *   boundries (even though 8-bit MCUs don't care).
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define FLASH_AR_VSET_L     0               //Output volts low byte
#define FLASH_AR_VSET_H     1               //Output volts high byte
#define FLASH_AR_VSET       (FLASH_AR_VSET_L / sizeof(uint16_t))
#define FLASH_AR_VMPP_L     2               //MPP volts low byte
#define FLASH_AR_VMPP_H     3               //MPP volts high byte
#define FLASH_AR_VMPP       (FLASH_AR_VMPP_L / sizeof(uint16_t))
#define FLASH_AR_CYCC_L     4               //Unit cycle count low byte
#define FLASH_AR_CYCC_H     5               //Unit cycle count high byte
#define FLASH_AR_CYCC       (FLASH_AR_CYCC_L / sizeof(uint16_t))
#define FLASH_AR_CSTS_L     6               //Cycle count last fault low byte
#define FLASH_AR_CSTS_H     7               //Cycle count last fault high byte
#define FLASH_AR_CSTS       (FLASH_AR_CSTS_L / sizeof(uint16_t))
#define FLASH_AR_VNOM_L     8               //Battery nominal voltage low byte
#define FLASH_AR_VNOM_H     9               //Battery nominal voltage high byte
#define FLASH_AR_VNOM       (FLASH_AR_VNOM_L / sizeof(uint16_t))
#define FLASH_AR_VBLK_L     10              //Bulk charge voltage low byte
#define FLASH_AR_VBLK_H     11              //Bulk charge voltage high byte
#define FLASH_AR_VBLK       (FLASH_AR_VBLK_L / sizeof(uint16_t))
#define FLASH_AR_VABS_L     12              //Absorption voltage low byte
#define FLASH_AR_VABS_H     13              //Absorption voltage high byte
#define FLASH_AR_VABS       (FLASH_AR_VABS_L / sizeof(uint16_t))
#define FLASH_AR_VFLT_L     14              //Float voltage low byte
#define FLASH_AR_VFLT_H     15              //Float voltage high byte
#define FLASH_AR_VFLT       (FLASH_AR_VFLT_L / sizeof(uint16_t))
#define FLASH_AR_TABS_L     16              //Absorption time low byte
#define FLASH_AR_TABS_H     17              //Absorption time high byte
#define FLASH_AR_TABS       (FLASH_AR_TABS_L / sizeof(uint16_t))
#define FLASH_AR_STS        18              //Last fault code
#define FLASH_AR_STATUS     19              //STATUS register
#define FLASH_AR_PCON       20              //PCON register
#define FLASH_AR_DC         21              //Max or open loop duty cycle
#define FLASH_AR_PLCADR     22              //Local PLC node address
#define FLASH_AR_CHGADR     23              //Charge controller logical address
#define FLASH_AR_CONTRL     24              //Control bits (ref. NVM constants)
#define FLASH_AR_BIU        25              //PLC BIU noise threshold
#define FLASH_AR_TXG        26              //PLC transmit gain
#define FLASH_AR_RXG        27              //PLC receiver gain
#define FLASH_AR_STSC       28              //Successive fault count

//Default data to initialize row with. Positions 30 & 31 reserved for checksum.
#define S_DF_FLASH_0        S_DF_VSET_L
#define S_DF_FLASH_1        S_DF_VSET_H
#define S_DF_FLASH_2        S_DF_VMPP_L
#define S_DF_FLASH_3        S_DF_VMPP_H
#define S_DF_FLASH_4        S_DF_TTIM_L
#define S_DF_FLASH_5        S_DF_TTIM_H
#define S_DF_FLASH_6        S_DF_TSTS_L
#define S_DF_FLASH_7        S_DF_TSTS_H
#define S_DF_FLASH_8        S_DF_VNOM_L
#define S_DF_FLASH_9        S_DF_VNOM_H
#define S_DF_FLASH_10       S_DF_VBLK_L
#define S_DF_FLASH_11       S_DF_VBLK_H
#define S_DF_FLASH_12       S_DF_VABS_L
#define S_DF_FLASH_13       S_DF_VABS_H
#define S_DF_FLASH_14       S_DF_VFLT_L
#define S_DF_FLASH_15       S_DF_VFLT_H
#define S_DF_FLASH_16       S_DF_TABS_L
#define S_DF_FLASH_17       S_DF_TABS_H
#define S_DF_FLASH_18       STS_NOF
#define S_DF_FLASH_19       S_DF_REG_STATUS
#define S_DF_FLASH_20       S_DF_REG_PCON
#define S_DF_FLASH_21       S_DC_RUN
#define S_DF_FLASH_22       S_DF_LCLADR
#define S_DF_FLASH_23       S_DF_CHGADR
#define S_DF_FLASH_24       S_DF_CONTRL
#define S_DF_FLASH_25       S_DF_BIU
#define S_DF_FLASH_26       S_DF_TXG
#define S_DF_FLASH_27       S_DF_RXG
#define S_DF_FLASH_28       S_DF_STSC
#define S_DF_FLASH_29       0

//Calculate default configuration row checksum value
#define S_DF_CFG_CS         (S_DF_FLASH_0+S_DF_FLASH_1+S_DF_FLASH_2+\
                                S_DF_FLASH_3+S_DF_FLASH_4+S_DF_FLASH_5+\
                                S_DF_FLASH_6+S_DF_FLASH_7+S_DF_FLASH_8+\
                                S_DF_FLASH_9+S_DF_FLASH_10+S_DF_FLASH_11+\
                                S_DF_FLASH_12+S_DF_FLASH_13+S_DF_FLASH_14+\
                                S_DF_FLASH_15+S_DF_FLASH_16+S_DF_FLASH_17+\
                                S_DF_FLASH_18+S_DF_FLASH_19+S_DF_FLASH_20+\
                                S_DF_FLASH_21+S_DF_FLASH_22+S_DF_FLASH_23+\
                                S_DF_FLASH_24+S_DF_FLASH_25+S_DF_FLASH_26+\
                                S_DF_FLASH_27+S_DF_FLASH_28+S_DF_FLASH_29)
#if(S_DF_CFG_CS < 256)
#define S_DF_FLASH_30       S_DF_CFG_CS
#define S_DF_FLASH_31       0
#else
#define S_DF_FLASH_30       (S_DF_CFG_CS & 0x00FF)
#define S_DF_FLASH_31       (S_DF_CFG_CS >> 8)
#endif


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * Function declarations.  See source file for explanation of each function.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void main(void);
void ConverterON(void);
void ConverterOFF(void);
void chgDutyCycle(uint16_t, uint16_t);
uint8_t cfgPLC(uint8_t);
uint8_t pingPLC(uint8_t);
__bit svcPLC(void);
void __interrupt() InterruptExec(void);
void Initialize(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * Preprocessor checks.  Maintain at end of file.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#if((S_DC_MAX*S_RT_TEMPDR) > CHAR_MAX)
#error "Max DC exceeds reduction factor."
#endif

#ifdef _OPT_DIS_CLOSL
#warning "DANGER! YOU HAVE SET OPEN LOOP OPERATION."
#endif


//End of file
