#!/bin/bash
#
# DESCRIPTION:
# Increments build number.  Reads current build # from build.txt and writes
# include file build.h with constant VERBUILD assigned.  The updated build
# number is also written back to build.txt
#
# Copyright (c) 2019 Brian Cornell, all rights reserved.
#
# DEPENDENCIES:  bash, rsync, rsync-hfs, tar, gzip, find, ssh, rm, test, sleep,
#                kill, chmod, scp, date.
#
# COMPONENTS:   build.sh - this script.
#               build.txt - contains integer build # on first line.
#		build.h - include file with build # defined.
#
# VERSION:  19111200.
#
# MODIFICATION HISTORY:
# 17 Jul 25, Brian Cornell: added variable $ConstName to allow version build 
#     constants from common & source to be evaluated in source include files.
#
# 9 Feb 25, Brian Cornell: added if block at end of script to execute another
#     command at the end of the script. This is intended to daisy-chain build
#     versioning for multiple MPLAB projects sharing a common source base.
#
#     Note that build versioning between the main & daisy chain source will not
#     match if there are not changes in both source with each build. Also, the
#     removal of a source file with no associated changes in remaining source
#     files will not trigger a new build version.
#
# 12 Nov 19, Brian Cornell, version 2:
#   + Full versioning of source using diff and only incrementing & saving
#     source when changes.  Creates versioning folders (with build # extension)
#     using FileRoot in source folder.  Update FileList with wildcard &
#     individual file names separated by spaces within parenthesis.
#
# 1 Oct 19, Brian Cornell:
#   + Created.
#
#

FileRoot="build"
FileList=()

ls *.c &>/dev/null
if [ $? = 0 ]
then
    FileList+=(*.c)
fi
ls *.h &>/dev/null
if [ $? = 0 ]
then
    FileList+=(*.h)
fi

BuildnumFil="$FileRoot.txt"
BuildincFil="$FileRoot.h"
BuildFolder=""
BuildNum=0
NewBuild=0
ConstName="_VBSOURCE"

exec 3<> $BuildnumFil
read BuildNum <&3
exec 3>&-
BuildFolder="$FileRoot.$BuildNum"

for df in "${FileList[@]}"
do
    diff $df $BuildFolder/$df &>/dev/null
    if [ $? != 0 ]
    then
        NewBuild=1
    fi
done

if [ $NewBuild != 0 ]
then
    let BuildNum=$BuildNum+1
    echo $BuildNum >$BuildnumFil
    BuildFolder="$FileRoot.$BuildNum"
    mkdir $BuildFolder

    echo "$0:  build $BuildNum"
    echo "//" >$BuildincFil
    echo "// Include file $BuildincFil" >>$BuildincFil
    echo "// Maintained by $0" >>$BuildincFil
    echo "// DO NOT EDIT" >>$BuildincFil
    echo "//" >>$BuildincFil
    echo "" >>$BuildincFil
    echo "#define $ConstName		$BuildNum" >>$BuildincFil

    for df in "${FileList[@]}"
    do
        cp "$df" "$BuildFolder/$df" &>/dev/null
    done
else
    echo "$0:  no build changes"
fi

if [ -n "${2}" ]
then
    cd $1
    $2
fi

#
#
# End of file.
