/*

DESCRIPTION:    Control firmware for development PV-panel based, interleave,
                flyback charger.

                Preliminary specifications:
                    + Input voltage: 28-45V DC
                    + Input current: maximum 18A
                    + Output voltage: 56V DC
                    + Output current: 9A
                    + Maximum continuous power: 500W
                    + Control mode: current
                    + Operating temperature: TBD

                An RS232 interface shared with ICSP provides an optional 
                console interface with a command set to monitor and control 
                operational parameters. Many parameters are stored in Non-
                Volatile Memory (NVM).

                RS232 settings: 115.2kbps, 8 data bits, 1 stop, no parity. The 
                console port is accessed over the micro-USB connector on the 
                PCB and is optimized for use with a GBIB-style control program.

                A status LED on the PCB indicates machine state:
                    + Red=fault
                    + Green=normal operation
                    + Red/Green=boot, HW or FW failure

SYNTAX:         n/a

KNOWN BUGS:     

COPYRIGHT:      (c) 2025, Brian Cornell, all rights reserved.

DEPENDENCIES:   see include file listing.

 TARGET MCU:    PIC16F1776

HW REVISION:    241209

VERSION:        3.0

MODIFICATION HISTORY:
 
     5 Jan 25, BWC, version 3 adapted from v2.0 branch.

    23 Jun 24, BWC, version 2 adapted from v1.0 branch of PVC code.
    

*/


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * COMPILER & CLIB INCLUDES.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <pic.h>
#include <xc.h>

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * FIRMWARE INCLUDES.
 * + binary:            boolean constants shared across firmware projects.
 * + sizes:             constants representing common object sizes, value 
 *                      limits, and characters.
 * + pic16f1776-map-v2: PIC specific peripheral alias, pin, NVM mapping, and
 *                      function declarations for shared functions.
 * + usart-v3:           usart function declarations & supporting constants,
 *                      objects, etc..
 * + config:            PIC config registers
 * + _xtal_freq:        sets the constant _XTAL_FREQ to the system oscillator
 *                      frequency (Fosc).  Used by the __delayxxx macros.
 * + usartc-v3:         EUSART peripheral user configuration constants.
 * + i2cm.h             I2C master constants, macros, function declarations.
 * + pvc:               main include file for this project.
 * + pic16f1776-vmv-v2: VMV configuration.
 * + plc                PLC constants, data definitions, macros shared with
 *                      remote console/controller.
 * 
 * PROJECT FOLDER ORGANIZATION & LOCATION OF INCLUDES.
 * /pvc-231029-fw-v3_0.X this (PVC) project.
 * /microchip/code      shared includes.
 * ../pvc-common        includes & code shared with remote console/controller.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#include <binary.h>                     //folder microchip/code
#include <sizes.h>                      //folder microchip/code
#include <pic16f1776-map-v2.h>          //folder microchip/code
#include <usart-v3.h>                   //folder microchip/code
#include "../pvc-common/config.h"
#ifndef _XTAL_FREQ
#include "../pvc-common/_xtal_freq.h"
#endif
#include "../pvc-common/usartc-v3.h"
#include "../pvc-common/i2cm.h"
#include "../pvc-common/pic16f1776-vmv-v2.h"
#include "../pvc-common/plc.h"          //Shared with remote console/controller
#include "pvc.h"


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * MODULE DATA.
 * + inv is the volts object storing input voltage acquisition data.
 * + outv is the volts object storing output voltage acquisition data.
 * + tp1 is the volts object storing the voltage analog of P1 temperature.
 * + tp2 is the volts object storing the voltage analog of P2 temperature.
 * + nvv is the volts object used to measure input voltage when updating NVM.
 * + plc is the I2C buffer for exchanging data with the PLC controller.
 * + chg is the charger telemetry object used to send data to the charge 
 *   controller.
 * + usart_nl is set TRUE to add a new line (NL) to the output of the 
 *   usart_wr_ui... functions.
 * + flag_ocp1 is set true by the ISR when an over-current is detected for P1.
 * + flag_ocp2 is set true by the ISR when an over-current is detected for P2.
 * + flag_ocle is set true by the phase over-current event detection blocks to
 *   indicated that a fault shutdown is required.
 * + Flags cv0, 1, 2 are set TRUE to indicate that valid console command 
 *   parameters have been set in function main objects cmdp1 / cmdp2.
 * + Flag flag_nvm is TRUE when the NVM update timer has expired.
 * + Flag flag_nvf is TRUE when NVM must be updated with a new fault code.
 * + Flag flag_run is TRUE when the boot cycle counter expires.
 * + Flag flag_pir is set TRUE to indicate a service required/problem with the 
 *   PLC.
 * + Flag flag_tmp is set TRUE when power reduction is in effect to reduce the
 *   converter's operating temperature.
 * + Flag flag_abs is TRUE when the charge algorithm absorption timer is active.
 * + Flag flag_chga is set TRUE when the standalone charge algorithm is enabled.
 * + reg_status is assigned the value of the STATUS register at boot.  Refer
 *   to the 'MCU FAILURE RECORDING' block for more detail.
 * + reg_pcon is assigned the value of the PCON register at boot.  Refer
 *   to the 'MCU FAILURE RECORDING' block for more detail.
 * + sts holds the status code, if any, associated with the machine state.
 * + state is a bit-mapped byte indicating the machine state (refer to STATE 
 *   MANAGEMENT AND STATUS CODES in include file).
 * + tmr_temp is a countdown timer for measuring converter temps. It is set to
 *   0 @ boot so temps are checked prior to starting the converter.
 * + tmr_rest is a countdown timer set when a recoverable fault occurs. It must
 *   be clear before a restart can occur.
 * + tmr_chga is a countdown timer to control the sampling rate of testing for
 *   charge state changes when the charge algorithm is active.
 * + plc_irqs is the IRQ status byte read from the PLC controller service block
 *   in the RUN loop.
 * + plc_bufs is the size of valid data (Tx or Rx) in buffer plc_buf.
 * + plc_rxi is the contents of the RX_MESSAGE_INFO register in the PLC.
 * + plc_cmd stores the command sent from the remote node.
 * + plc_rqa contains the logical address a response is expected from.
 * + plc_lga stores the remote node (source) logical address.
 * + plc_rxd stores the PLC receive data buffer from the last completed request.
 * + plc_rqt is the PLC request timeout timer.
 * + plc_txg stores the PLC Tx gain value loaded from NVM or assigned default.
 * + plc_rxg stores the PLC Rx gain value loaded from NVM or assigned default.
 * + plc_sts contains the positional status code within the PLC Service 
 *   block to record the appropriate fault code.
 * + cga_state indicates the charge algorithm operating state.
 * + occp1 & occp2 are the over-current counters maintained in the fault
 *   monitoring section of the RUN loop.
 * + tmr_kpa is the keep alive timer decremented by the ISR. In controlled mode 
 *   it functions as a safety feature: the converter must receive valid 
 *   communication from the charge controller within S_LM_KPA_TIM seconds before
 *   a fault is set.
 * + cga_vnom is the standalone charge algorithm nominal battery voltage.
 * + cga_vblk is the standalone charge algorithm bulk charging voltage.
 * + cga_vabs is the standalone charge algorithm absorption charging voltage.
 * + cga_vflt is the standalone charge algorithm battery float voltage.
 * + cga_tabs is the standalone charge algorithm absorption float time.
 * + tmr_abs is the charge algorithm absorption timer decremented by the ISR.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
volts   inv={.channel=PIN_IA_INV, .mvr=S_RT_ADC_MV_H, .adc1v=S_QY_ADC_1V_H,\
            .scale=S_RT_VIN},
        outv={.channel=PIN_IA_OUTV, .mvr=S_RT_ADC_MV_H, .adc1v=S_QY_ADC_1V_H,\
            .scale=S_RT_VOUT},
        tp1={.channel=PIN_IA_TFP1, .mvr=S_RT_ADC_MV_H, .adc1v=S_QY_ADC_1V_H,\
            .scale=S_RT_TEMP},
        tp2={.channel=PIN_IA_TFP2, .mvr=S_RT_ADC_MV_H, .adc1v=S_QY_ADC_1V_H,\
            .scale=S_RT_TEMP},
        nvv={.channel=PIN_IA_INV, .mvr=S_RT_ADC_MV_L, .adc1v=S_QY_ADC_1V_L,\
            .scale=S_RT_NVV};

plci2c plc;
chargert chg;

extern __bit usart_nl;

__bit   flag_ocp1,
        flag_ocp2,
        flag_ocle,
        flag_cv0,
        flag_cv1,
        flag_cv2,
        flag_nvm,
        flag_nvf,
        flag_run,
        flag_pir,
        flag_pxx,
        flag_tmp,
        flag_abs,
        flag_chga;
uint8_t reg_status,
        reg_pcon,
        sts=STS_NOF,
        state,
        tmr_temp=0,
        tmr_rest=0,
        tmr_chga=0,
        plc_irqs,
        plc_bufs,
        plc_rxi,
        plc_cmd,
        plc_rqa,
        plc_lga,
        plc_rxd[PLC_SZ_DATA],
        plc_rqt=0,
        plc_txg=S_DF_TXG,
        plc_rxg=S_DF_RXG,
        plc_sts,
        cga_state=CGA_STOP;
uint16_t occp1=0,
         occp2=0,
         tmr_kpa=S_LM_KPA_TIM,
         tmr_abs=0,
         cga_tabs=0;

union {
    uint8_t     b[sizeof(uint16_t)];
    uint16_t    w;
} cga_vnom={.w=0};
union {
    uint8_t     b[sizeof(uint16_t)];
    uint16_t    w;
} cga_vblk={.w=0};
union {
    uint8_t     b[sizeof(uint16_t)];
    uint16_t    w;
} cga_vabs={.w=0};
union {
    uint8_t     b[sizeof(uint16_t)];
    uint16_t    w;
} cga_vflt={.w=0};

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * I2C MODULE EXTERNAL DATA.
 * + i2c_idle is TRUE when there is no active read/write in process.
 * + i2c_rsen is TRUE to issue an I2C bus RESTART at the end of a read/write
 *   to keep the bus active.
 * + i2c_rmode controls fore/background reads, initialized in function 
 *   i2c_m_cfg
 * + i2c_tor is the read timout timer.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
extern __bit i2c_idle,
             i2c_rsen,
             i2c_rmode;
extern uint8_t i2c_tor;


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * OPERATING CONFIGURATION MEMORY.
 * 
 * Non-volatile parameter storage & management. Object nvm_cfg resides in 
 * program flash at the top of the high-endurance section (to make the lower 
 * portion contiguous & available for code).  Object ram_cfg is the working 
 * copy in RAM.
 * 
 * Initialization of nvm_cfg & ram_cfg MUST match: these are system defaults and 
 * enable recovery after NVM corruption or to restore via console commands.  
 * Refer to detailed explanation of NVM management at beginning of function 
 * Initialize.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
const nvmcfg nvm_cfg __at(0x1FE0)={.byte={S_DF_FLASH_0,S_DF_FLASH_1,\
    S_DF_FLASH_2,S_DF_FLASH_3,S_DF_FLASH_4,S_DF_FLASH_5,S_DF_FLASH_6,\
    S_DF_FLASH_7,S_DF_FLASH_8,S_DF_FLASH_9,S_DF_FLASH_10,S_DF_FLASH_11,\
    S_DF_FLASH_12,S_DF_FLASH_13,S_DF_FLASH_14,S_DF_FLASH_15,S_DF_FLASH_16,\
    S_DF_FLASH_17,S_DF_FLASH_18,S_DF_FLASH_19,S_DF_FLASH_20,S_DF_FLASH_21,\
    S_DF_FLASH_22,S_DF_FLASH_23,S_DF_FLASH_24,S_DF_FLASH_25,S_DF_FLASH_26,\
    S_DF_FLASH_27,S_DF_FLASH_28,S_DF_FLASH_29,S_DF_FLASH_30,S_DF_FLASH_31}};
nvmcfg ram_cfg={.byte={S_DF_FLASH_0,S_DF_FLASH_1,S_DF_FLASH_2,S_DF_FLASH_3,\
    S_DF_FLASH_4,S_DF_FLASH_5,S_DF_FLASH_6,S_DF_FLASH_7,S_DF_FLASH_8,\
    S_DF_FLASH_9,S_DF_FLASH_10,S_DF_FLASH_11,S_DF_FLASH_12,S_DF_FLASH_13,\
    S_DF_FLASH_14,S_DF_FLASH_15,S_DF_FLASH_16,S_DF_FLASH_17,S_DF_FLASH_18,\
    S_DF_FLASH_19,S_DF_FLASH_20,S_DF_FLASH_21,S_DF_FLASH_22,S_DF_FLASH_23,\
    S_DF_FLASH_24,S_DF_FLASH_25,S_DF_FLASH_26,S_DF_FLASH_27,S_DF_FLASH_28,\
    S_DF_FLASH_29,S_DF_FLASH_30,S_DF_FLASH_31}};



/*Function main.  Control program entry point (from root code) for control 
 * program. All aspects of operation begin & end here. At entry the 
 * Initialization function is called to configure peripherals, data objects, 
 * etc. The STATUS & PCON register values are saved in case an MCU trap was
 * encountered. 
 * 
 * The WDT (Watchdog) timer is enabled for hard fault recovery and the RUN loop 
 * entered: it is an infinite loop that consists of several discrete blocks that
 * manage all aspects of operation. They are ordered by dependencies (e.g. data
 * collection must happen before fault detection) and/or criticality (e.g. fault
 * shutdown before console service). Refer to the documentation within each 
 * block for details.
 * 
 * Returns:     never
 *
 * Parameters:  none
 *
*/
void main(void)
{
    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * FUNCTION DATA.
     * + cmdb points to the start of the console command buffer.
     * + counter is a general purpose loop counter and scratch variable.  It 
     *   WILL NOT be used for multiple iterations thru the RUN loop.
     * + cmdb_sz is the size of the console command buffer.
     * + cmdp0 is the integer value representing the console command.
     * + cmdp1 is the integer value of the first console command parameter.
     * + cmdp2 is the integer value of the second console command parameter.
     * + ctmp represents the hottest temperature from the phases in a given 
     *   sample.
     * + cdc represents the highest duty cycle value of the phases in a given
     *   sample.
     * + ovcc counts the number of over-current faults that have occured since
     *   boot.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    char *cmdb;
    uint8_t counter=0,
            cmdb_sz,
            cmdp0=CMDNONCMD,
            cmdp1,
            ovcc=0;
    uint16_t cmdp2,
             dcp1=0,
             dcp2=0,
             ctmp,
             cdc;


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * MCU FAILURE CAPTURE.
     * 
     * If an MCU failure (e.g. stack overflow, watchdog, etc.) is detected at 
     * boot it's important to understand what caused the fault. The STATUS &
     * PCON registers are saved at the start of the main function for use by
     * function Initialize.
     +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    reg_status=S_AL_REG_STATUS;
    reg_pcon=S_AL_REG_PCON;


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * INITIALIZATION.
     * 
     * Function Initialize tests for NVM & MCU faults, loads configuration data,
     * and configures the MCU and on-board peripherals. A subsequent delay 
     * provides settle time and must be long enough for the PLC controller to 
     * boot before attempting to configure.
     * 
     * The PLC initialization is completed regardless of possible initialization
     * errors to support possible recovery via commands from the charge 
     * controller or remote console.
     +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    Initialize();
    __delay_ms(S_TM_INT);

    if ((counter=cfgPLC(TRUE)) != STS_NOF)
    {
        SETSTSF(counter);
    }


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * RUN LOOP PRIORITIZATION & WATCHDOG MANAGEMENT.
     * 
     * Once entered, the run-time loop's only exit points are a reboot or power
     * cycle. The loop consists of several discrete blocks responsible for 
     * data collection, monitoring, and control.  They are organized in order
     * of criticality and/or dependency of subsequent blocks.  Categorically the
     * order is:
     *      + Data collection,
     *      + Fault monitoring,
     *      + Shut-down,
     *      + NVM updating / fault recording,
     *      + Standalone charge algorithm,
     *      + Fault recovery,
     *      + Boot (autostart) and restart,
     *      + Status LED,
     *      + PLC service,
     *      + Console service.
     * 
     * The watchdog timer is expected to be on in the run loop regardless of
     * state.  It should be set to a value that accounts for the longest loop
     * time with at least a 100% margin.  If it must be disabled in anticipation
     * of a long delay then it is the responsibility of that code block to 
     * re-enable.
     +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    S_AL_WDT_ENABLE=TRUE;                   //Enable watchdog

    while (TRUE)                            //Begin RUN loop
    {
        CLRWDT();                           //Clear watchdog timer

#ifdef _OPT_LOOP_TIMING
        PIN_OD_STATUS^=TRUE;
#endif
        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * DATA COLLECTION.
         * 
         * This block measures system voltages & currents and COG (duty cycle) 
         * timing.
         * 
         * Duty cycle is captured using the timer gate pin timing function. The 
         * gate pin is connected to the COG-A output. To start timing the timer 
         * is cleared and the go bit set. Timing begins on the next rising edge 
         * of the COG-A pin and stops when it falls.
         * 
         * To eliminate fixed delays waiting for the timing measurement to
         * complete, measurement relies on the RUN loop timing exceeding it. 
         * Timing is saved to the dcxx objects prior to the start of the next 
         * measurement iteration. Hence, the reported DC is always one 'RUN
         * loop' behind.
         * 
         * Since temperature is a relatively slow moving parameter it is only
         * measured periodically to reduce average RUN loop execution time. 
         * tmr_temp is maintained by the system timer ISR.
         *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        ReadVoltage(&inv);
        ReadVoltage(&outv);

        //Measure duty cycle
        S_AL_DCTEN_P1=FALSE;                    //Ensure timing has stopped
        S_AL_DCTEN_P2=FALSE;
        dcp1=S_AL_DCTMR_P1;                     //Save DC
        dcp2=S_AL_DCTMR_P2;
        S_AL_DCTMR_P1=0;                        //Clear (high byte is not used)
        S_AL_DCTMR_P2=0;
        S_AL_DCTEN_P1=TRUE;                     //Start capture
        S_AL_DCTEN_P2=TRUE;
        if (dcp1>dcp2) cdc=dcp1;                //Assign highest DC
        else cdc=dcp2;

        if (!(tmr_temp))                        //Time to check temps
        {
            ReadVoltage(&tp1);
            ReadVoltage(&tp2);
            tmr_temp=S_TM_TEMP;                 //Reset timer
            if (tp1.vmv<tp2.vmv) ctmp=tp1.vmv;  //Assign hottest value
            else ctmp=tp2.vmv;
        }


#ifdef _OPT_SD_FAULT
        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * FAULT MONITORING.
         * 
         * This block tests for various fault conditions that set the operating
         * STATE to recoverable (STE_RCVRF) or non-recoverable (STE_FAULT). With
         * recoverable faults the firmware will resume normal operation when the
         * recovery criteria are met. A non-recoverable fault requires a reboot
         * command or power cycle.
         *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        {

            /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
             * HIGH VOUT.
             * 
             * High Vout is treated as a safety fault and detection & shutdown 
             * is done in hardware using a DAC, comparator, and CLC to command 
             * the shutdown of the COGs. This logic tests for a discrepancy 
             * between one of the COG auto-shutdown bits in the machine state 
             * object: the status is set and a proper shutdown of the plant is 
             * completed at the bottom of the fault monitoring block.
             *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
            if ((S_AL_COGAS_P1) && (state & STE_RUN))
            {
                SETSTSF(STS_HOV);
            }

            /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
             * HIGH TEMPERATURE.
             * 
             * First try reducing DC starting at S_LM_TMP_DCL, then shutdown at 
             * hard limit of S_LM_FLT_TMP. Note that temperatures in object ctmp 
             * are represented by the vmv voltage across the NTC voltage 
             * divider: a lower voltage represents a higher temperature.
             *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
            if (ctmp<S_LM_FLT_TMP)                      //Shutdown d/t overheat
            {
                SETSTSR(STS_TMP);
            }
            else if ((ctmp>S_LM_TMP_DCU) && (flag_tmp)) //Release DC limit
            {
                flag_tmp=FALSE;
                chgDutyCycle(0, (uint16_t) ram_cfg.byte[FLASH_AR_DC]);
            }
            /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
             * Converter is hot, but still safe to run. Reduce by 
             * S_RT_TEMPRD/S_RT_TEMPDR in an attempt to avoid complete shutdown
             * d/t overheat while still delivering some power. The duty cycle 
             * check makes sure that converter is at least operating at 
             * S_RT_TEMPDR. Object flag_tmp is deliberately set inside of DC 
             * check because depending on the impedance of the PV/converter 
             * array & battery stack it is possible for the inverter to 
             * oscillate the battery bus voltage.
             *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
            else if ((ctmp<S_LM_TMP_DCL) && (!(flag_tmp)))
            {
                if (cdc>S_RT_TEMPDR)
                {
                    flag_tmp=TRUE;
                    chgDutyCycle(0, (uint16_t)\
                            ((cdc/S_RT_TEMPDR)*(S_RT_TEMPDR-S_RT_TEMPRD)));
                }
            }
#endif

            /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
             * EXCESSIVE SWITCH OVER-CURRENT EVENTS.
             * 
             * A compile option is provided to set a non-recoverable fault
             * when the heuristic over-current count limit is exceeded. The 
             * counters remain active when fault detection is disabled to allow 
             * console observation for trouble-shooting or adjustment of the 
             * limit.
             * 
             * Current mode control ensures that the plant remains within its 
             * SOA, but repeated over-current events with successive restarts 
             * indicates a more serious problem. S_LM_FLT_OCC sets the limit on 
             * the over-current events that can occur before the converter is 
             * shut down. Object flag_ocle is set by all phases when the limit 
             * is reached and a conditional test common to all phases determines 
             * if the converter shuts down with a recoverable or non-recoverable 
             * fault based on incrementing object ovcc & testing against 
             * S_LM_OVC_CYC. The over-current interrupt is only re-enabled when 
             * below the S_LM_FLT_OCC limit: this ensures that events occurring
             * after shutdown has been commanded do not erroneously increment 
             * ovcc.
             +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
            if (flag_ocp1)
            {
                flag_ocp1=FALSE;
#ifdef _OPT_OC_FAULT
                if (++occp1 > S_LM_FLT_OCC)
                    flag_ocle=TRUE;                     //Fault limit exceeded
                else
                S_AL_OCIE_P1=TRUE;                      //Re-enable interrupt
#else
                ++occp1;
#endif
            }
            else if (occp1) --occp1;

            if (flag_ocp2)
            {
                flag_ocp2=FALSE;
#ifdef _OPT_OC_FAULT
                if (++occp2 > S_LM_FLT_OCC)
                    flag_ocle=TRUE;                     //Fault limit exceeded
                else
                S_AL_OCIE_P2=TRUE;                      //Re-enable interrupt
#else
                ++occp2;
#endif
            }
            else if (occp2) --occp2;

            //Process the over-current fault based on object ovcc
            if (flag_ocle)
            {
                flag_ocle=FALSE;
                if (++ovcc>S_LM_OVC_CYC)
                {
                    SETSTSF(STS_OCM);
                }
                else
                {
                    SETSTSR(STS_OCL);
                }
            }

#ifdef _OPT_SD_FAULT
            /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
             * KEEP ALIVE TIMER EXPIRATION.
             * 
             * In CONTROLLED mode the converter must receive valid communication 
             * from the charge controller every S_LM_KPA_TIM seconds. The PLC
             * control block resets the keep alive timer (tmr_kpa) each time a 
             * valid communication is processed. For safety the charger is shut
             * down with a fault to prevent battery over-charging or other 
             * safety hazards.
             *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
            if ((ram_cfg.byte[FLASH_AR_CONTRL] & S_BL_ECTRL) && (!(tmr_kpa)))
            {
                SETSTSR(STS_KPA);
            }
#endif
        }


        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * FAULT SHUT-DOWN.
         * 
         * Stop the converter in the event of a FAULT when in the RUN state. The
         * PRG GO bit is used to determine when the converter is running (the 
         * COG auto-shutdown bit cannot be used because the high voltage fault
         * can set this). When a fault is detected, object state is changed from
         * RUN to the recoverable or fault value.
         *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#ifdef _OPT_SD_FAULT
        if ((S_AL_PRGGO_P1) && !(state & STE_RUN))
        {
            ConverterOFF();
        }
#endif

        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * NVM STATUS RECORDING.
         * 
         * This block updates NVM with status codes when a fault occurs, for 
         * operational updates made elsewhere in the RUN loop, or to update the 
         * unit's cycle time. Flag flag_nvm is set TRUE by the timing block in 
         * the ISR when the boot cycle timer expires or updates are made to
         * NVM values via the console or charge controller.
         * 
         * Flag flag_nvf is set TRUE when a fault occurs that should be recorded
         * in NVM. Both the status code & cycle count are saved.
         * 
         * To prevent the corruption of NVM, it is critical that Vin be stable
         * since the write cycle requires over 5mS. To ensure this, Vin is 
         * measured using the 'low' range which can read a maximum of 2.048V.
         * The PIC does not operate with a Vdd below 2.3V which ensures that
         * the ADC conversion will accurately render Vin. A Vin of 20V is 
         * considered safe because: 1) it provides a 3V cushion from the bias
         * PS UVLO of 17V, 2) the converter will be idle @ Vin~20V because 
         * MPP values are expected to be higher for the panels this converter
         * is designed to operate with and, 3) the hold-up time starting at
         * Vin~20V is ~ 20mS.
         * 
         * A Vin of 20V presents on the PIC's sense pin as 800mV (divider ratio
         * of 25), and using low range the expected ADC value is 400 (.8/.002).
         * The ADC value as opposed to the VMV is used for greater accuracy.
         * 
         * If Vin is insufficient the NVM flags will remain set to continue 
         * attempts until successful or the unit is powered off.
         +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#ifdef _OPT_DIS_NVF
        if (flag_nvm)
#else
        if (flag_nvm | flag_nvf)
#endif
        {
            ReadVoltage(&nvv);
            if (nvv.adc > S_LM_INV_NVM)
            {
                /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                 * Only update fault data when committed to writing and when 
                 * there is a fault value to record (otherwise, clean boots 
                 * overwrite the fault code and the record of what happened &
                 * when is lost). If the same code is being written to NVM 
                 * update the successive count object if less than the maximum 
                 * value that the object size can hold. 
                 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
                if (sts != STS_NOF)
                {
                    if (ram_cfg.byte[FLASH_AR_STS] == sts)  //Multiple events
                    {
                            if (ram_cfg.byte[FLASH_AR_STSC] < S_LM_U8)
                                ++ram_cfg.byte[FLASH_AR_STSC];
                    }
                    else ram_cfg.byte[FLASH_AR_STSC]=1;     //First event
                    ram_cfg.byte[FLASH_AR_STS]=sts;
                    ram_cfg.word[FLASH_AR_CSTS]=ram_cfg.word[FLASH_AR_CYCC];
                }
                writeFlashRow((uint16_t) &nvm_cfg, (unsigned char *) &ram_cfg);

                //Clear NVM update flags
                flag_nvm=FALSE;
                flag_nvf=FALSE;
            }
        }


        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * STANDALONE CHARGE ALGORITHM.
         * 
         * Sample charging state every S_TM_CHGA (via object tmr_chga) when 
         * charge algorithm is active.
         *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        if ((ram_cfg.byte[FLASH_AR_CONTRL] & S_BL_ECHGA) && (!(tmr_chga)))
        {
            tmr_chga=S_TM_CHGA;                     //Reset timer
            if ((state & STE_STOP) && (outv.adc < cga_vnom.w))
            {
                /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                 *                      Start charge cycle
                 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
                DAC1REFH=cga_vblk.b[S_AR_U16_H];
                DAC1REFL=cga_vblk.b[S_AR_U16_L];
                DAC1LD=TRUE;                        //Load bulk charge value
                cga_state=CGA_BLK;                  //Set charge operating state
                ConverterON();                      //Start converter
            }
            else if (state & STE_RUN)
            {
                /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                 *                     Bulk charging active
                 * 
                 * Bulk charging does not terminate until the MPP loop isn't 
                 * active AND output voltage meets or exceeds the bulk charging
                 * voltage.
                 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
                if (cga_state & CGA_BLK)
                {
                    if ((inv.adc > ram_cfg.word[FLASH_AR_VMPP]) &&\
                            (outv.adc >= cga_vblk.w))
                    {
                        if (cga_vabs.w)             //Apply absorption charge
                        {
                            DAC1REFH=cga_vabs.b[S_AR_U16_H];
                            DAC1REFL=cga_vabs.b[S_AR_U16_L];
                            DAC1LD=TRUE;
                            cga_state=CGA_ABS;
                            flag_abs=TRUE;
                            tmr_abs=cga_tabs;
                        }
                        else if (cga_vflt.w)        //Apply float charge
                        {
                            DAC1REFH=cga_vflt.b[S_AR_U16_H];
                            DAC1REFL=cga_vflt.b[S_AR_U16_L];
                            DAC1LD=TRUE;
                            cga_state=CGA_FLT;
                        }
                        else                        //End charge cycle
                        {
                            cga_state=CGA_STOP;
                            ConverterOFF();
                        }
                    }
                }
                /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                 *                  Absorption charging active
                 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
                else if (cga_state & CGA_ABS)
                {
                    if (!(flag_abs))                //Absorption timer expired
                    {
                        if (cga_vflt.w)             //Apply float charge
                        {
                            DAC1REFH=cga_vflt.b[S_AR_U16_H];
                            DAC1REFL=cga_vflt.b[S_AR_U16_L];
                            DAC1LD=TRUE;
                            cga_state=CGA_FLT;
                        }
                        else                        //End charge cycle
                        {
                            cga_state=CGA_STOP;
                            ConverterOFF();
                        }
                    }
                }
                /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                 *                   Float charging active
                 * 
                 * A new charging cycle is started from the float stage but not
                 * bulk or absorption. This is done on the logic that the
                 * absorption voltage is close to bulk and it prevents the rapid
                 * oscillation of output voltages.
                 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
                else if (outv.adc < cga_vnom.w)
                {
                    DAC1REFH=cga_vblk.b[S_AR_U16_H];
                    DAC1REFL=cga_vblk.b[S_AR_U16_L];
                    DAC1LD=TRUE;                    //Load bulk charge value
                    cga_state=CGA_BLK;              //Set charge operating state
                }
            }
        }


        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * RECOVERABLE FAULT MONITORING.
         * 
         * This block uses waterfall processing to clear recoverable fault
         * conditions.
         * 
         * PLC SERVICE block recovery is handled separately as a conditional 
         * block at the bottom of the PLC SERVICE block....
         *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#ifdef _OPT_RC_FAULT
        while ((state & STE_RCVRF) && (!(tmr_rest)))
        {
            /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
             * Temperature recovery: allow restart once temperature cools to
             * S_LM_RCV_TMP. Function ConverterON resets max. DC to NVM value.
             +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
            if ((sts==STS_TMP) && (ctmp<S_LM_RCV_TMP)) break;

            /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
             * Over-current recovery: No explicit test required for over-
             * current - just restart.
             +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

            /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
             * Keep alive timer recovery: restart if KPA has been reset.
             +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
            if ((sts==STS_KPA) && !(tmr_kpa)) break;    //NOT reset

            /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
             * Faults cleared: setup for restart.
             +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
            state=STE_RSTRT;
            sts=STS_NOF;
            break;
        }
#endif


        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * BOOT AND RECOVERY START.
         * 
         * Completes the boot process based on NVM control bit values. Control
         * bit precedence (enabled bit set):
         *  + CONTROLLED
         *  + CHARGE
         *  + AUTOSTART
         * 
         * A default unrecoverable fault is set if all bit tests are FALSE.
         * Since the initialize routine evaluates the control bits to respect
         * precedence this should never happen.
         * 
         * Any operational mode that auto-starts the converter cannot do so
         * prior to the ISR boot cycle counter timer setting object flag_run 
         * TRUE.
         *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        if (state & STE_BTRRT)
        {
            if (ram_cfg.byte[FLASH_AR_CONTRL]==S_BL_EMANU) state=STE_STOP;
            else if (flag_run)
            {
                if (ram_cfg.byte[FLASH_AR_CONTRL] & S_BL_ECTRL)
                {
                    if (ram_cfg.byte[FLASH_AR_PLCADR]) ConverterON();
                    else
                    {
                        SETSTSF(STS_ADR);
                    }
                }
                else if (ram_cfg.byte[FLASH_AR_CONTRL] & S_BL_ECHGA)
                    state=STE_STOP;
                else if (ram_cfg.byte[FLASH_AR_CONTRL] & S_BL_EAUTO)
                    ConverterON();
                else
                {
                    SETSTSF(STS_IVC);                   //Invalid control bits
                }
            }
        }


#ifndef _OPT_LOOP_TIMING
        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * LED MANAGEMENT.
         * Set the LED color based on machine state:
         *  + Red=[recoverable] fault
         *  + Green=stopped, idle, run, and technically boot but that state 
         *    cannot occur if the first RUN loop has made it this far.
         *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        if (state & STE_RCRFT) PIN_OD_LED=TRUE;
        else PIN_OD_LED=FALSE;
#endif


        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * PLC SERVICE.
         * 
         * Polls the PLC interrupt request (IRQ) pin and when true calls svcPLC
         * to retrieve data & reset the PLC for the next message.
         * 
         * If flag_pir is TRUE, a prior IRQ service was not completed 
         * successfully and the system is already in a FAULT state. To avoid
         * further complications to the malfunction the IRQ is not serviced. An
         * EXCLUSIVE OR evaluation is not used because the fault could occur 
         * further in the service routine after the PLC IRQ is reset.
         * 
         * Object plc_cmd is cleared at the beginning of service (by svcPLC) to 
         * prevent the possibility of re-processing a command when an 
         * unanticipated condition occurs. Hence, plc_cmd should not be used 
         * outside of function svcPLC and this block.
         * 
         * Object plc_sts stores the fault code relevant to each processing 
         * section for capture & NVM logging should a fault occur (at the 
         * bottom of the PLC service block).
         *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        if (PIN_ID_PIR && !(flag_pir))
        {
            while (TRUE)
            {
                if (svcPLC()==TRUE) break;

                /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                 * PROCESS RECEIVED DATA BASED ON COMMAND.
                 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
                plc_sts=STS_PPC;                            //Fault status code
                if (plc_cmd==PLC_RCD_RBT)                   //Reboot converter
                    cmdp0=CMDXRTMCU;
                else if (plc_cmd==PLC_RCD_RNV)              //Set NVM defaults
                    cmdp0=CMDXNVDFL;
                else if (plc_cmd==PLC_RCD_STR)              //Start converter
                    ConverterON();
                else if (plc_cmd==PLC_RCD_STP)              //Stop converter
                    ConverterOFF();
                //Vout increment/decrement
                else if (plc_cmd==PLC_RCD_VOI)
                {
                    union {
                        uint8_t     b[sizeof(uint16_t)];
                        uint16_t    w;
                    } vodac;

                    vodac.b[S_AR_U16_L]=DAC1REFL;
                    vodac.b[S_AR_U16_H]=DAC1REFH;

                    if (plc_rxd[PLC_OF_RXD]) vodac.w+=S_VL_VOUTINC; //Increment
                    else vodac.w-=S_VL_VOUTINC;                     //Decrement

                    if (vodac.w<=S_LM_MAX_VSET)             //Limit increase
                    {
                        DAC1REFL=vodac.b[S_AR_U16_L];
                        DAC1REFH=vodac.b[S_AR_U16_H];
                        DAC1LD=TRUE;                        //Load new value
                    }
                }
                /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                 * Received remote node PLC version (ping): display the 
                 * approximate time in 10mS increments.
                 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
                else if ((plc_cmd==PLC_RCD_GRV) && (plc_rqt))
                {
                    if (plc_lga==plc_rqa)
                        usart_wr_ui16((uint16_t) (PLC_TM_RQT-plc_rqt));
                    plc_rqt=0;
                }
                //Set node logical address 
                else if (plc_cmd==PLC_RCD_LAD)
                {
                    ram_cfg.byte[FLASH_AR_PLCADR]=plc_rxd[PLC_OF_RXD];
                    flag_nvm=TRUE;                          //Update NVM
                }
                //Set charge controller logical address
                else if (plc_cmd==PLC_RCD_CCA)
                {
                    ram_cfg.byte[FLASH_AR_CHGADR]=plc_rxd[PLC_OF_RXD];
                    flag_nvm=TRUE;                          //Update NVM
                }
                //Set MPP voltage
                else if (plc_cmd==PLC_RCD_SMV)
                {
                    ram_cfg.word[FLASH_AR_VMPP]=((setv *) plc_rxd)->mpp;
                    DAC5REFH=ram_cfg.byte[FLASH_AR_VMPP_H];
                    DAC5REFL=ram_cfg.byte[FLASH_AR_VMPP_L];
                    DAC5LD=TRUE;                            //Load new value
                    flag_nvm=TRUE;                          //Update NVM
                }
                //Set output voltage
                else if (plc_cmd==PLC_RCD_SOV)
                {
                    //Check for unsafe high Vout
                    if ((((setv *) plc_rxd)->out)<=S_LM_MAX_VSET)
                    {
                        ram_cfg.word[FLASH_AR_VSET]=((setv *) plc_rxd)->out;
                        DAC1REFH=ram_cfg.byte[FLASH_AR_VSET_H];
                        DAC1REFL=ram_cfg.byte[FLASH_AR_VSET_L];
                        DAC1LD=TRUE;                        //Load new value
                        flag_nvm=TRUE;                      //Update NVM
                    }
                }
                //Clear fault from NVM
                else if (plc_cmd==PLC_RCD_CFT)
                {
                    ram_cfg.word[FLASH_AR_CSTS]=0;
                    ram_cfg.byte[FLASH_AR_STSC]=0;
                    ram_cfg.byte[FLASH_AR_STS]=STS_NOF;
                    ram_cfg.byte[FLASH_AR_STATUS]=S_DF_REG_STATUS;
                    ram_cfg.byte[FLASH_AR_PCON]=S_DF_REG_PCON;
                    flag_nvm=TRUE;                          //Update NVM
                }
                //Set operating mode
                else if (plc_cmd==PLC_RCD_MOD)
                {
                    //Only update NVM if valid mode sent
                    flag_nvm=TRUE;
                    if (plc_rxd[PLC_OF_RXD]==S_BL_ECTRL)
                        ram_cfg.byte[FLASH_AR_CONTRL]=S_BL_ECTRL;
                    else if (plc_rxd[PLC_OF_RXD]==S_BL_ECHGA)
                        ram_cfg.byte[FLASH_AR_CONTRL]=S_BL_ECHGA;
                    else if (plc_rxd[PLC_OF_RXD]==S_BL_EAUTO)
                        ram_cfg.byte[FLASH_AR_CONTRL]=S_BL_EAUTO;
                    else if (plc_rxd[PLC_OF_RXD]==S_BL_EMANU)
                        ram_cfg.byte[FLASH_AR_CONTRL]=S_BL_EMANU;
                    else flag_nvm=FALSE;                    //Invalid, skip NVM
                }
                //Send operating voltages
                else if (plc_cmd==PLC_RCD_GOP)
                {
                    //Set tx cfg, destination addr, cmd ID, & data
                    plc.gov.size=(sizeof(txcfg)+sizeof(opv));
                    plc.gov.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.gov.txc.reg=PLC_OF_RTC;
                    plc.gov.txc.cfg=PLC_CF_RTC;
                    plc.gov.txc.lga[PLC_OF_WDT]=plc_lga;
                    plc.gov.txc.cmd=PLC_RCD_GOP;
                    plc.gov.ov.inv=inv.adc;
                    plc.gov.ov.outv=outv.adc;
                    plc.gov.ov.cdc=cdc;
                    WRPLCNACK();

                    //Send response
                    i2c_rsen=FALSE;                         //Release I2C bus
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=(PLC_RTL_SND | sizeof(opv));
                    WRPLCNACK();
                }
                //Send telemetry data
                else if (plc_cmd==PLC_RCD_TEL)
                {
                    //Set tx cfg, destination addr, cmd ID, & data
                    plc.tmy.size=(sizeof(txcfg)+sizeof(tlmtry));
                    plc.tmy.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.tmy.txc.reg=PLC_OF_RTC;
                    plc.tmy.txc.cfg=PLC_CF_RTC;
                    plc.tmy.txc.lga[PLC_OF_WDT]=plc_lga;
                    plc.tmy.txc.cmd=PLC_RCD_TEL;
                    plc.tmy.ty.cgas=cga_state;
                    plc.tmy.ty.state=state;
                    plc.tmy.ty.sts=sts;
                    plc.tmy.ty.ccadr=ram_cfg.byte[FLASH_AR_CHGADR];
                    plc.tmy.ty.ctrl=ram_cfg.byte[FLASH_AR_CONTRL];
                    plc.tmy.ty.ttime=ram_cfg.word[FLASH_AR_CYCC];
                    plc.tmy.ty.inv=inv.adc;
                    plc.tmy.ty.outv=outv.adc;
                    plc.tmy.ty.tp1=tp1.vmv;
                    plc.tmy.ty.tp2=tp2.vmv;
                    plc.tmy.ty.dcp1=dcp1;
                    plc.tmy.ty.dcp2=dcp2;
                    WRPLCNACK();

                    //Send response
                    i2c_rsen=FALSE;                         //Release I2C bus
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=(PLC_RTL_SND | sizeof(tlmtry));
                    WRPLCNACK();
                }
                //Send set voltages
                else if (plc_cmd==PLC_RCD_GSV)
                {
                    //Set tx cfg, destination addr, cmd ID, & data
                    plc.gsv.size=(sizeof(txcfg)+sizeof(setv));
                    plc.gsv.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.gsv.txc.reg=PLC_OF_RTC;
                    plc.gsv.txc.cfg=PLC_CF_RTC;
                    plc.gsv.txc.lga[PLC_OF_WDT]=plc_lga;
                    plc.gsv.txc.cmd=PLC_RCD_GSV;
                    plc.gsv.sv.out=ram_cfg.word[FLASH_AR_VSET];
                    plc.gsv.sv.mpp=ram_cfg.word[FLASH_AR_VMPP];
                    WRPLCNACK();

                    //Send response
                    i2c_rsen=FALSE;                         //Release I2C bus
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=(PLC_RTL_SND | sizeof(setv));
                    WRPLCNACK();
                }
                //Send fault stored in NVM
                else if (plc_cmd==PLC_RCD_GFT)
                {
                    //Set tx cfg, destination addr, cmd ID, & data
                    plc.gft.size=(sizeof(txcfg)+sizeof(fault));
                    plc.gft.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.gft.txc.reg=PLC_OF_RTC;
                    plc.gft.txc.cfg=PLC_CF_RTC;
                    plc.gft.txc.lga[PLC_OF_WDT]=plc_lga;
                    plc.gft.txc.cmd=PLC_RCD_GFT;
                    plc.gft.ft.time=ram_cfg.word[FLASH_AR_CSTS];
                    plc.gft.ft.sts=ram_cfg.byte[FLASH_AR_STS];
                    plc.gft.ft.stsc=ram_cfg.byte[FLASH_AR_STSC];
                    plc.gft.ft.r_sts=ram_cfg.byte[FLASH_AR_STATUS];
                    plc.gft.ft.r_pcon=ram_cfg.byte[FLASH_AR_PCON];
                    WRPLCNACK();

                    //Send response
                    i2c_rsen=FALSE;                         //Release I2C bus
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=(PLC_RTL_SND | sizeof(fault));
                    WRPLCNACK();
                }
                //Send FW/HW version numbers
                else if (plc_cmd==PLC_RCD_GRL)
                {
                    //Set tx cfg, destination addr, cmd ID, & data
                    plc.gcr.size=(sizeof(txcfg)+sizeof(hwfwr));
                    plc.gcr.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.gcr.txc.reg=PLC_OF_RTC;
                    plc.gcr.txc.cfg=PLC_CF_RTC;
                    plc.gcr.txc.lga[PLC_OF_WDT]=plc_lga;
                    plc.gcr.txc.cmd=PLC_RCD_GRL;
                    plc.gcr.rl.vmaj=VERMAJOR;
                    plc.gcr.rl.vmin=VERMINOR;
                    plc.gcr.rl.vbld=VERBUILD;
                    plc.gcr.rl.hwr=HWRELEASE;
                    WRPLCNACK();

                    //Send response
                    i2c_rsen=FALSE;                         //Release I2C bus
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=(PLC_RTL_SND | sizeof(hwfwr));
                    WRPLCNACK();
                }
                //Send charge algorithm values
                else if (plc_cmd==PLC_RCD_GCA)
                {
                    //Set tx cfg, destination addr, cmd ID, & data
                    plc.gft.size=(sizeof(txcfg)+sizeof(chga));
                    plc.gft.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.gft.txc.reg=PLC_OF_RTC;
                    plc.gft.txc.cfg=PLC_CF_RTC;
                    plc.gft.txc.lga[PLC_OF_WDT]=plc_lga;
                    plc.gft.txc.cmd=PLC_RCD_GCA;
                    plc.gca.ca.vnom=ram_cfg.word[FLASH_AR_VNOM];
                    plc.gca.ca.vblk=ram_cfg.word[FLASH_AR_VBLK];
                    plc.gca.ca.vabs=ram_cfg.word[FLASH_AR_VABS];
                    plc.gca.ca.vflt=ram_cfg.word[FLASH_AR_VFLT];
                    plc.gca.ca.tabs=ram_cfg.word[FLASH_AR_TABS];
                    WRPLCNACK();

                    //Send response
                    i2c_rsen=FALSE;                         //Release I2C bus
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=(PLC_RTL_SND | sizeof(chga));
                    WRPLCNACK();
                }
                //Send PLC Tx & Rx gain values
                else if (plc_cmd==PLC_RCD_GPG)
                {
                    //Set tx cfg, destination addr, cmd ID, & data
                    plc.ggn.size=(sizeof(txcfg)+sizeof(plcg));
                    plc.ggn.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.ggn.txc.reg=PLC_OF_RTC;
                    plc.ggn.txc.cfg=PLC_CF_RTC;
                    plc.ggn.txc.lga[PLC_OF_WDT]=plc_lga;
                    plc.ggn.txc.cmd=PLC_RCD_GPG;
                    plc.ggn.pg.txg=plc_txg;
                    plc.ggn.pg.rxg=plc_rxg;
                    WRPLCNACK();

                    //Send response
                    i2c_rsen=FALSE;                         //Release I2C bus
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=(PLC_RTL_SND | sizeof(plcg));
                    WRPLCNACK();
                }
                //Set charge algorithm values (reboot required to take effect)
                else if (plc_cmd==PLC_RCD_SCA)
                {
                    ram_cfg.word[FLASH_AR_VNOM]=((chga *) plc_rxd)->vnom;
                    ram_cfg.word[FLASH_AR_VBLK]=((chga *) plc_rxd)->vblk;
                    ram_cfg.word[FLASH_AR_VABS]=((chga *) plc_rxd)->vabs;
                    ram_cfg.word[FLASH_AR_VFLT]=((chga *) plc_rxd)->vflt;
                    ram_cfg.word[FLASH_AR_TABS]=((chga *) plc_rxd)->tabs;
                    flag_nvm=TRUE;                          //Update NVM
                }
                //Set PLC Tx & Rx gain values (reboot required to take effect)
                else if (plc_cmd==PLC_RCD_SPG)
                {
                    ram_cfg.byte[FLASH_AR_TXG]=((plcg *) plc_rxd)->txg;
                    ram_cfg.byte[FLASH_AR_RXG]=((plcg *) plc_rxd)->rxg;
                    flag_nvm=TRUE;                          //Update NVM
                }
                /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                 * Unrecognized command; clear flag_pir & exit loop. This 
                 * bypasses the charge controller command check (so no reset of
                 * keep alive timer).
                 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
                else
                {
                    flag_pir=FALSE;
                    break;
                }

                /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                 * Valid command processed: reset KPA if from charge controller,
                 * then clear flag_pir, and then exit loop.
                 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
                if (plc_lga==ram_cfg.byte[FLASH_AR_CHGADR])
                    tmr_kpa=S_LM_KPA_TIM;
                flag_pir=FALSE;
                break;
            }
            I2CHKBUS();

            /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
             * PLC SERVICE BLOCK FAULT DETECTION.
             * 
             * Any failure in I2C communication with the PLC leaves the object
             * flag_pir set TRUE. It's possible that a PLC failure could happen
             * in conjunction with another fault, so fault handling attempts to
             * capture the event while respecting the fault handling framework
             * and attempting to restore PLC communications:
             *  + If the PLC fault is unique, the NVM locations used to store 
             *    the MCU registers STATUS & PCON are re-purposed to store 
             *    additional data that may help determine a root cause.
             *  + The recovery timer, tmr_rest, is set to the recoverable fault
             *    recovery time to ensure plenty of time for the PLC controller
             *    to reboot. This is done irregardless of system STATE.
             *  + The PLC controller is rebooted to begin PLC fault recovery.
             *  + PLC fault recovery is completed by the PLC fault recovery
             *    block when the recovery timer expires.
             +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
            if (flag_pir)
            {
                SETSTSR(plc_sts);
                if (sts==plc_sts)       //Only TRUE if another status wasn't set
                {
                    ram_cfg.byte[FLASH_AR_STATUS]=plc_cmd;
                    ram_cfg.byte[FLASH_AR_PCON]=plc_irqs;
                }
                tmr_rest=S_TM_REST;
                RESETPLC();
            }
        }
        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * PLC FAULT RECOVERY.
         * 
         * Recovery is managed separately from the recoverable fault block to 
         * maintain communications in the event that a previous fault is active,
         * both recoverable & unrecoverable. In the event that a recoverable
         * fault is active (either PLC or preexisting), this block works in 
         * tandem with the recoverable fault block:
         *  + When a PLC fault occurs in the PLC service block the recoverable
         *    fault macro SETSTSR is used to set the fault status for NVM
         *    storage & start the recovery timer.
         *  + The converter is stopped.
         *  + The recoverable fault block resets system state & status & enables
         *    converter restart.
         * 
         * Regardless of fault condition, this block will execute because object
         * flag_pir is evaluated instead of sts. It's expected that the PLC 
         * fault detection logic at the bottom of the PLC service block will
         * have rebooted the PLC controller and set the fault recovery timer,
         * tmr_rest, to ensure adequate time for the PLC to boot (an artifact of
         * this is that the recovery time may be extended if a preexisting 
         * recoverable fault is active).
         * 
         * An attempt will be made to configure the PLC when the recovery timer
         * expires: if successful, object flag_pir is cleared and any remaining
         * fault recovery (if recoverable) is handled by the recoverable fault 
         * block (this may have already occurred depending on where in the RUN
         * loop the recovery timer expired). If configuration fails object 
         * flag_pxx is set to prevent repetitive attempts. The system state &
         * status are manually (e.g., not using the SETSTSF macro) set to their
         * respective fault & status codes to preserve the precipitating event
         * status data in NVM. In this situation a locally attached console 
         * could retrieve both the NVM event and real-time status.
         +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        else if (flag_pir ^ flag_pxx)
        {
            if (!(tmr_rest))
            {
                if (cfgPLC(FALSE)==STS_NOF) flag_pir=FALSE;
                else
                {
                    flag_pxx=TRUE;          //Prevent further recovery attempts
                    state=STE_FAULT;
                    sts=STS_PFR;
                }
            }
        }


        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * CONSOLE SERVICE.
         * 
         * IMPORTANT: do not place any code expected to execute on each RUN 
         * loop iteration beyond this point. The console service handler should 
         * always be placed at bottom of RUN loop.
         * 
         * This block services the user console accessed via the RS232 micro-USB
         * service port. All commands are two bytes in length with up to two,
         * optional, parameters. Refer to the COMMAND CODES section of pvc.h for
         * command strings, explanation, and values.
         * 
         * The command line accepts a 2-character command string followed by
         * up to two integer parameters. The cmdb object points to the NULL
         * terminated command string and the objects cmdp1 & cmdp2 contain the
         * integer parameters. Bit flags flag_cv1 & flag_cv2 are TRUE to 
         * indicate that the respective integer contains a value from the
         * command line.
         * 
         * The command structure is optimized to minimize processing time & code 
         * size balanced with not being too cryptic for the user. That said,
         * character echo is off and the commands provide no response on 
         * validity or success. No bounds checking is done with the exception of
         * the NVM byte address.
         * 
         * Care must be taken when entering NVM data to ensure the correct byte
         * is being updated. Further, 16-bit values are stored in little endian
         * format and the user must calculate the decimal equivalents of the
         * high/low hexadecimal values.
         * 
         * IMPORTANT: all set commands and NVM updates MUST be done with the 
         * converter stopped. NVM updates require an MCU reset to take effect.
         *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

        //Remote commands take precedence
        if (cmdp0==CMDNONCMD)
        {
            //Retrieve command buffer & size, skip processing if too small.
            if (usartStatRx() != USARTSTSRXRDY) continue;
            cmdb=(char *) usartGetRx();             //Get command buffer
            if ((cmdb_sz=(uint8_t) usartGetSz()) < CMDSIZSTR)   //Too small
            {
                usartSetRx(ENABLE);
                continue;
            }

            //Parse buffer: command integer & optional integer parameters
            for (counter=0, flag_cv0=FALSE, flag_cv1=FALSE, flag_cv2=FALSE; \
                    (counter < cmdb_sz); ++counter)
            {
                if (*(cmdb+counter) > S_CH_SPACE)
                {
                    if (!(flag_cv0))                //Found command
                    {
                        flag_cv0=TRUE;
                        cmdp0=(uint8_t) atoi(cmdb+counter);
                    }
                    else if (!(flag_cv1))           //Found first parameter
                    {
                        flag_cv1=TRUE;
                        cmdp1=(uint8_t) atoi(cmdb+counter);
                    }
                    else if (!(flag_cv2))           //Found second parameter
                    {
                        flag_cv2=TRUE;
                        cmdp2=(uint16_t) atoi(cmdb+counter);
                    }
                    for (; (*(cmdb+counter) > S_CH_SPACE); ++counter);
                }
            }
        }

        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * PROCESS COMMANDS.
         * 
         * These don't take parameters.
         +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        //Get firmware release
        if (cmdp0==CMDGETREL)
        {
            usart_nl=FALSE;
            usart_wr_ui16(VERMAJOR);
            usart_wr_chr(S_CH_SPACE);
            usart_wr_ui16(VERMINOR);
            usart_wr_chr(S_CH_SPACE);
            usart_wr_ui16(VERBUILD);
            usart_wr_chr(S_CH_SPACE);
            usart_nl=TRUE;
            usart_wr_ui16(HWRELEASE);
        }
        //Get PLC MAC address
        else if (cmdp0==CMDGETPAD)
        {
            i2c_rsen=TRUE;                              //Hold I2C bus
            plc.wr.size=PLC_SZ_RRG;                     //Register address only
            plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);           //I2C address
            plc.wr.reg=PLC_OF_RPA;                      //Phy. address register
            if (i2c_m_Wd((unsigned char *) &plc) == I2C_ACK)
            {
                i2c_rsen=FALSE;
                plc.rd.size=PLC_SZ_PAD;                 //Read full address
                plc.rd.i2ca=I2CSETRD(PLC_AD_I2C);
                if (i2c_m_Rd((unsigned char *) &plc) == I2C_ACK)
                {
                    for (counter=PLC_OF_RDT; (counter < (PLC_SZ_PAD)); 
                            ++counter)
                    {
                        //Use cmdp1 as scratch variable
                        cmdp1=(((plc.rd.data[counter]>>S_BP_U8_H)&S_MK_U8_L)\
                                +S_CH_ZERO);
                        if (cmdp1>S_CH_NINE) cmdp1+=S_QY_9TOA;
                        putch((char) cmdp1);
                        cmdp1=((plc.rd.data[counter]&S_MK_U8_L)+S_CH_ZERO);
                        if (cmdp1>S_CH_NINE) cmdp1+=S_QY_9TOA;
                        putch((char) cmdp1);
                    }
                    putch(S_CH_NEWLINE);
                }
            }
            I2CHKBUS();
        }
        //Restore NVM defaults
        else if (cmdp0==CMDXNVDFL)
        {
            writeFlashRow((uint16_t) &nvm_cfg, NULL);
        }
        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * Reset system: first do clean stop if running. Disable watchdog to 
         * prevent erroneous reboot while resetting the PLC controller, and 
         * clear PCON bits to prevent error trap at MCU boot.
         +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        else if (cmdp0==CMDXRTMCU)
        {
            if (state & STE_RUN) ConverterOFF();
            S_AL_WDT_ENABLE=FALSE;
            RESETPLC();
            S_AL_REG_PCON=S_LM_PCON_BT;
            RESET();
        }
        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * All remaining commands require at least first parameter: test for 
         * this before parsing command.
         +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        else if (flag_cv1)
        {
            //Get over-current count
            if (cmdp0==CMDGETCNT)
            {
                if (cmdp1) usart_wr_ui16(occp2);
                else usart_wr_ui16(occp1);
            }
            //Get [measured] duty cycle
            else if (cmdp0==CMDGETDCY)
            {
                if (cmdp1) usart_wr_ui16(dcp2);
                else usart_wr_ui16(dcp1);
            }
            //Get system status
            else if (cmdp0==CMDGETSTS)
            {
                if (cmdp1) usart_wr_ui16(sts);
                else usart_wr_ui16(state);
            }
            //Get temps
            else if (cmdp0==CMDGETEMP)
            {
                if (cmdp1) usart_wr_ui16(tp2.vmv);
                else usart_wr_ui16(tp1.vmv);
            }
            //Get volts
            else if (cmdp0==CMDGETVOL)
            {
                if (cmdp1) usart_wr_ui16(outv.adc);
                else usart_wr_ui16(inv.adc);
            }
            //Read NVM byte
            else if (cmdp0==CMDGETBYT)
            {
                if (cmdp1 < S_SZ_FLASHR)
                    usart_wr_ui16((uint16_t) ram_cfg.byte[cmdp1]);
            }
            //Read NVM word
            else if (cmdp0==CMDGETWOR)
            {
                if (cmdp1 < S_SZ_FLASHW)
                    usart_wr_ui16(ram_cfg.word[cmdp1]);
            }
            //Set NVM byte
            else if (cmdp0==CMDSETBYT)
            {
                if ((flag_cv2) && (cmdp1 < S_SZ_FLASHC) && \
                        (ram_cfg.byte[cmdp1] != (uint8_t) cmdp2))
                {
                    ram_cfg.byte[cmdp1]=(uint8_t) cmdp2;
                    flag_nvm=TRUE;
                }
            }
            //Set converter on / off
            else if (cmdp0==CMDSETCON)
            {
                if (cmdp1) ConverterON();
                else ConverterOFF();
            }
            //Set NVM word
            else if (cmdp0==CMDSETWOR)
            {
                if ((flag_cv2) && (cmdp1 < (S_SZ_FLASHW-1)) && \
                        (ram_cfg.word[cmdp1] != cmdp2))
                {
                    ram_cfg.word[cmdp1]=cmdp2;
                    flag_nvm=TRUE;
                }
            }
            else if (cmdp0==CMDSETSSR)
            {
                if (cmdp1) SSRON();
                else SSROFF();
            }
            //Ping PLC
            else if (cmdp0==CMDXPGPLC)
            {
                //Don't allow broadcast or another request if prior still active
                if ((cmdp1!=PLC_AD_BRC) && !(plc_rqt)) pingPLC(cmdp1);
            }
        }

        cmdp0=CMDNONCMD;
        usartSetRx(ENABLE);                 //Look for new user input
    }
} //End function main


/* Function ConverterON. Starts converter; if already started it has no effect.
 * No pre-start diagnostic checks are done since this is expected to be
 * handled by the RUN loop and reflected in the machine state object.
 * 
 * Soft start is employed. The duty cycle is started at S_DC_MIN and incremented 
 * to the DC stored in NVM. If open loop this is the DC the unit will operate 
 * at; if closed the control loop will regulate DC up to the maximum set by 
 * NVM.
 * 
 * If the IDLE state is set at entry this function will set the state to STOP
 * and expect the RUN loop logic in the charge algorithm block to resume
 * control of the converter. The IDLE state is only used in charger mode.
 * 
 * Returns:     none.
 *
 * Parameters:  none
 *
*/
void ConverterON()
{
    if (state & STE_BRS)
    {
        //Reset RUN loop control objects
        occp1=0;
        occp2=0;
        flag_ocp1=FALSE;
        flag_ocp2=FALSE;
        S_AL_OCIF_P1=FALSE;                     //Ensure P1 over-current clear
        S_AL_OCIF_P2=FALSE;                     //Ensure P2 over-current clear
        S_AL_OCIE_P1=TRUE;                      //Enable P1 over-current count
        S_AL_OCIE_P2=TRUE;                      //Enable P2 over-current count

        S_AL_DC_P1=S_DC_MIN;                    //Start @ min. DC
        S_AL_DC_P2=S_DC_MIN;                    //Start @ min. DC
        PWMLD=S_EN_PWMLD;                       //Load DC simultaneously
        while (PWMLD);                          //Wait for load to complete

        S_AL_PRGGO_P1=TRUE;                     //Start P1 slope comp
        S_AL_PRGGO_P2=TRUE;                     //Start P2 slope comp
        S_AL_COGAS_P1=FALSE;                    //Enable P1 COG
        S_AL_COGAS_P2=FALSE;                    //Enable P2 COG

        SSRON();                                //Enable output
        chgDutyCycle(S_DC_MIN, ram_cfg.byte[FLASH_AR_DC]);   //Ramp DC

        state=STE_RUN;                          //Set machine state to RUN
    }
    else if (state & STE_IDLE) state=STE_STOP;

    return;
} //End function ConverterON


/* Function ConverterOFF.  Stops converter:
 *  + All converter phases are simultaneously stopped by setting the duty cycle
 *    to zero.
 *  + The COGs & PRGs are then stopped.
 *  + The machine state is set to STOP unless already in a FAULT state.
 * 
 * When in charger mode the IDLE machine state is used to keep the converter
 * off during otherwise normal operation. It is assumed that the user has 
 * commanded it off. Without this (e.g., if set to STOP state), the charge 
 * algorithm control block will automatically restart the converter when 
 * conditions are met.
 * 
 * IMPORTANT: the IDLE machine state is not persistent across boots. If the 
 * charger must remain STOPPED across boots (e.g., for service), set the
 * charger to MANUAL control since this always boots to the STOPPED state.
 * 
 * Returns:     none. Machine state is set to STOP, IDLE, or left unchanged 
 *              based on operating conditions.
 *
 * Parameters:  none
 *
*/
void ConverterOFF()
{
    S_AL_DC_P1=S_DC_OFF;                        //Turn off with no DC
    S_AL_DC_P2=S_DC_OFF;                        //Turn off with no DC
    PWMLD=S_EN_PWMLD;                           //Load DC simultaneously
    while (PWMLD);                              //Wait for load to complete

    SSROFF();                                   //Disable output
    S_AL_COGAS_P1=TRUE;                         //Stop P1
    S_AL_COGAS_P2=TRUE;                         //Stop P2
    S_AL_PRGGO_P1=FALSE;                        //Stop P1 slope comp
    S_AL_PRGGO_P2=FALSE;                        //Stop P2 slope comp

    if (!(state & STE_RCRFT)) state=STE_STOP;   //Leave set if a fault state

    //Override to keep converter off when in charger mode
    if ((ram_cfg.byte[FLASH_AR_CONTRL] & S_BL_ECHGA) && (state==STE_STOP))
        state=STE_IDLE;

    return;
} //End function ConverterOFF


/* Function chgDutyCycle. Moves the duty cycle from the start to final value 
 * incrementally for 'soft' transitions. The constant S_TM_DCDELAY sets the 
 * delay, in uS, between DC increments.
 * 
 * Returns:     none
 *
 * Parameters:  start=16-bit starting DC value. If zero (0) then P1's current 
 *              DC is used.
 *              final=16-bit final DC value
 *
*/
void chgDutyCycle(uint16_t start, uint16_t final)
{
    if (!(start)) start=S_AL_DC_P1;             //Start with current DC
    while (start != final)
    {
        if (start < final) ++start; else --start;
        S_AL_DC_P1=start;
        S_AL_DC_P2=start;
        PWMLD=S_EN_PWMLD;
        __delay_us(S_TM_DCDELAY);
    }

    return;
} //End function chgDutyCycle


/* Function cfgPLC. Configures the PLC controller and optionally checks for a
 * duplicate logical address. The duplicate check is optional to allow for it
 * to be performed at boot before the watchdog is active. When reconfiguring
 * during operation (e.g, to recover from a fault), the duplicate check can
 * be skipped so the watchdog doesn't have to be disabled.
 * 
 * Returns:     uint8_t indicating status:
 *                  STS_NOF=no error,
 *                  STS_PCG=PLC fail to configure controller (NACK).
 *
 * Parameters:  chkdup is set to a boolean to indicate if the duplicate check
 *                  should be performed:
 *                  TRUE=perfrom duplicate check,
 *                  FALSE=skip duplicate check.
 *
*/
uint8_t cfgPLC(uint8_t chkdup)
{
    uint8_t rc=STS_PCG;

    while (TRUE)
    {
        i2c_rsen=TRUE;                                  //Set I2C bus busy

        //Set node logical address
        plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
        plc.wr.size=PLC_SZ_RCF;
        plc.wr.reg=PLC_OF_RLL;
        if (chkdup==TRUE) plc.wr.data[PLC_OF_WDT]=PLC_AD_BRC;
        else plc.wr.data[PLC_OF_WDT]=ram_cfg.byte[FLASH_AR_PLCADR];
        WRPLCNACK();

        //Set the Tx config
        plc.wr.reg=PLC_OF_RTC;
        plc.wr.data[PLC_OF_WDT]=PLC_CF_RTC;
        WRPLCNACK();

        //Set the threshold noise, modem, Tx & Rx gain registers
        //Load BIU noise threshold, Tx & Rx gain from NVM
        plc.wr.reg=PLC_OF_RTN;
        plc.wr.data[PLC_OF_WDT]=ram_cfg.byte[FLASH_AR_BIU];
        plc.wr.data[PLC_OF_WDT+1]=PLC_CF_RMC;
        plc.wr.data[PLC_OF_WDT+2]=plc_txg;
        plc.wr.data[PLC_OF_WDT+3]=plc_rxg;
        plc.wr.size=5;                                  //Count data + register
        WRPLCNACK();

        //Configure interrupts
        SETINTPLC();
        WRPLCNACK();

        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * Set the PLC mode last: there should be no concern of receiving a 
         * packet between this instant & the duplicate check because an 
         * 'illegal' (for this implementation) is assigned to the PLC.
         * 
         * The I2C bus is held active for the duplicate check, and if it is 
         * skipped (because of invalid assignment) the I2CCHKBUS macro will
         * ensure it is released.
         * 
         * The return code, rc, is cleared if the PLC mode set is successful.
         * The duplicate address check block assigns its own codes for failures.
         +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
        plc.wr.size=PLC_SZ_RCF;
        plc.wr.reg=PLC_OF_RMD;
        plc.wr.data[PLC_OF_WDT]=PLC_CF_RMD;
        WRPLCNACK();
        rc=STS_NOF;

        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * Check for duplicate address: only performed if a legitimate address
         * is assigned & object dupchk is TRUE.
         * 
         * A ping is sent using the converter's assigned address from NVM. The 
         * unit waits for the response timer (set by ping) to expire or the PLC
         * to raise the interrupt pin. The PLC is serviced and the response 
         * validated for a duplicate address.
         * 
         * The duplicate block expects the I2C bus to be active since the PLC's
         * clear time requirement (between previous config sends) will not have
         * been met. Failures set the appropriate fault code. If the duplicate
         * check is clear the NVM stored address is assigned to the PLC
         +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        while ((ram_cfg.byte[FLASH_AR_PLCADR]!=PLC_AD_BRC) && (chkdup==TRUE))
        {
            if ((rc=pingPLC(ram_cfg.byte[FLASH_AR_PLCADR]))!=STS_NOF) break;
            while ((plc_rqt) && !(PIN_ID_PIR));
            plc_rqt=0;                              //Ensure clear
            if (PIN_ID_PIR)
            {
                if (svcPLC())
                {
                    rc=STS_PFC;
                    break;
                }
                /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                 * There's a low probability that a packet could be received 
                 * that is not a response to the ping. These are ignored, but
                 * the 'else' delay is included just in case the PLC service was
                 * for anything other than new data received (PLC_RIS_NDA)
                 * since, in this situation, the service function will release
                 * the I2C bus and the PLC controller requires a minimum I2C
                 * bus idle time.
                 * 
                 * Note that this delay isn't required between the ping & PLC
                 * service since the shortest possible round-trip time is on the
                 * order of mS.
                 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
                if ((plc_cmd==PLC_RCD_GRV) && (plc_lga==plc_rqa))
                {
                    rc=STS_PDA;
                    break;
                }
                else __delay_us(PLC_TM_BUF);
            }
            rc=STS_PFA;
            i2c_rsen=FALSE;                         //Release I2C bus
            //Set node logical address
            plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
            plc.wr.size=PLC_SZ_RCF;
            plc.wr.reg=PLC_OF_RLL;
            plc.wr.data[PLC_OF_WDT]=ram_cfg.byte[FLASH_AR_PLCADR];
            if (WRPLC()==I2C_ACK) rc=STS_NOF;
            break;
        }
        break;
    }

    I2CHKBUS();
    return(rc);
} //End function cfgPLC


/* Function pingPLC. Initiates a 'ping' to another PLC logical node. The ping
 * is accomplished using the 'get remote node version' command which the remote
 * PLC responds to without host charge controller intervention. Upon successful
 * send of the command the PLC request timer (plc_rqt) is set.
 * 
 * To save execution time object la is re-purposed to hold the the return code.
 * 
 * Returns:     uint8_t indicating status:
 *                  STS_NOF=no error,
 *                  STS_PFP=PLC failed to send ping
 *
 * Parameters:  la=logical address of remote PLC node to ping.
 *
*/
uint8_t pingPLC(uint8_t la)
{
    while (TRUE)
    {
        //Set Tx config, PLC address, remote command
        i2c_rsen=TRUE;
        plc.t.size=sizeof(txcfg);
        plc.t.i2ca=I2CSETWR(PLC_AD_I2C);
        plc.t.txc.reg=PLC_OF_RTC;
        plc.t.txc.cfg=PLC_CF_RTC;
        plc.t.txc.lga[PLC_OF_WDT]=la;
        plc.t.txc.cmd=PLC_RCD_GRV;
        plc_rqa=(uint8_t) la;                       //To validate against resp.
        la=STS_PFP;
        WRPLCNACK();

        //Send request
        i2c_rsen=FALSE;
        plc.wr.size=PLC_SZ_RCF;
        plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
        plc.wr.reg=PLC_OF_RTL;
        plc.wr.data[PLC_OF_WDT]=PLC_RTL_SND;
        WRPLCNACK();

        plc_rqt=PLC_TM_RQT;
        la=STS_NOF;
        break;
    }

    I2CHKBUS();
    return(la);
} //End function pingPLC


/* Function svcPLC. Services the PLC: retrieves PLC status & data, stores in
 * respective module objects, and clears the PLC data buffers & status so it
 * is ready for next service. The I2C bus is held active for the PLC if further 
 * communication is required based on received data. It is the responsibility of
 * the caller to release the I2C bus.
 * 
 * Returns:     __bit indicating status:
 *                  FALSE=no error,
 *                  TRUE=error, object plc_sts contains status code representing
 *                  locus of failure.
 *
 * Parameters:  none.
 *
*/
__bit svcPLC(void)
{
    flag_pir=TRUE;                              //Now servicing PLC
    plc_sts=STS_PIS;                            //Fault status code
    i2c_rsen=TRUE;                              //Hold I2C bus
    plc_cmd=PLC_RCD_INV;                        //No valid command
    plc_irqs=PLC_RIS_CLR;                       //Invalid status
    plc.wr.size=PLC_SZ_RRG;                     //Register address only
    plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);           //I2C address
    plc.wr.reg=PLC_OF_RIS;                      //IRQ status register
    while (TRUE)
    {
        //Set register pointer to IRQ status register
        WRPLCNACK();

        //Read IRQ status register
        plc.rd.i2ca=I2CSETRD(PLC_AD_I2C);
        RDPLCNACK();
        plc_irqs=(uint8_t) plc.rd.data[PLC_OF_RDT];
        plc_sts=STS_PIT;                        //Fault status code
        SETINTPLC();                            //Clear PLC interrupts
        WRPLCNACK();

        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * RETRIEVE RECEIVED DATA.
         *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        if (plc_irqs & PLC_RIS_NDA)
        {
            plc_sts=STS_PRD;                    //Fault status code
            //Set PLC register to RX_MESSAGE_INFO
            plc.wr.size=PLC_SZ_RRG;
            plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
            plc.wr.reg=PLC_OF_RRI;
            WRPLCNACK();

            //Read PLC register RX_MESSAGE_INFO & get data buffer size
            plc.rd.i2ca=I2CSETRD(PLC_AD_I2C);
            RDPLCNACK();
            plc_rxi=(uint8_t) plc.rd.data[PLC_OF_RDT];  //Rx msg info
            plc_bufs=(plc_rxi & PLC_RRI_RXS);           //Rx buffer size

            //Retrieve source address, command, and data
            plc.wr.size=PLC_SZ_RRG;
            plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
            plc.wr.reg=PLC_OF_RRS;
            WRPLCNACK();
            plc.rdc.size=(plc_bufs+(PLC_SZ_CMD+PLC_SZ_PAD));
            plc.rdc.i2ca=I2CSETRD(PLC_AD_I2C);
            RDPLCNACK();
            plc_lga=plc.rdc.lga[PLC_OF_LAD];        //Logical address
            plc_cmd=plc.rdc.cmd;                    //Command ID
            //Save copy of Rx buffer to free PLC object.
            memcpy((void *) plc_rxd, (void *) plc.rdc.data, plc_bufs);
        }
        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * PROCESSING FOR THINGS OTHER THAN RX DATA MIGHT GO HERE.
         * 
         * In this version/build they are not required: BUI, no PLC ACK,
         * no response received, Rx packet dropped, and Tx data sent.
         *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * CLEAR NEW_RX_MSG REGISTER IN PLC.
         * 
         * This is done once received data & supporting metadata are
         * saved off. This enables receipt of new messages by clearing 
         * NEW_RX_MSG in the RX_Message_INFO register. This also clears 
         * Status_RX_Packet_Dropped & Status_RX_Data_Available in the
         * INT_Status register (clearing interrupts does not clear
         * these).
         * 
         * This also releases the PLC receive buffer & overwrites
         * object plc's data. Only release the I2C bus if no new data 
         * was received.
         * 
         * For simplicity, the remainder of the PLC service block 
         * executes even if there is no new data to service. This 
         * ensures that PLC service is completed properly but at the
         * expense of wasted execution time. The command block tests 
         * will all be FALSE because object plc_cmd is set to invalid 
         * command (PLC_RCD_INV) at the start of the PLC service block 
         * and will not have been assigned to a valid command.
         +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        plc_sts=STS_PCN;                            //Fault status code
        if (!(plc_irqs & PLC_RIS_NDA)) i2c_rsen=FALSE;
        plc.wr.size=PLC_SZ_RCF;
        plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
        plc.wr.reg=PLC_OF_RRI;
        plc.wr.data[PLC_OF_WDT]=PLC_RRI_CLR;
        WRPLCNACK();

        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * Service completed without error but I2C bus is held active for PLC
         * if further communication is required based on received data. It is
         * the responsibility of the caller to release the I2C bus.
         *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        return(FALSE);
    }

    return(TRUE);                                   //Fault, plc_sts has code
} //End function svcPLC


/*Function InterruptExec.  MCU interrupt handler.  Manages all enabled
 * interrupts which should be ordered according to priority high-low:
 *  + Over-current event capture
 *  + System timer
 *  + I2C
 *  + USART
 *
 * Returns:     none
 *
 * Parameters:  none
 *
*/
void __interrupt() InterruptExec(void)
{
    static uint8_t  msc=0,                          //System mS counter
                    tmr_cc=S_TM_CCTMR;              //Cycle counter update timer

    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * OVER-CURRENT EVENT CAPTURE. The negative transition of the over-current 
     * comparator generates an interrupt. This block captures the event, sets 
     * the respective bit object flag_oc TRUE, and disables further over-
     * current interrupts. The RUN loop is expected to monitor the flag and 
     * take appropriate action (including re-enabling the interrupt).
     * 
     * This approach avoids the problems associated with ISR-based over-current
     * schemes that attempt to capture each event: a sustained over-current 
     * generates an interrupt storm that results in the PIC resetting. Of 
     * course, this scheme expects that the hardware can endure sustained over-
     * current events without damage.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    if (S_AL_OCIE_P1 & S_AL_OCIF_P1)
    {
        flag_ocp1=TRUE;
        S_AL_OCIE_P1=FALSE;
        S_AL_OCIF_P1=FALSE;
    }
    if (S_AL_OCIE_P2 & S_AL_OCIF_P2)
    {
        flag_ocp2=TRUE;
        S_AL_OCIE_P2=FALSE;
        S_AL_OCIF_P2=FALSE;
    }

    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * SYSTEM TIMER. Non-precision timed background service. ~10 mS frequency.
     * Maintains various timing objects:
     *  + i2c_tor - timer for I2C slave acknowledge (ACK).
     *  + tmr_temp - timer to measure converter temps.
     *  + tmr_cc - one-time countdown timer after boot controlling the cycle
     *    counter increment & recording in NVM.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    if (S_AL_SYST_IE & S_AL_SYST_IF)
    {
        if (i2c_tor) --i2c_tor;                     //I2C timeout timer
        if (plc_rqt) --plc_rqt;                     //PLC request timer
        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         *                          ONE SECOND ELAPSED
         +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        if (++msc == S_QY_SYST_1S)
        {
            msc=0;                                  //Reset mS counter
            if (tmr_temp) --tmr_temp;               //Temp timer
            if (tmr_rest) --tmr_rest;               //Recoverable fault timer
            if (tmr_kpa) --tmr_kpa;                 //Keep alive timer
            if (tmr_chga) --tmr_chga;               //Charge sampling timer
            if (tmr_abs)                            //Absorption timer
            {
                if (--tmr_abs==0) flag_abs=FALSE;
            }
            if (tmr_cc)                             //Cycle counter timer
            {
                if (--tmr_cc==0)
                {
                    ++ram_cfg.word[FLASH_AR_CYCC];
                    flag_nvm=TRUE;
                    flag_run=TRUE;
                }
            }
        }

        S_AL_SYST_IF=FALSE;
    }

    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * I2C READ HANDLER. Reads slave data in background.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    if (SSP1IE & SSP1IF) i2c_m_isrRd();

    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * USART HANDLER. Service RS232 console.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    usartISR();

    return;
} //End function InterruptExec


/*Function Initialize. Called immediately after boot by function main to 
 * configure MCU, port, and on-board peripherals. Initialization of external
 * components should be handled by function main and not in this function.
 *  + Sets boot fault trap & tests,
 *  + Sets system clock speed & watchdog timer,
 *  + Validates NVM configuration row & loads to RAM,
 *  + Tests for MCU faults,
 *  + Initializes module-wide control objects,
 *  + Configures pins, peripherals, interrupts,
 *  + Starts system timer,
 *  + Enables interrupts.
 *
 * Returns:     none
 *
 * Parameters:  none
 *
*/
void Initialize(void)
{
    di();                               //No interrupts during initialization    

    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * INITIALIZE STATUS CODE.
     * 
     * Object sts, the system status code, is set to no fault. If clear at the
     * end of this function, object state, the machine state, is set to BOOT.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    sts=STS_NOF;

    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * INITIALIZE SYSTEM CLOCK & WATCHDOG.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    OSCCON=CFG_OSC_C;                   //Configure system clock
    while (!(HFIOFL));                  //Wait for oscillator to stabilize
    WDTCON=CFG_WDT_CON;                 //Watchdog


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * BOOT & NVM PROTECTION AND MANAGEMENT.
     *
     * Recovering from MCU/firmware failures (e.g. watchdog, stack OF/UF, etc.)
     * and NVM corruption is critical for safety and field reliability. While
     * neither should happen unless there's a true failure, statistically it's 
     * possible when operating in environmental extremes or the more likely
     * existence of a firmware bug.
     * 
     * The goal is to maintain safe operation, keep the unit intact, and record
     * the failure state (in that order).
     * 
     * DATA MANAGEMENT
     * A single row (32 bytes) of NVM data is used for the storage of unit
     * configuration. Object nvm_cfg resides in flash (NVM) and is initialized 
     * with default values. During normal operation object ram_cfg contains a 
     * copy of nvm_cfg and is used to update nvm_cfg. It must also be 
     * initialized with the same default values as nvm_cfg.
     * 
     * DATA VALIDATION
     * NVM data is validated with a simple 16-bit checksum stored in the last
     * two bytes of nvm_cfg.
     * 
     * BOOT PROCESS
     * When function main first gains control from the root code it assigns 
     * objects reg_pcon & reg_status the values of the respective registers so
     * their initial state is preserved.
     * 
     * The NVM configuration checksum is checked first: the NVM row is read to
     * calculate the checksum and compared to the stored value. If valid, 
     * configuration data is loaded to object ram_cfg. If invalid, an NVM fault
     * status is set and the initialization values in ram_cfg are used. Macro
     * SETSTSF will set the fault status code in ram_cfg for subsequent writing
     * to NVM (if possible).
     * 
     * Since the MCU's architecture requires the entire NVM row to be written
     * it's critical that ram_cfg contains valid data. This is why it's not
     * updated with nvm_cfg when checksum's don't match. By writing the status
     * code it also ensures that if NVM was truly corrupt it will now contain
     * valid data (provided there's not an MCU flash failure) and there's a 
     * chance of the unit returning to operation when power cycled.
     * 
     * The MCU boot status is then evaluated using the stored values in objects
     * reg_pcon & reg_status. Tests include reboot due to watchdog, stack OF/UF,
     * and possibly other conditions. Any tests that evaluate TRUE will set an
     * MCU fault and attempt to record the event in NVM (including the values
     * in reg_pcon & reg_status. In the event that an NVM fault is already set
     * the MCU fault will not be recorded but the values in reg_pcon & 
     * reg_status will be. In a forensic review of the NVM fault data this would
     * render as an NVM fault along with non-zero values for the PCON & STATUS 
     * registers.
     * 
     * When an MCU fault is recorded, the register values remain in NVM until 
     * another MCU trap occurs, the NVM status code is cleared, or defaults are
     * restored. Register values are only recorded when an MCU status is set so 
     * these will be 0x00 for any other code (when other fault status codes are
     * set they do not clear these values).
     * 
     * The remainder of the initialization function continues unaltered and
     * when complete the main function enters the RUN loop. The converter
     * should not start if any status code other than BOOT is set. A code block 
     * in the main function loop is responsible for storing error codes 
     * resulting from the boot process, or otherwise, in NVM.
     * 
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    //NVM check & load
    if (nvmChksum((unsigned char *) &nvm_cfg) != (unsigned int) \
            nvm_cfg.word[S_AR_FLCS_W])
    {
        SETSTSF(STS_NVM);
    }
    else memcpy((void *) &ram_cfg, (void *) &nvm_cfg, S_SZ_FLASHR);

    //MCU boot tests
    if (((reg_pcon & S_LM_PCON_EX) != S_LM_PCON_BT) || \
            ((reg_status & S_LM_STAT_EX) != S_LM_STAT_BT))
    {
        ram_cfg.byte[FLASH_AR_STATUS]=reg_status;
        ram_cfg.byte[FLASH_AR_PCON]=reg_pcon;
        SETSTSF(STS_MCU);
    }


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * INITIALIZE CONTROL DATA:
     * + usart_nl is set TRUE to add new line character to console output.
     * + flag pir is cleared to ensure service of PLC interrupt.
     * + flag pxx is cleared to allow PLC recovery upon fault.
     * + flag nvm is cleared to erroneous NVM writes.
     * + flag nvf is cleared to erroneous NVM writes.
     * + flag tmp is cleared to prevent erroneous power reduction at boot.
     * + flag_ocle is cleared to prevent erroneous fault indication at boot.
     * + flag_run is cleared to prevent converter operation prior to the boot
     *   cycle counter expiring.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    usart_nl=TRUE;
    flag_pir=FALSE;
    flag_pxx=FALSE;
    flag_nvm=FALSE;
    flag_nvf=FALSE;
    flag_tmp=FALSE;
    flag_ocle=FALSE;
    flag_run=FALSE;


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * ENFORCE CONTROL BIT PRECEDENCE.
     *  + Controlled,
     *  + Charge algorithm,
     *  + Autostart,
     *  + Manual control..
     * 
     * The ram_cfg object is tested and assigned only the proper control bit
     * in precedence to eliminate the need for additional logic test overhead
     * in the RUN loop (multiple bits could be erroneously set by an operator
     * directly modifying NVM).
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    if (ram_cfg.byte[FLASH_AR_CONTRL] & S_BL_ECTRL)
        ram_cfg.byte[FLASH_AR_CONTRL]=S_BL_ECTRL;
    else if (ram_cfg.byte[FLASH_AR_CONTRL] & S_BL_ECHGA)
        ram_cfg.byte[FLASH_AR_CONTRL]=S_BL_ECHGA;
    else if (ram_cfg.byte[FLASH_AR_CONTRL] & S_BL_EAUTO)
        ram_cfg.byte[FLASH_AR_CONTRL]=S_BL_EAUTO;
    else ram_cfg.byte[FLASH_AR_CONTRL]=S_BL_EMANU;


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * STANDALONE CHARGE ALGORITHM.
     * 
     * Test the charging values if enabled & load to their respective operating
     * objects. Rules:
     *  + Vnom >= S_LM_MIN_VNOM && < Vblk,
     *  + Vblk > Vnom && <= S_LM_MAX_VBLK,
     *  + Vabs <= Vblk && > Vflt,
     *  + Vflt < Vabs || < Vblk if Vabs not set,
     *  + Tabs >= S_LM_MIN_TABS seconds.
     * 
     * Values in object ram_cfg are not used directly: they are evaluated 
     * against the rule set and, if valid, are loaded to a set of discrete 
     * operating objects (all starting with 'cga_'). This is done to preserve
     * the configured values so that the operator can review them via NVM
     * read or remote console queries to correct if invalid.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    if (ram_cfg.byte[FLASH_AR_CONTRL] & S_BL_ECHGA)
    {
        uint8_t s=STS_CGA;

        //Start waterfall by testing Vnom
        while ((ram_cfg.word[FLASH_AR_VNOM] >= S_LM_MIN_VNOM) &&\
                (ram_cfg.word[FLASH_AR_VNOM] < ram_cfg.word[FLASH_AR_VBLK]))
        {
            //Assign valid Vnom value
            cga_vnom.w=ram_cfg.word[FLASH_AR_VNOM];

            //Vblk tests: assign if valid
            if (ram_cfg.word[FLASH_AR_VBLK] > S_LM_MAX_VBLK) break;
            cga_vblk.w=ram_cfg.word[FLASH_AR_VBLK];

            //Vabs tests if set (both voltage & time must be non-zero)
            if ((ram_cfg.word[FLASH_AR_VABS]) && (ram_cfg.word[FLASH_AR_TABS]))
            {
                if (ram_cfg.word[FLASH_AR_VABS] > ram_cfg.word[FLASH_AR_VBLK])\
                        break;
                //Test Vabs against Vflt if set
                if ((ram_cfg.word[FLASH_AR_VFLT])&&(ram_cfg.word[FLASH_AR_VABS]\
                        <= ram_cfg.word[FLASH_AR_VFLT])) break;
                //Test for minimum absorption time
                if (ram_cfg.word[FLASH_AR_TABS] < S_LM_MIN_TABS) break;
                //Assign valid Vabs & Tabs values
                cga_vabs.w=ram_cfg.word[FLASH_AR_VABS];
                cga_tabs=ram_cfg.word[FLASH_AR_TABS];
            }

            //Vflt tests
            if (ram_cfg.word[FLASH_AR_VFLT])
            {
                if (cga_vabs.w)
                {
                    if (ram_cfg.word[FLASH_AR_VFLT] >= cga_vabs.w) break;
                }
                else if (ram_cfg.word[FLASH_AR_VFLT] >= cga_vblk.w) break;
                //Assign valid Vflt value
                cga_vflt.w=ram_cfg.word[FLASH_AR_VFLT];
            }

            s=STS_NOF;
            break;
        }

        if (s == STS_CGA)
        {
            SETSTSF(s);
        }
    }


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * VALIDATE PLC TX & RX GAIN VALUES.
     * 
     * This is done to minimize (but cannot eliminate) the chances of the 
     * operator programming gain values that prevent PLC communication. The 
     * gain values are checked against the range of valid values: if invalid, 
     * the default values are used (initialized at top of module). While this 
     * ensures the assignment of valid values, it cannot account for the line 
     * impedances of a particular implementation which could result in an 
     * 'orphaned' charger.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    if (ram_cfg.byte[FLASH_AR_TXG]<=S_LM_MAX_PTXG)
        plc_txg=ram_cfg.byte[FLASH_AR_TXG];
    if ((ram_cfg.byte[FLASH_AR_RXG]<=S_LM_MAX_PRXG) &&\
            (ram_cfg.byte[FLASH_AR_RXG]>=S_LM_MIN_PRXG))
        plc_rxg=ram_cfg.byte[FLASH_AR_RXG];


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * SET PORT CONFIGURATION:
     * + Set digital latch initial states
     * + Pin in/out-put states
     * + Analog / digital
     * + Open drain (tri-state)
     * + Slew rate
     * + Logic input levels
     * + SSR must be set off since the enable method relies on switching 
     *   between the PWM & latch using PPS. The PPS register is not cleared
     *   (e.g., set to latch) for non power-up resets.
     * 
     * Important: weak pull-ups are not used (OPTION_REG WPUEN is left @ default
     * of disabled) to avoid power-up spurious switching.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    PORTA=S_IN_PORTA;
    PORTB=S_IN_PORTB;
    PORTC=S_IN_PORTC;
    //Port A
    TRISA=IO_PORTA;
    ANSELA=AD_PORTA;
    ODCONA=OD_PORTA;
    SLRCONA=SR_PORTA;
    INLVLA=IL_PORTA;
    //Port B
    TRISB=IO_PORTB;
    ANSELB=AD_PORTB;
    ODCONB=OD_PORTB;
    SLRCONB=SR_PORTB;
    INLVLB=IL_PORTB;
    //Port C
    TRISC=IO_PORTC;
    ANSELC=AD_PORTC;
    ODCONC=OD_PORTC;
    SLRCONC=SR_PORTC;
    INLVLC=IL_PORTC;
    
    SSROFF();


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * Initialize USART for command console.
     * 
     * This should be done at end of port / pin config since it configures PPS
     * for the USART. Console echo OFF.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    usartSetRx(ENABLE);
    usartSetEcho(FALSE);


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * Initialize MSSP for I2C communication with PLC controller. Pin and PPS 
     * configuration must be completed separately.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    i2c_m_cfg();


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * CONFIGURE FVR & DACs:
     * 1=Vref for Vout control loop
     * 2=Vref for current limit
     * 5=Vref for MPP control loop
     * 3=Vdd (as of v3.0,build 69 not used)
     * 7=Vout over-volt protection
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    FVRCON=CFG_FVR_C;

    DAC1CON0=CFG_DAC_1_C;
    DAC1REFH=ram_cfg.byte[FLASH_AR_VSET_H];
    DAC1REFL=ram_cfg.byte[FLASH_AR_VSET_L];
    DAC1LD=TRUE;                                //Load reference value

    DAC2CON0=CFG_DAC_2_C;
    DAC2REFH=S_LM_CLIM_H;
    DAC2REFL=S_LM_CLIM_L;
    DAC2LD=TRUE;                                //Load reference value

    DAC5CON0=CFG_DAC_5_C;
    DAC5REFH=ram_cfg.byte[FLASH_AR_VMPP_H];
    DAC5REFL=ram_cfg.byte[FLASH_AR_VMPP_L];
    DAC5LD=TRUE;                                //Load reference value

    DAC3CON0=CFG_DAC_3_C;
    DAC3REF=CFG_DAC_3_R;

    DAC7CON0=CFG_DAC_7_C;
    DAC7REF=S_LM_HIVOUT;


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * CONFIGURE ADC:  enable and connect to first measurement source in RUN 
     * loop. It's expected that all further management of ADC is done thru 
     * function ReadVoltage.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    ADCON0=CFG_ADC_CON_0;
    ADCON1=CFG_ADC_CON_1;
    ADCON2=CFG_ADC_CON_2;


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * CONFIGURE OP-AMPS:
     * 2=Vout error amp
     * 3=MPP error amp
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    OPA2ORS=CFG_OPA_2_O;
    OPA2NCHS=CFG_OPA_2_N;
    OPA2PCHS=CFG_OPA_2_P;
    OPA2CON=CFG_OPA_2_C;

    OPA3ORS=CFG_OPA_3_O;
    OPA3NCHS=CFG_OPA_3_N;
    OPA3PCHS=CFG_OPA_3_P;
    OPA3CON=CFG_OPA_3_C;


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * CONFIGURE COMPARATORS:
     * 1=P2 slope compensation
     * 2=P2 current limit
     * 3=P1 slope compensation
     * 4=P1 current limit
     * 5=Vout over-volt fault sense
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    CM1CON1=CFG_CM_1_C1;
    CM1NSEL=CFG_CM_1_N;
    CM1PSEL=CFG_CM_1_P;
    CM1CON0=CFG_CM_1_C0;

    CM2CON1=CFG_CM_2_C1;
    CM2NSEL=CFG_CM_2_N;
    CM2PSEL=CFG_CM_2_P;
    CM2CON0=CFG_CM_2_C0;

    CM3CON1=CFG_CM_3_C1;
    CM3NSEL=CFG_CM_3_N;
    CM3PSEL=CFG_CM_3_P;
    CM3CON0=CFG_CM_3_C0;

    CM4CON1=CFG_CM_4_C1;
    CM4NSEL=CFG_CM_4_N;
    CM4PSEL=CFG_CM_4_P;
    CM4CON0=CFG_CM_4_C0;

    CM5CON1=CFG_CM_5_C1;
    CM5NSEL=CFG_CM_5_N;
    CM5PSEL=CFG_CM_5_P;
    CM5CON0=CFG_CM_5_C0;


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * CONFIGURE TIMERS:
     * 2=SSR period for PWM9
     * 3=P1 DC measurement
     * 5=P2 DC measurement
     * SYST=system chrono timing
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    T2CLKCON=CFG_TMR_2_CLK;
    T2HLT=CFG_TMR_2_HLT;
    T2RST=CFG_TMR_2_RST;
    T2PR=CFG_TMR_2_PR;
    T2CON=CFG_TMR_2_CON;

    T3CON = CFG_TMR_3_CON;
    T3GCON = CFG_TMR_3_GCON;

    T5CON = CFG_TMR_5_CON;
    T5GCON = CFG_TMR_5_GCON;

    S_AL_SYST_CLKCON=CFG_SYST_CLK;
    S_AL_SYST_HLT=CFG_SYST_HLT;
    S_AL_SYST_RST=CFG_SYST_RST;
    S_AL_SYST_PR=CFG_SYST_PR;
    S_AL_SYST_CON=CFG_SYST_CON;


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * CONFIGURE PWMs:
     * 5=P1
     * 6=P2
     * 9=SSR drive
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    PWM5INTE=CFG_PWM_5_INTE;
    PWM5CLKCON=CFG_PWM_5_CLK;
    PWM5LDCON=CFG_PWM_5_LDCON;
    PWM5OFCON=CFG_PWM_5_OFCON;
    PWM5PH=CFG_PWM_5_PHASE;
    PWM5DC=(uint16_t) ram_cfg.byte[FLASH_AR_DC];
    PWM5PR=CFG_PWM_5_PERIOD;
    PWM5OF=CFG_PWM_5_OFFSET;
    PWM5TMR=CFG_PWM_5_TIMER;
    PWM5CON=CFG_PWM_5_CON;

    PWM6INTE=CFG_PWM_6_INTE;
    PWM6CLKCON=CFG_PWM_6_CLK;
    PWM6LDCON=CFG_PWM_6_LDCON;
    PWM6OFCON=CFG_PWM_6_OFCON;
    PWM6PH=CFG_PWM_6_PHASE;
    PWM6DC=(uint16_t) ram_cfg.byte[FLASH_AR_DC];
    PWM6PR=CFG_PWM_6_PERIOD;
    PWM6OF=CFG_PWM_6_OFFSET;
    PWM6TMR=CFG_PWM_6_TIMER;
    PWM6CON=CFG_PWM_6_CON;

    PWM9DCH=(uint8_t) CFG_PWM_9_DCH;
    PWM9DCL=(uint8_t) CFG_PWM_9_DCL;
    PWM9CON=CFG_PWM_9_CON;


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * CONFIGURE PRGS:
     * 1=P1 slope comp
     * 2=P2 slope comp
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    PRG1FTSS=CFG_PRG_1_FTSS;
    PRG1RTSS=CFG_PRG_1_RTSS;
    PRG1CON2=S_DF_SCOMP;
    PRG1INS=CFG_PRG_1_INS;
    PRG1CON1=CFG_PRG_1_CON1;
    PRG1CON0=CFG_PRG_1_CON0;

    PRG2FTSS=CFG_PRG_2_FTSS;
    PRG2RTSS=CFG_PRG_2_RTSS;
    PRG2CON2=S_DF_SCOMP;
    PRG2INS=CFG_PRG_2_INS;
    PRG2CON1=CFG_PRG_2_CON1;
    PRG2CON0=CFG_PRG_2_CON0;


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * CONFIGURE CLCS:
     * 2=Gate connector from comparator C5 output to COG auto-shutdown for Vout
     *   over-volt fault
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    CLC2POL=CFG_CLC_2_POL;
    CLC2SEL0=CFG_CLC_2_SEL_0;
    CLC2SEL1=CFG_CLC_2_SEL_1;
    CLC2SEL2=CFG_CLC_2_SEL_2;
    CLC2SEL3=CFG_CLC_2_SEL_3;
    CLC2GLS0=CFG_CLC_2_GLS_0;
    CLC2GLS1=CFG_CLC_2_GLS_1;
    CLC2GLS2=CFG_CLC_2_GLS_2;
    CLC2GLS3=CFG_CLC_2_GLS_3;
    CLC2CON=CFG_CLC_2_CON;


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * CONFIGURE COGs:
     * 1=P1
     * 2=P2
     * 
     * Compile option _OPT_DIS_CLOSL dictates loop state. CON0 registers are 
     * loaded last to enable with set configuration.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    COG1DBR=CFG_COG_1_DBR;
    COG1DBF=CFG_COG_1_DBF;
    COG1BLKR=CFG_COG_1_BLKR;
    COG1BLKF=CFG_COG_1_BLKF;
    COG1PHR=CFG_COG_1_PHR;
    COG1PHF=CFG_COG_1_PHF;
    COG1ASD1=CFG_COG_1_ASD1;
    COG1ASD0=CFG_COG_1_ASD0;
    COG1RIS0=CFG_COG_1_RIS0;
    COG1RIS1=CFG_COG_1_RIS1;
    COG1FIS1=CFG_COG_1_FIS1;
    COG1RSIM0=CFG_COG_1_RSIM0;
    COG1RSIM1=CFG_COG_1_RSIM1;
    COG1FSIM0=CFG_COG_1_FSIM0;
    COG1FSIM1=CFG_COG_1_FSIM1;
    COG1CON1=CFG_COG_1_CON1;

    COG2DBR=CFG_COG_2_DBR;
    COG2DBF=CFG_COG_2_DBF;
    COG2BLKR=CFG_COG_2_BLKR;
    COG2BLKF=CFG_COG_2_BLKF;
    COG2PHR=CFG_COG_2_PHR;
    COG2PHF=CFG_COG_2_PHF;
    COG2ASD1=CFG_COG_2_ASD1;
    COG2ASD0=CFG_COG_2_ASD0;
    COG2RIS0=CFG_COG_2_RIS0;
    COG2RIS1=CFG_COG_2_RIS1;
    COG2FIS1=CFG_COG_2_FIS1;
    COG2RSIM0=CFG_COG_2_RSIM0;
    COG2RSIM1=CFG_COG_2_RSIM1;
    COG2FSIM0=CFG_COG_2_FSIM0;
    COG2FSIM1=CFG_COG_2_FSIM1;
    COG2CON1=CFG_COG_2_CON1;

#ifndef _OPT_DIS_CLOSL
    SETLOOPCLOSED();
#else
    SETLOOPOPEN();
#endif
    COG1CON0=CFG_COG_1_CON0;
    COG2CON0=CFG_COG_2_CON0;


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * MAP PERIPHERAL PINS:
     * + PWMs
     * + PRG triggers
     * + Gate drives
     * + Duty cycle measurement
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    PPS_P_P5P1=PPS_M_P5P1;                      //P1 PWM
    PRG1RPPS=PPS_I_P1P1;                        //Ramp rising trigger
    PRG1FPPS=PPS_I_P1P1;                        //Ramp falling trigger

    PPS_P_P6P2=PPS_M_P6P2;                      //P2 PWM
    PRG2RPPS=PPS_I_P2P2;                        //Ramp rising trigger
    PRG2FPPS=PPS_I_P2P2;                        //Ramp falling trigger

#ifdef _OPT_ENA_P1
    PPS_P_C1P1=PPS_M_C1P1;                      //P1 gate drive
#endif
    T3GPPS=PPS_I_T3P1;                          //DC measurement

#ifdef _OPT_ENA_P2
    PPS_P_CGP2=PPS_M_CGP2;                      //P2 gate drive
#endif
    T5GPPS=PPS_I_T5P2;                          //DC measurement

    //MSSP
    PPS_P_I2C_D_O=PPS_M_I2C_D_O;
    PPS_P_I2C_C_O=PPS_M_I2C_C_O;
    SSPDATPPS=PPS_M_I2C_D_I;
    SSPCLKPPS=PPS_M_I2C_C_I;


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * INITIALIZATION COMPLETE:
     * + Test for fault and if none set the STATE flags accordingly
     * + Enable peripheral & global interrupts
     * + Enable peripheral interrupts serviced by ISR
     * + Start system timer
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    if (sts == STS_NOF) state=STE_BOOT;

    PEIE=ENABLE;                        //Peripheral interrupts
    ei();                               //Global interrupts
    S_AL_SYST_IE=TRUE;                  //Enable system timer interrupt
    S_AL_SYST_EN=TRUE;                  //Enable system timer

    return;
} //End function Initialize



//End of file
