/*

DESCRIPTION:    Remote console for PVC chargers to manuall control & develop
                control automation for charger module (PVCRC).

                Preliminary specifications:

                An RS232 interface shared with ICSP provides an optional 
                console interface with a command set to monitor and control 
                operational parameters. Many parameters are stored in Non-
                Volatile Memory (NVM).

                RS232 settings: 115.2kbps, 8 data bits, 1 stop, no parity. The 
                console port is accessed over the micro-USB connector on the 
                PCB and is optimized for use with a GBIB-style control program.

                A status LED on the PCB indicates machine state:
                    + Red=
                    + Green=
                    + Red/Green=boot, HW or FW failure

SYNTAX:         n/a

KNOWN BUGS:     

COPYRIGHT:      (c) 2025, Brian Cornell, all rights reserved.

DEPENDENCIES:   see include file listing.

 TARGET MCU:    PIC16F1776

HW REVISION:    241209

VERSION:        3.0

MODIFICATION HISTORY:
 
     9 Feb 25, BWC, version 3 adapted from charger pvc-231029-fw-v3_0.X

    23 Jun 24, BWC, version 2 adapted from v1.0 branch of PVC code.
    

*/


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * COMPILER & CLIB INCLUDES.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <pic.h>
#include <xc.h>

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * FIRMWARE INCLUDES.
 * + binary:            boolean constants shared across firmware projects.
 * + sizes:             constants representing common object sizes, value 
 *                      limits, and characters.
 * + pic16f1776-map-v2: PIC specific peripheral alias, pin, NVM mapping, and
 *                      function declarations for shared functions.
 * + usart-v3:           usart function declarations & supporting constants,
 *                      objects, etc..
 * + config:            PIC config registers
 * + _xtal_freq:        sets the constant _XTAL_FREQ to the system oscillator
 *                      frequency (Fosc).  Used by the __delayxxx macros.
 * + usartc-v3:         EUSART peripheral user configuration constants.
 * + i2cm.h             I2C master constants, macros, function declarations.
 * + pvc:               main include file for this project.
 * + pic16f1776-vmv-v2: VMV configuration.
 * + plc                PLC constants, data definitions, macros shared with
 *                      remote console/controller.
 * 
 * PROJECT FOLDER ORGANIZATION & LOCATION OF INCLUDES.
 * /pvc-231029-fw-v3_0.X this (PVC) project.
 * /microchip/code      shared includes.
 * ../pvc-common        includes & code shared with remote console/controller.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#include <binary.h>                     //folder microchip/code
#include <sizes.h>                      //folder microchip/code
#include <pic16f1776-map-v2.h>          //folder microchip/code
#include <usart-v3.h>                   //folder microchip/code
#include "../pvc-common/config.h"
#ifndef _XTAL_FREQ
#include "../pvc-common/_xtal_freq.h"
#endif
#include "../pvc-common/usartc-v3.h"
#include "../pvc-common/i2cm.h"
#include "../pvc-common/pic16f1776-vmv-v2.h"
#include "../pvc-common/plc.h"          //Shared with remote console/controller
#include "pvc.h"


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * MODULE DATA.
 * + plc is the I2C buffer for exchanging data with the PLC controller.
 * + usart_nl is set TRUE to add a new line (NL) to the output of the 
 *   usart_wr_ui... functions.
 * + Flags cv0, 1, 2 are set TRUE to indicate that valid console command 
 *   parameters have been set in function main objects cmdp1 / cmdp2.
 * + Flag flag_pir is set TRUE to indicate a service required/problem with the 
 *   PLC.
 * + Flag flag_rqt is set true in ISR when the PLC request timer reaches zero.
 * + sts holds the status code, if any, associated with the machine state.
 * + msc maintains the system timer clock ticks for one second.
 * + state is a bit-mapped byte indicating the machine state (refer to STATE 
 *   MANAGEMENT AND STATUS CODES in include file).
 * + plc_irqs is the IRQ status byte read from the PLC controller service block
 *   in the RUN loop.
 * + plc_bufs is the size of valid data (Tx or Rx) in buffer plc_buf.
 * + plc_rxi is the contents of the RX_MESSAGE_INFO register in the PLC.
 * + plc_cmd stores the command sent from the remote node.
 * + plc_rqa contains the logical address a response is expected from.
 * + plc_lga stores the remote node (source) logical address.
 * + plc_rxd stores the PLC receive data buffer from the last completed request.
 * + plc_rqt is the PLC request timeout timer.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
volts   vdd={.channel=S_AL_ADCCH_VDD, .mvr=S_RT_ADC_MV_L, .adc1v=S_QY_ADC_1V_L,\
            .scale=S_RT_VDD},
        pvv={.channel=PIN_IA_INV, .mvr=S_RT_ADC_MV_H, .adc1v=S_QY_ADC_1V_H,\
            .scale=S_RT_VIN},
        pvc={.channel=PIN_IA_PVCS, .mvr=S_RT_ADC_MV_H, .adc1v=S_QY_ADC_1V_H,\
            .scale=1};

plci2c  plc;
chargert chg;                                   //Retrieved charger telemetry
chargerv chgv;                                  //Retrieved charger set voltages
chargerf chgf;                                  //Retrieved charger NVM fault
chargero chgo;                                  //Retrieved charger op. volts
chargera chgc;                                  //Retrieved charge alg values
chargerr chgr;                                  //Retrieved charger version #s
chargerg chgg;                                  //Retrieved charger PLC gain

extern __bit usart_nl;
extern unsigned char usart_txl;                 //Last byte sent to console

__bit   flag_cv1,
        flag_cv2,
        flag_pir,
        flag_rqt;
uint8_t sts=STS_NOF,
        msc=0,
        state,
        plc_irqs,
        plc_bufs,
        plc_rxi,
        plc_cmd,
        plc_rqa,
        plc_lga,
        plc_rxd[PLC_SZ_DATA],
        plc_rqt=0;

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * I2C MODULE EXTERNAL DATA.
 * + i2c_idle is TRUE when there is no active read/write in process.
 * + i2c_rsen is TRUE to issue an I2C bus RESTART at the end of a read/write
 *   to keep the bus active.
 * + i2c_rmode controls fore/background reads, initialized in function 
 *   i2c_m_cfg
 * + i2c_tor is the read timout timer.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
extern __bit i2c_idle,
             i2c_rsen,
             i2c_rmode;
extern uint8_t i2c_tor;


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * OPERATING CONFIGURATION MEMORY.
 * 
 * Non-volatile parameter storage & management. Object nvm_cfg resides in 
 * program flash at the top of the high-endurance section (to make the lower 
 * portion contiguous & available for code).  Object ram_cfg is the working 
 * copy in RAM.
 * 
 * Initialization of nvm_cfg & ram_cfg MUST match: these are system defaults and 
 * enable recovery after NVM corruption or to restore via console commands.  
 * Refer to detailed explanation of NVM management at beginning of function 
 * Initialize.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
const nvmcfg nvm_cfg __at(0x1FE0)={.byte={S_DF_FLASH_0,S_DF_FLASH_1,\
    S_DF_FLASH_2,S_DF_FLASH_3,S_DF_FLASH_4,S_DF_FLASH_5,S_DF_FLASH_6,\
    S_DF_FLASH_7,S_DF_FLASH_8,S_DF_FLASH_9,S_DF_FLASH_10,S_DF_FLASH_11,\
    S_DF_FLASH_12,S_DF_FLASH_13,S_DF_FLASH_14,S_DF_FLASH_15,S_DF_FLASH_16,\
    S_DF_FLASH_17,S_DF_FLASH_18,S_DF_FLASH_19,S_DF_FLASH_20,S_DF_FLASH_21,\
    S_DF_FLASH_22,S_DF_FLASH_23,S_DF_FLASH_24,S_DF_FLASH_25,S_DF_FLASH_26,\
    S_DF_FLASH_27,S_DF_FLASH_28,S_DF_FLASH_29,S_DF_FLASH_30,S_DF_FLASH_31}};
nvmcfg ram_cfg={.byte={S_DF_FLASH_0,S_DF_FLASH_1,S_DF_FLASH_2,S_DF_FLASH_3,\
    S_DF_FLASH_4,S_DF_FLASH_5,S_DF_FLASH_6,S_DF_FLASH_7,S_DF_FLASH_8,\
    S_DF_FLASH_9,S_DF_FLASH_10,S_DF_FLASH_11,S_DF_FLASH_12,S_DF_FLASH_13,\
    S_DF_FLASH_14,S_DF_FLASH_15,S_DF_FLASH_16,S_DF_FLASH_17,S_DF_FLASH_18,\
    S_DF_FLASH_19,S_DF_FLASH_20,S_DF_FLASH_21,S_DF_FLASH_22,S_DF_FLASH_23,\
    S_DF_FLASH_24,S_DF_FLASH_25,S_DF_FLASH_26,S_DF_FLASH_27,S_DF_FLASH_28,\
    S_DF_FLASH_29,S_DF_FLASH_30,S_DF_FLASH_31}};



/*Function main.  Control program entry point (from root code) for control 
 * program. All aspects of operation begin & end here. At entry the 
 * Initialization function is called to configure peripherals, data objects, 
 * etc. The STATUS & PCON register values are saved in case an MCU trap was
 * encountered. 
 * 
 * The WDT (Watchdog) timer is enabled for hard fault recovery and the RUN loop 
 * entered: it is an infinite loop that consists of several discrete blocks that
 * manage all aspects of operation. They are ordered by dependencies (e.g. data
 * collection must happen before fault detection) and/or criticality (e.g. fault
 * shutdown before console service). Refer to the documentation within each 
 * block for details.
 * 
 * Returns:     never
 *
 * Parameters:  none
 *
*/
void main(void)
{
    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * FUNCTION DATA.
     * + cmdb points to the start of the console command buffer.
     * + counter is a general purpose loop counter and scratch variable.  It 
     *   WILL NOT be used for multiple iterations thru the RUN loop.
     * + reg_status is assigned the value of the STATUS register at boot.  Refer
     *   to the 'MCU FAILURE RECORDING' block for more detail.
     * + reg_pcon is assigned the value of the PCON register at boot.  Refer
     *   to the 'MCU FAILURE RECORDING' block for more detail.
     * + cmdb_sz is the size of the console command buffer.
     * + cmdp1 is the integer value of the first console command parameter.
     * + cmdp1_o is the integer offset from the start of object cmdb for cmdp1.
     * + cmdp1_s is the size of cmdp1 (ASCII).
     * + cmdp2_o is the integer offset from the start of object cmdb for cmdp2.
     * + cmdp2 is the integer value of the second console command parameter.
     * + ttime is temporary 'safe' (not subject to corruption via updating in
     *   the ISR since it is larger than the MCU's native data size) storage of 
     *   the unit's total time.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    char *cmdb,
          cmdb_cpy[USARTSZRXBUF],
          *cmd_sep=": ";
    uint8_t counter=0,
            reg_status,
            reg_pcon,
            cmdb_sz,
            cmdp1,
            cmdp1_o,
            cmdp1_s,
            cmdp2_o;
    uint16_t ttime,
             cmdp2,
             pvc_cal=0;                 //Current sense calibration offset


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * MCU FAILURE RECORDING.
     * 
     * If an MCU failure (e.g. stack overflow, watchdog, etc.) is detected at 
     * boot it's important to understand what caused the fault. The STATUS &
     * PCON registers are saved at the start of the main function. Once 
     * initialization is complete, but before the run loop, these are saved to 
     * object ram_cfg for writing to NVM if the MCU status code is set.
     * 
     * They will remain in NVM until another MCU trap occurs, the NVM status
     * code is cleared, or defaults restored. Register values are only recorded 
     * when an MCU status is set so these will be 0x00 for any other code.
     * 
     * These registers are saved outside of the MCU fault detection code in 
     * function Initialize to minimize the change for alteration during the
     * function call.
     +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    reg_status=S_AL_REG_STATUS;
    reg_pcon=S_AL_REG_PCON;


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * INITIALIZATION.
     * 
     * Function Initialize configures the MCU and on-board peripherals. 
     * Afterwards a delay provides settle time and must be long enough for the
     * PLC controller to boot before attempting to configure.  Local objects 
     * are initialized before entering the RUN loop:
     * + Save STATUS & PCON registers in ram_cfg object for subsequent write to
     *   NVM if MCU error by the 'fault monitoring & recovery control' block
     *   within the RUN loop.
     * + The PLC initialization is completed regardless of possible 
     *   initialization errors to support possible recovery via commands from
     *   the charge controller.
     * + Perform initial calibration on the PVC current sense. The working 
     *   assumption is that there is no load when the console boots.
     +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    Initialize();
    __delay_ms(S_TM_INT);

    if (sts == STS_MCU)
    {
        ram_cfg.byte[FLASH_AR_STATUS]=(unsigned char) reg_status;
        ram_cfg.byte[FLASH_AR_PCON]=(unsigned char) reg_pcon;
    }

    if ((counter=cfgPLC()) != STS_NOF)
    {
        SETSTS(counter);
        state=STE_FAULT;
    }

    //Calibrate current sense
    ReadVoltage(&pvc);
    pvc_cal=pvc.adc;


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * RUN LOOP PRIORITIZATION & WATCHDOG MANAGEMENT.
     * 
     * Once entered, the run-time loop's only exit points are a reboot or power
     * cycle. The loop consists of several discrete blocks responsible for 
     * data collection, monitoring, and control.  They are organized in order
     * of criticality and/or dependency of subsequent blocks.  Categorically the
     * order is:
     *      + Sensor calibration
     *      + I2C slave data queries
     *      + Data collection
     *      + Fault monitoring
     *      + Shut-down
     *      + NVM updating / fault recording
     *      + Maximum Power Point tracking
     *      + Fault recovery
     *      + Boot (autostart) and restart
     *      + Status LED
     *      + Console service
     * 
     * The watchdog timer is expected to be on in the run loop regardless of
     * state.  It should be set to a value that accounts for the longest loop
     * time with at least a 100% margin.  If it must be disabled in anticipation
     * of a long delay then it is the responsibility of that code block to 
     * re-enable.
     +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    S_AL_WDT_ENABLE=TRUE;                   //Enable watchdog

    while (TRUE)                            //Begin RUN loop
    {
        CLRWDT();                           //Clear watchdog timer
        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * DATA COLLECTION.
         * 
         * This block measures system voltages & currents and COG (duty cycle) 
         * timing.
         *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        ReadVoltage(&pvv);
        ReadVoltage(&pvc);


        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * LED MANAGEMENT.
         * Set the LED color based on machine state:
         *  + Red=[recoverable] fault
         *  + Green=stopped, idle, run, and technically boot but that state 
         *    cannot occur if the first RUN loop has made it this far.
         *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        if (state & STE_RCRFT) PIN_OD_LED=TRUE;
        else PIN_OD_LED=FALSE;


        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * PLC SERVICE.
         * 
         * Polls the PLC interrupt request (IRQ) pin and when true reads the PLC
         * IRQ status register and services accordingly.
         * 
         * Before servicing the PLC the request timeout flag, flag_rqt is tested
         * and if TRUE indicates that a prior request (command) was sent to a 
         * node and it did not respond in PLC_TM_RQT seconds. A "timeout"
         * message is sent to the console and the flag cleared. This flag is set
         * in the ISR when the timing object plc_rqt is decremented to zero. If 
         * a request is processed before plc_rqt reaches zero, the PLC service
         * block sets it to zero which prevents the ISR from subsequently 
         * setting flag_rqt.
         * 
         * If flag_pir is TRUE, a prior IRQ service was not completed 
         * successfully and the system is already in a FAULT state. To avoid
         * further complications to the malfunction the IRQ is not serviced. An
         * EXCLUSIVE OR evaluation is not used because the fault could occur 
         * further in the service routine after the PLC IRQ is reset.
         * 
         * Object plc_cmd is cleared at the beginning of service to prevent the
         * possibility of re-processing a command when an unanticipated 
         * condition occurs. Hence, plc_cmd should not be used outside of this.
         *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        if (flag_rqt)
        {
            usart_wr_str("to");
            flag_rqt=FALSE;
        }
        if (PIN_ID_PIR && !(flag_pir))
        {
            flag_pir=TRUE;                              //Now servicing PLC
            i2c_rsen=TRUE;                              //Hold I2C bus
            plc_cmd=PLC_RCD_INV;                        //No valid command
            plc.wr.size=PLC_SZ_RRG;                     //Register address only
            plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);           //I2C address
            plc.wr.reg=PLC_OF_RIS;                      //IRQ status register
            while (TRUE)
            {
                //Set register pointer to IRQ status register
                WRPLCNACK();

                //Read IRQ status register
                plc.rd.i2ca=I2CSETRD(PLC_AD_I2C);
                RDPLCNACK();
                plc_irqs=(uint8_t) plc.rd.data[PLC_OF_RDT];
                SETINTPLC();                            //Clear PLC interrupts
                WRPLCNACK();

                /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                 * RETRIEVE RECEIVED DATA.
                 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
                if (plc_irqs & PLC_RIS_NDA)
                {
                    //Set PLC register to RX_MESSAGE_INFO
                                                plc.wr.size=PLC_SZ_RRG;
                    plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.wr.reg=PLC_OF_RRI;
                    WRPLCNACK();

                    //Read PLC register RX_MESSAGE_INFO & get data buffer size
                    plc.rd.i2ca=I2CSETRD(PLC_AD_I2C);
                    RDPLCNACK();
                    plc_rxi=(uint8_t) plc.rd.data[PLC_OF_RDT];  //Rx msg info
                    plc_bufs=(plc_rxi & PLC_RRI_RXS);           //Rx buffer size

                    //Retrieve source address, command, and data
                    plc.wr.size=PLC_SZ_RRG;
                    plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.wr.reg=PLC_OF_RRS;
                    WRPLCNACK();
                    plc.rdc.size=(plc_bufs+(PLC_SZ_CMD+PLC_SZ_PAD));
                    plc.rdc.i2ca=I2CSETRD(PLC_AD_I2C);
                    RDPLCNACK();
                    plc_lga=plc.rdc.lga[PLC_OF_LAD];        //Logical address
                    plc_cmd=plc.rdc.cmd;                    //Command ID
                    //Save copy of Rx buffer to free PLC object.
                    memcpy((void *) plc_rxd, (void *) plc.rdc.data, plc_bufs);
                }
                /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                 * PROCESSING FOR THINGS OTHER THAN RX DATA MIGHT GO HERE.
                 * 
                 * In this version/build they are not required: BUI, no PLC ACK,
                 * no response received, Rx packet dropped, and Tx data sent.
                 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

                /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                 * CLEAR NEW_RX_MSG REGISTER IN PLC.
                 * 
                 * This is done once received data & supporting metadata are
                 * saved off. This enables receipt of new messages by clearing 
                 * NEW_RX_MSG in the RX_Message_INFO register. This also clears 
                 * Status_RX_Packet_Dropped & Status_RX_Data_Available in the
                 * INT_Status register (clearing interrupts does not clear
                 * these).
                 * 
                 * This also releases the PLC receive buffer & overwrites
                 * object plc's data. For this remote console the I2C bus is
                 * always released since the remaining work involves storing
                 * retrieved data in the appropriate local objects & displaying.
                 * 
                 * For simplicity, the remainder of the PLC service block 
                 * executes even if there is no new data to service. This 
                 * ensures that PLC service is completed properly but at the
                 * expense of wasted execution time. The command block tests 
                 * will all be FALSE because object plc_cmd is set to invalid 
                 * command (PLC_RCD_INV) at the start of the PLC service block 
                 * and will not have been assigned to a valid command.
                 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
                i2c_rsen=FALSE;
                plc.wr.size=PLC_SZ_RCF;
                plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
                plc.wr.reg=PLC_OF_RRI;
                plc.wr.data[PLC_OF_WDT]=PLC_RRI_CLR;
                WRPLCNACK();

                /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                 * PROCESS RECEIVED DATA BASED ON COMMAND.
                 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

                /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                 * Received remote node PLC version (ping): display the 
                 * approximate time in 10mS increments.
                 +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
                if ((plc_cmd==PLC_RCD_GRV) && (plc_rqt))
                {
                    if (plc_lga==plc_rqa)
                        usart_wr_ui16((uint16_t) (PLC_TM_RQT-plc_rqt));
                    plc_rqt=0;
                }
                //Receive telemetry data response
                else if ((plc_cmd==PLC_RCD_TEL) && (plc_rqt))
                {
                    chg.ty.state=STE_INVLD;             //Cleared if valid data
                    chg.lga=PLC_AD_BRC;                 //Set if valid data
                    if (plc_lga!=plc_rqa) chg.ty.sts=STS_TRA;
                    else if (plc_bufs!=sizeof(tlmtry)) chg.ty.sts=STS_TSZ;
                    else
                    {
                        chg.lga=plc_lga;
                        memcpy((void *)&chg.ty,(void *)plc_rxd,sizeof(tlmtry));
                    }
                    dspTydata();
                }
                //Receive set voltages data response
                else if ((plc_cmd==PLC_RCD_GSV) && (plc_rqt))
                {
                    chgv.lga=PLC_AD_BRC;                //Set if valid data
                    if ((plc_lga == plc_rqa) && (plc_bufs == sizeof(setv)))
                    {
                        chgv.lga=plc_lga;
                        memcpy((void *)&chgv.sv, (void *)plc_rxd, sizeof(setv));
                    }
                    DSPSTV();
                }
                //Receive NVM fault data response
                else if ((plc_cmd==PLC_RCD_GFT) && (plc_rqt))
                {
                    chgf.lga=PLC_AD_BRC;                //Set if valid data
                    if ((plc_lga == plc_rqa) && (plc_bufs == sizeof(fault)))
                    {
                        chgf.lga=plc_lga;
                        memcpy((void *)&chgf.ft,(void *)plc_rxd,sizeof(fault));
                    }
                    DSPFTD();
                }
                //Receive operating voltages data response
                else if ((plc_cmd==PLC_RCD_GOP) && (plc_rqt))
                {
                    chgo.lga=PLC_AD_BRC;                //Set if valid data
                    if ((plc_lga == plc_rqa) && (plc_bufs == sizeof(opv)))
                    {
                        chgo.lga=plc_lga;
                        memcpy((void *)&chgo.ov,(void *)plc_rxd,sizeof(opv));
                    }
                    DSPOPV();
                }
                //Receive charge algorithm values
                else if ((plc_cmd==PLC_RCD_GCA) && (plc_rqt))
                {
                    chgc.lga=PLC_AD_BRC;                //Set if valid data
                    if ((plc_lga == plc_rqa) && (plc_bufs == sizeof(chga)))
                    {
                        chgc.lga=plc_lga;
                        memcpy((void *)&chgc.ca,(void *)plc_rxd,sizeof(chga));
                    }
                    DSPCGA();
                }
                //Receive FW & HW version numbers
                else if ((plc_cmd==PLC_RCD_GRL) && (plc_rqt))
                {
                    chgr.lga=PLC_AD_BRC;                //Set if valid data
                    if ((plc_lga == plc_rqa) && (plc_bufs == sizeof(hwfwr)))
                    {
                        chgr.lga=plc_lga;
                        memcpy((void *)&chgr.rl,(void *)plc_rxd,sizeof(hwfwr));
                    }
                    DSPCRL();
                }
                //Receive PLC Tx & Rx gain values
                else if ((plc_cmd==PLC_RCD_GPG) && (plc_rqt))
                {
                    chgg.lga=PLC_AD_BRC;                //Set if valid data
                    if ((plc_lga == plc_rqa) && (plc_bufs == sizeof(plcg)))
                    {
                        chgg.lga=plc_lga;
                        memcpy((void *)&chgg.pg,(void *)plc_rxd,sizeof(plcg));
                    }
                    DSPPGN();
                }

                plc_rqt=0;
                flag_pir=FALSE;
                break;
            }
            I2CHKBUS();

            //Set fault condition if problem communicating with PLC
            if (flag_pir)
            {
                SETSTS(STS_NAK);
            }
        }


        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * CONSOLE SERVICE.
         * 
         * IMPORTANT: do not place any code expected to execute on each RUN 
         * loop iteration beyond this point. The console service handler should 
         * always be placed at bottom of RUN loop.
         * 
         * This block services the user console accessed via the RS232 micro-USB
         * service port. All commands are two bytes in length with a single, 
         * optional, integer parameter. Refer to the definition of object 
         * CmdList for command strings, explanation, and values.
         * 
         * The command line accepts a 2-character command string followed by
         * up to two integer parameters. The cmdb object points to the NULL
         * terminated command string and the objects cmdp1 & cmdp2 contain the
         * integer parameters. Bit flags flag_cv1 & flag_cv2 are TRUE to 
         * indicate that the respective integer contains a value from the
         * command line.
         * 
         * The command structure is optimized to minimize processing time & code 
         * size balanced with not being too cryptic for the user. That said,
         * character echo is off and the commands provide no response on 
         * validity or success. No bounds checking is done with the exception of
         * the NVM byte address.
         * 
         * Care must be taken when entering NVM data to ensure the correct byte
         * is being updated. Further, 16-bit values are stored in little endian
         * format and the user must calculate the decimal equivalents of the
         * high/low hexadecimal values.
         * 
         * IMPORTANT: all set commands and NVM updates MUST be done with the 
         * converter stopped. NVM updates require an MCU reset to take effect.
         *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/

        //Retrieve command buffer & size, skip processing if too small.
        if (usartStatRx() != USARTSTSRXRDY) continue;
        cmdb=(char *) usartGetRx();             //Get command buffer
        if ((cmdb_sz=(uint8_t) usartGetSz()) < CMDSIZSTR)   //Too small
        {
            usartSetRx(ENABLE);
            continue;
        }
        memcpy((void *) &cmdb_cpy, (void *) cmdb, USARTSZRXBUF);

        //Parse buffer: command string & optional integer parameters
        *(cmdb+CMDSIZSTR)=S_CH_NULL;            //Terminate command string
        for (counter=(CMDSIZSTR+1), flag_cv1=FALSE, flag_cv2=FALSE; \
                (counter < cmdb_sz); ++counter)
        {
            if (*(cmdb+counter) > S_CH_SPACE)
            {
                if (!(flag_cv1))                //Found first parameter
                {
                    flag_cv1=TRUE;
                    cmdp1_s=1;
                    cmdp1_o=counter;
                    cmdp1=(uint8_t) atoi(cmdb+counter);
                }
                else if (!(flag_cv2))           //Found second parameter
                {
                    flag_cv2=TRUE;
                    cmdp2_o=counter;
                    cmdp2=(uint16_t) atoi(cmdb+counter);
                }
                for (; (*(cmdb+counter) > S_CH_SPACE); ++counter);
                if (flag_cv1 ^ flag_cv2) cmdp1_s=(counter-cmdp1_o);
            }
        }

        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * PROCESS COMMANDS.
         * 
         * First display command prefix (e.g., the command just typed).
         * 
         * Display command don't take parameters: they display what has already
         * been retrieved.
         +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        if (usart_txl != USARTKEYNL) putch(USARTKEYNL);
        DSPSEP();

        //Display charge algorithm values
        if (!(strcmp(cmdb, CmdList[CMDDSPCGA])))
        {
            DSPCGA();
        }
        //Display telemetry data
        else if (!(strcmp(cmdb, CmdList[CMDDSPTYD])))
        {
            dspTydata();
        }
        //Display operating voltages
        else if (!(strcmp(cmdb, CmdList[CMDDSPOPV])))
        {
            DSPOPV();
        }
        //Display set voltages
        else if (!(strcmp(cmdb, CmdList[CMDDSPSTV])))
        {
            DSPSTV();
        }
        //Display NVM fault data
        else if (!(strcmp(cmdb, CmdList[CMDDSPFTD])))
        {
            DSPFTD();
        }
        //Display FW/HW release numbers
        else if (!(strcmp(cmdb, CmdList[CMDDSPREL])))
        {
            DSPCRL();
        }
        //Display PLC Tx & Rx gain values
        else if (!(strcmp(cmdb, CmdList[CMDDSPPGN])))
        {
            DSPPGN();
        }
        //Read firmware release (local)
        else if (!(strcmp(cmdb, CmdList[CMDREDREL])))
        {
            usart_nl=FALSE;
            usart_wr_ui16(VERMAJOR);
            usart_wr_chr(S_CH_DECIMAL);
            usart_wr_ui16(VERMINOR);
            usart_wr_chr(S_CH_DECIMAL);
            usart_nl=TRUE;
            usart_wr_ui16(VERBUILD);
        }
        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * Reset console: first do clean stop if running. Disable watchdog to 
         * prevent erroneous reboot while resetting the PLC controller, and 
         * clear PCON bits to prevent error trap at MCU boot.
         +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        else if (!(strcmp(cmdb, CmdList[CMDXRTMCU])))
        {
            S_AL_WDT_ENABLE=FALSE;
            RESETPLC();
            S_AL_REG_PCON=S_LM_PCON_BT;
            RESET();
        }
        //Get PLC MAC address (local)
        else if (!(strcmp(cmdb, CmdList[CMDGETPMC])))
        {
            i2c_rsen=TRUE;                              //Hold I2C bus
            plc.wr.size=PLC_SZ_RRG;                     //Register address only
            plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);           //I2C address
            plc.wr.reg=PLC_OF_RPA;                      //Phy. address register
            if (i2c_m_Wd((unsigned char *) &plc) == I2C_ACK)
            {
                i2c_rsen=FALSE;
                plc.rd.size=PLC_SZ_PAD;                 //Read full address
                plc.rd.i2ca=I2CSETRD(PLC_AD_I2C);
                if (i2c_m_Rd((unsigned char *) &plc) == I2C_ACK)
                {
                    for (counter=PLC_OF_RDT; (counter < (PLC_SZ_PAD)); 
                            ++counter)
                    {
                        //Use cmdp1 as scratch variable
                        cmdp1=(((plc.rd.data[counter]>>S_BP_U8_H)&S_MK_U8_L)\
                                +S_CH_ZERO);
                        if (cmdp1>S_CH_NINE) cmdp1+=S_QY_9TOA;
                        putch((char) cmdp1);
                        cmdp1=((plc.rd.data[counter]&S_MK_U8_L)+S_CH_ZERO);
                        if (cmdp1>S_CH_NINE) cmdp1+=S_QY_9TOA;
                        putch((char) cmdp1);
                    }
                    putch(S_CH_NEWLINE);
                }
            }
            I2CHKBUS();
        }
        //Calibrate current sense (local)
        else if (!(strcmp(cmdb, CmdList[CMDXPVCCS])))
        {
            pvc_cal=pvc.adc;
        }
        //Display PV power (local)
        else if (!(strcmp(cmdb, CmdList[CMDDSPPVP])))
        {
            //Calculate the current, use pvc.adc as working object
            if (pvc_cal>pvc.adc) pvc.adc=pvc_cal;           //Zero out bad cal.
            pvc.adc=(((pvc.adc-pvc_cal)*S_RT_ADC_MV_H)/8);  //80mV per amp
            usart_nl=FALSE;
            usart_wr_ui16(pvv.adc);                         //Display PV volts
            putch(S_CH_SPACE);
            usart_wr_ui16(pvc.adc);                         //Display PV current
            putch(S_CH_SPACE);
            usart_nl=TRUE;
            usart_wr_ui16((uint16_t)\
                    (((long unsigned)pvv.adc*(long unsigned)pvc.adc)/100));
        }
        /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
         * All remaining commands require at least first parameter: test for 
         * this before parsing command.
         +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
        else if (flag_cv1)
        {
            //Get (read) PLC register (local)
            if (!(strcmp(cmdb, CmdList[CMDGETPLR])))
            {
                i2c_rsen=TRUE;
                plc.wr.size=PLC_SZ_RRG;
                plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
                plc.wr.reg=(uint8_t) cmdp1;
                if (i2c_m_Wd((unsigned char *) &plc) == I2C_ACK)
                {
                    i2c_rsen=FALSE;
                    plc.rd.size=PLC_SZ_RRG;
                    plc.rd.i2ca=I2CSETRD(PLC_AD_I2C);
                    if (i2c_m_Rd((unsigned char *) &plc) == I2C_ACK)
                        usart_wr_ui16((uint16_t) plc.rd.data[PLC_OF_RDT]);
                }
                I2CHKBUS();
            }
            //Get system status (local)
            else if (!(strcmp(cmdb, CmdList[CMDGETSTS])))
            {
                if (cmdp1) usart_wr_ui16(sts);
                else usart_wr_ui16(state);
            }
            //Get operating voltages <addr>
            else if (!(strcmp(cmdb, CmdList[CMDGETOPV])))
            {
                //Don't allow broadcast or another request if prior still active
                while ((cmdp1!=PLC_AD_BRC) && !(plc_rqt))
                {
                    //Set Tx config, PLC address, remote command
                    i2c_rsen=TRUE;
                    plc.t.size=sizeof(txcfg);
                    plc.t.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.t.txc.reg=PLC_OF_RTC;
                    plc.t.txc.cfg=PLC_CF_RTC;
                    plc.t.txc.lga[PLC_OF_WDT]=(uint8_t) cmdp1;
                    plc.t.txc.cmd=PLC_RCD_GOP;
                    plc_rqa=(uint8_t) cmdp1;        //To validate against resp.
                    WRPLCNACK();

                    //Send request
                    i2c_rsen=FALSE;
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=PLC_RTL_SND;
                    WRPLCNACK();

                    plc_rqt=PLC_TM_RQT;
                    break;
                }
                I2CHKBUS();
            }
            //Get telemetry <addr>
            else if (!(strcmp(cmdb, CmdList[CMDGETELM])))
            {
                //Don't allow broadcast or another request if prior still active
                while ((cmdp1!=PLC_AD_BRC) && !(plc_rqt))
                {
                    //Set Tx config, PLC address, remote command
                    i2c_rsen=TRUE;
                    plc.t.size=sizeof(txcfg);
                    plc.t.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.t.txc.reg=PLC_OF_RTC;
                    plc.t.txc.cfg=PLC_CF_RTC;
                    plc.t.txc.lga[PLC_OF_WDT]=(uint8_t) cmdp1;
                    plc.t.txc.cmd=PLC_RCD_TEL;
                    plc_rqa=(uint8_t) cmdp1;        //To validate against resp.
                    WRPLCNACK();

                    //Send request
                    i2c_rsen=FALSE;
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=PLC_RTL_SND;
                    WRPLCNACK();

                    plc_rqt=PLC_TM_RQT;
                    break;
                }
                I2CHKBUS();
            }
            //Get set voltages <addr>
            else if (!(strcmp(cmdb, CmdList[CMDGETSTV])))
            {
                //Don't allow broadcast or another request if prior still active
                while ((cmdp1!=PLC_AD_BRC) && !(plc_rqt))
                {
                    //Set Tx config, PLC address, remote command
                    i2c_rsen=TRUE;
                    plc.t.size=sizeof(txcfg);
                    plc.t.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.t.txc.reg=PLC_OF_RTC;
                    plc.t.txc.cfg=PLC_CF_RTC;
                    plc.t.txc.lga[PLC_OF_WDT]=(uint8_t) cmdp1;
                    plc.t.txc.cmd=PLC_RCD_GSV;
                    plc_rqa=(uint8_t) cmdp1;        //To validate against resp.
                    WRPLCNACK();

                    //Send request
                    i2c_rsen=FALSE;
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=PLC_RTL_SND;
                    WRPLCNACK();

                    plc_rqt=PLC_TM_RQT;
                    break;
                }
                I2CHKBUS();
            }
            //Get NVM fault data <addr>
            else if (!(strcmp(cmdb, CmdList[CMDGETFTD])))
            {
                //Don't allow broadcast or another request if prior still active
                while ((cmdp1!=PLC_AD_BRC) && !(plc_rqt))
                {
                    //Set Tx config, PLC address, remote command
                    i2c_rsen=TRUE;
                    plc.t.size=sizeof(txcfg);
                    plc.t.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.t.txc.reg=PLC_OF_RTC;
                    plc.t.txc.cfg=PLC_CF_RTC;
                    plc.t.txc.lga[PLC_OF_WDT]=(uint8_t) cmdp1;
                    plc.t.txc.cmd=PLC_RCD_GFT;
                    plc_rqa=(uint8_t) cmdp1;        //To validate against resp.
                    WRPLCNACK();

                    //Send request
                    i2c_rsen=FALSE;
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=PLC_RTL_SND;
                    WRPLCNACK();

                    plc_rqt=PLC_TM_RQT;
                    break;
                }
                I2CHKBUS();
            }
            //Get charge algorithm values <addr>
            else if (!(strcmp(cmdb, CmdList[CMDGETCGA])))
            {
                //Don't allow broadcast or another request if prior still active
                while ((cmdp1!=PLC_AD_BRC) && !(plc_rqt))
                {
                    //Set Tx config, PLC address, remote command
                    i2c_rsen=TRUE;
                    plc.t.size=sizeof(txcfg);
                    plc.t.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.t.txc.reg=PLC_OF_RTC;
                    plc.t.txc.cfg=PLC_CF_RTC;
                    plc.t.txc.lga[PLC_OF_WDT]=(uint8_t) cmdp1;
                    plc.t.txc.cmd=PLC_RCD_GCA;
                    plc_rqa=(uint8_t) cmdp1;        //To validate against resp.
                    WRPLCNACK();

                    //Send request
                    i2c_rsen=FALSE;
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=PLC_RTL_SND;
                    WRPLCNACK();

                    plc_rqt=PLC_TM_RQT;
                    break;
                }
                I2CHKBUS();
            }
            //Get FW/HW release numbers <addr>
            else if (!(strcmp(cmdb, CmdList[CMDGETREL])))
            {
                //Don't allow broadcast or another request if prior still active
                while ((cmdp1!=PLC_AD_BRC) && !(plc_rqt))
                {
                    //Set Tx config, PLC address, remote command
                    i2c_rsen=TRUE;
                    plc.t.size=sizeof(txcfg);
                    plc.t.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.t.txc.reg=PLC_OF_RTC;
                    plc.t.txc.cfg=PLC_CF_RTC;
                    plc.t.txc.lga[PLC_OF_WDT]=(uint8_t) cmdp1;
                    plc.t.txc.cmd=PLC_RCD_GRL;
                    plc_rqa=(uint8_t) cmdp1;        //To validate against resp.
                    WRPLCNACK();

                    //Send request
                    i2c_rsen=FALSE;
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=PLC_RTL_SND;
                    WRPLCNACK();

                    plc_rqt=PLC_TM_RQT;
                    break;
                }
                I2CHKBUS();
            }
            //Get PLC Tx & Rx gain values
            else if (!(strcmp(cmdb, CmdList[CMDGETPGN])))
            {
                //Don't allow broadcast or another request if prior still active
                while ((cmdp1!=PLC_AD_BRC) && !(plc_rqt))
                {
                    //Set Tx config, PLC address, remote command
                    i2c_rsen=TRUE;
                    plc.t.size=sizeof(txcfg);
                    plc.t.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.t.txc.reg=PLC_OF_RTC;
                    plc.t.txc.cfg=PLC_CF_RTC;
                    plc.t.txc.lga[PLC_OF_WDT]=(uint8_t) cmdp1;
                    plc.t.txc.cmd=PLC_RCD_GPG;
                    plc_rqa=(uint8_t) cmdp1;        //To validate against resp.
                    WRPLCNACK();

                    //Send request
                    i2c_rsen=FALSE;
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=PLC_RTL_SND;
                    WRPLCNACK();

                    plc_rqt=PLC_TM_RQT;
                    break;
                }
                I2CHKBUS();
            }
            //Set PLC logical address <MAC, addr>
            else if (!(strcmp(cmdb, CmdList[CMDSETPLA])))
            {
                if ((cmdp1_s==PLC_SZ_PAA) & flag_cv2 & (cmdp2!=PLC_AD_BRC))
                {
                    i2c_rsen=TRUE;
                    counter=PLC_OF_WDT;
                    plc.wr.size=PLC_SZ_WLA;
                    plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.wr.reg=PLC_OF_RTC;
                    plc.wr.data[counter]=PLC_CF_RTP;

                    //Write MAC address to PLC data buffer
                    //Use flag_cv1 as a nibble flag
                    for (++counter, cmdp1_s+=cmdp1_o, flag_cv1=FALSE; 
                            (cmdp1_o<cmdp1_s); ++cmdp1_o)
                    {
                        if (!(flag_cv1))                    //High nibble
                        {
                            if (*(cmdb+cmdp1_o) > S_CH_NINE)
                                plc.wr.data[counter]=(uint8_t) \
                                        ((*(cmdb+cmdp1_o)-S_CH_SEVEN)\
                                        <<S_BP_U8_H);
                            else plc.wr.data[counter]=(uint8_t) \
                                    ((*(cmdb+cmdp1_o)-S_CH_ZERO)<<S_BP_U8_H);
                            flag_cv1=TRUE;
                        }
                        else                                //Low nibble
                        {
                            if (*(cmdb+cmdp1_o) > S_CH_NINE)
                                plc.wr.data[counter]|=\
                                        (*(cmdb+cmdp1_o)-S_CH_SEVEN);
                            else plc.wr.data[counter]|=\
                                    (*(cmdb+cmdp1_o)-S_CH_ZERO);
                            flag_cv1=FALSE;
                            //Binary address increments every 2 ASCII bytes
                            ++counter;
                        }
                    }

                    plc.wr.data[counter]=PLC_RCD_LAD;       //Logical addr cmd
                    plc.wr.data[++counter]=(uint8_t) cmdp2; //Logical addr
                    if (i2c_m_Wd((unsigned char *) &plc)==I2C_ACK)
                    {
                        i2c_rsen=FALSE;
                        plc.wr.size=PLC_SZ_RCF;
                        plc.wr.reg=PLC_OF_RTL;
                        plc.wr.data[PLC_OF_WDT]=(PLC_RTL_SND | PLC_SZ_LAD);
                        WRPLC();
                    }
                    I2CHKBUS();
                }
            }
            //Set converter off/on <addr, 0/1>
            else if (!(strcmp(cmdb, CmdList[CMDSETCNV])))
            {
                while (flag_cv2)
                {
                    i2c_rsen=TRUE;
                    plc.t.size=sizeof(txcfg);
                    plc.t.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.t.txc.reg=PLC_OF_RTC;
                    SETACFG(t);
                    plc.t.txc.lga[PLC_OF_WDT]=(uint8_t) cmdp1;
                    if (cmdp2) plc.t.txc.cmd=PLC_RCD_STR;
                    else plc.t.txc.cmd=PLC_RCD_STP;
                    WRPLCNACK();

                    //Send command
                    i2c_rsen=FALSE;
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=PLC_RTL_SND;
                    WRPLC();
                    break;
                }
                I2CHKBUS();
            }
            //Set MPP voltage <addr, value>
            else if (!(strcmp(cmdb, CmdList[CMDSETMPV])))
            {
                while (flag_cv2)
                {
                    i2c_rsen=TRUE;
                    plc.gsv.size=(sizeof(txcfg)+sizeof(setv));
                    plc.gsv.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.gsv.txc.reg=PLC_OF_RTC;
                    SETACFG(gsv);
                    plc.gsv.txc.lga[PLC_OF_WDT]=(uint8_t) cmdp1;
                    plc.gsv.txc.cmd=PLC_RCD_SMV;
                    plc.gsv.sv.mpp=cmdp2;
                    WRPLCNACK();

                    //Send command
                    i2c_rsen=FALSE;
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=(PLC_RTL_SND | sizeof(setv));
                    WRPLC();
                    break;
                }
                I2CHKBUS();
            }
            //Set output voltage <addr, value>
            else if (!(strcmp(cmdb, CmdList[CMDSETOPV])))
            {
                while (flag_cv2)
                {
                    i2c_rsen=TRUE;
                    plc.gsv.size=(sizeof(txcfg)+sizeof(setv));
                    plc.gsv.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.gsv.txc.reg=PLC_OF_RTC;
                    SETACFG(gsv);
                    plc.gsv.txc.lga[PLC_OF_WDT]=(uint8_t) cmdp1;
                    plc.gsv.txc.cmd=PLC_RCD_SOV;
                    plc.gsv.sv.out=cmdp2;
                    WRPLCNACK();

                    //Send command
                    i2c_rsen=FALSE;
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=(PLC_RTL_SND | sizeof(setv));
                    WRPLC();
                    break;
                }
                I2CHKBUS();
            }
            /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
             * Set charge controller address <addr> <addr>
             * 
             * A converter cannot also be the charge controller nor can the 
             * charge controller's address be the broadcast address
             * (PLC_AD_BRC).
             +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
            else if (!(strcmp(cmdb, CmdList[CMDSETCCA])))
            {
                while ((flag_cv2) && (cmdp2!=PLC_AD_BRC) && (cmdp1!=cmdp2))
                {
                    i2c_rsen=TRUE;
                    plc.sc.size=(sizeof(txcfg)+PLC_SZ_LAD);
                    plc.sc.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.sc.txc.reg=PLC_OF_RTC;
                    SETACFG(sc);
                    plc.sc.txc.lga[PLC_OF_WDT]=(uint8_t) cmdp1;
                    plc.sc.txc.cmd=PLC_RCD_CCA;
                    plc.sc.val=(uint8_t) cmdp2;
                    WRPLCNACK();

                    //Send command
                    i2c_rsen=FALSE;
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=(PLC_RTL_SND | PLC_SZ_LAD);
                    WRPLC();
                    break;
                }
                I2CHKBUS();
            }
            /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
             * Set operating mode <addr, value>
             * Object cmdp2 should be: S_BL_ECTRL, S_BL_ECHGA, or S_BL_EAUTO,
             * or S_BL_EMANU.
             +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
            else if (!(strcmp(cmdb, CmdList[CMDSETMOD])))
            {
                while (flag_cv2)
                {
                    i2c_rsen=TRUE;
                    plc.sc.size=(sizeof(txcfg)+PLC_SZ_UI8);
                    plc.sc.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.sc.txc.reg=PLC_OF_RTC;
                    SETACFG(sc);
                    plc.sc.txc.lga[PLC_OF_WDT]=(uint8_t) cmdp1;
                    plc.sc.txc.cmd=PLC_RCD_MOD;
                    plc.sc.val=(uint8_t) cmdp2;
                    WRPLCNACK();

                    //Send command
                    i2c_rsen=FALSE;
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=(PLC_RTL_SND | PLC_SZ_UI8);
                    WRPLC();
                    break;
                }
                I2CHKBUS();
            }
            /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
             * Set charge algorithm values <addr, comma-separated-values>
             +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
            else if (!(strcmp(cmdb, CmdList[CMDSETCGA])))
            {
                uint16_t *ca;                   //Points to values in plc.gca.ca
                uint8_t cnt;                    //Counts assigned values

                while (flag_cv2)
                {
                    i2c_rsen=TRUE;
                    plc.gca.size=(sizeof(txcfg)+sizeof(chga));
                    plc.gca.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.gca.txc.reg=PLC_OF_RTC;
                    SETACFG(gca);
                    plc.gca.txc.lga[PLC_OF_WDT]=(uint8_t) cmdp1;
                    plc.gca.txc.cmd=PLC_RCD_SCA;

                    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                     * Object cmdp2 must be parsed into five individual values
                     * from the comma-separated string pointed to by cmdp2_o.
                     * 
                     * To do this efficiently the chga object, plc.gca.ca, is
                     * treated as an array of uint16_t values. Local object ca
                     * is initialized to point to the first one and local object
                     * cnt counts the number of commas processed. Function 
                     * object counter is used to point to the start of the 
                     * current value in the comma-separated string, and cmdp2_o
                     * is expected to point to the start of command parameter 2
                     * at the start of processing. When a comma is encountered,
                     * it is replaced with a NULL and the value starting at
                     * the offset of counter is passed to function atoi to
                     * convert to an integer. All offset & pointer objects are
                     * subsequently updated to process the next value.
                     * 
                     * The final value is captured when the for loop completes
                     * unless the operator enters a trailing comma/values: if
                     * so the loop will capture the final value & then 
                     * terminate.
                     * 
                     * While efficient, this is dangerous because this code
                     * has no way of knowing if object chga is updated. This is
                     * okay for the remote console since the charger performs
                     * a stringent evaluation of the charging parameters when
                     * loading from NVM.
                     +++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
                    for (plc.gca.ca.vnom=0,plc.gca.ca.vblk=0,plc.gca.ca.vabs=0,\
                            plc.gca.ca.vflt=0,plc.gca.ca.tabs=0,\
                            ca=(uint16_t *)&plc.gca.ca,cnt=0,counter=cmdp2_o;\
                            ((cmdp2_o < cmdb_sz) && (cnt<(sizeof(chga)/2)));\
                            ++cmdp2_o)
                    {
                        if (*(cmdb+cmdp2_o)==S_CH_COMMA)
                        {
                            *(cmdb+cmdp2_o)=S_CH_NULL;
                            *ca=(uint16_t) atoi(cmdb+counter);
                            ++cnt;
                            ++ca;
                            counter=(cmdp2_o+1);
                        }
                        
                    }
                    if (cnt<(sizeof(chga)/2)) *ca=(uint16_t) atoi(cmdb+counter);
                    WRPLCNACK();                //Load data to PLC

                    //Send command
                    i2c_rsen=FALSE;
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=(PLC_RTL_SND | sizeof(chga));
                    WRPLC();
                    break;
                }
                I2CHKBUS();
            }
            /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
             * Set PLC Tx & Rx gain values <addr, comma-separated-values>
             +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
            else if (!(strcmp(cmdb, CmdList[CMDSETPGN])))
            {
                uint8_t *pg;                    //Points to values in plc.ggn.pg
                uint8_t cnt;                    //Counts assigned values

                while (flag_cv2)
                {
                    i2c_rsen=TRUE;
                    plc.gca.size=(sizeof(txcfg)+sizeof(plcg));
                    plc.gca.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.gca.txc.reg=PLC_OF_RTC;
                    SETACFG(gca);
                    plc.gca.txc.lga[PLC_OF_WDT]=(uint8_t) cmdp1;
                    plc.gca.txc.cmd=PLC_RCD_SPG;

                    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
                     * Object cmdp2 must be parsed into two individual values
                     * from the comma-separated string pointed to by cmdp2_o.
                     * 
                     * To do this efficiently the chgg object, plc.ggn.pg, is
                     * treated as an array of uint18_t values. Local object pg
                     * is initialized to point to the first one and local object
                     * cnt counts the number of commas processed. Object counter
                     * is used to point to the start of the current value in the
                     * comma-separated string, and cmdp2_o is expected to point
                     * to the start of command parameter 2 at the start of 
                     * processing. When a comma is encountered, it is replaced 
                     * with a NULL and the value starting at the offset of 
                     * counter is passed to function atoi to convert to an 
                     * integer. All offset & pointer objects are subsequently 
                     * updated to process the next value.
                     * 
                     * The final value is captured when the for loop completes
                     * unless the operator enters a trailing comma/values: if
                     * so the loop will capture the final value & then 
                     * terminate.
                     * 
                     * While efficient, this is dangerous because this code
                     * has no way of knowing if object plcg is updated. This is
                     * okay for the remote console since the charger confirms
                     * the validity of gain values when loading from NVM.
                     +++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
                    for (plc.ggn.pg.txg=0,plc.ggn.pg.rxg=0,\
                            pg=(uint8_t *)&plc.ggn.pg,cnt=0,counter=cmdp2_o;\
                            ((cmdp2_o < cmdb_sz) && (cnt<sizeof(plcg)));\
                            ++cmdp2_o)
                    {
                        if (*(cmdb+cmdp2_o)==S_CH_COMMA)
                        {
                            *(cmdb+cmdp2_o)=S_CH_NULL;
                            *pg=(uint8_t) atoi(cmdb+counter);
                            ++cnt;
                            ++pg;
                            counter=(cmdp2_o+1);
                        }
                        
                    }
                    if (cnt<sizeof(plcg)) *pg=(uint8_t) atoi(cmdb+counter);
                    WRPLCNACK();                //Load data to PLC

                    //Send command
                    i2c_rsen=FALSE;
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=(PLC_RTL_SND | sizeof(plcg));
                    WRPLC();
                    break;
                }
                I2CHKBUS();
            }
            //Incr/decr Vout <addr, 0/1>
            else if (!(strcmp(cmdb, CmdList[CMDINCVOI])))
            {
                while (flag_cv2)
                {
                    i2c_rsen=TRUE;
                    plc.sc.size=(sizeof(txcfg)+PLC_SZ_LAD);
                    plc.sc.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.sc.txc.reg=PLC_OF_RTC;
                    SETACFG(sc);
                    plc.sc.txc.lga[PLC_OF_WDT]=(uint8_t) cmdp1;
                    plc.sc.txc.cmd=PLC_RCD_VOI;
                    plc.sc.val=(uint8_t) cmdp2;             //Increase Vout !=0
                    WRPLCNACK();

                    //Send command
                    i2c_rsen=FALSE;
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=(PLC_RTL_SND | PLC_SZ_UI8);
                    WRPLC();
                    break;
                }
                I2CHKBUS();
            }
            //Reset charger <addr>
            else if (!(strcmp(cmdb, CmdList[CMDXRTCHG])))
            {
                while (TRUE)
                {
                    i2c_rsen=TRUE;
                    plc.t.size=sizeof(txcfg);
                    plc.t.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.t.txc.reg=PLC_OF_RTC;
                    SETACFG(t);
                    plc.t.txc.lga[PLC_OF_WDT]=(uint8_t) cmdp1;
                    plc.t.txc.cmd=PLC_RCD_RBT;
                    WRPLCNACK();

                    //Send command
                    i2c_rsen=FALSE;
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=PLC_RTL_SND;
                    WRPLC();
                    break;
                }
                I2CHKBUS();
            }
            //Clear NVM fault data <addr>
            else if (!(strcmp(cmdb, CmdList[CMDXFTCLR])))
            {
                while (TRUE)
                {
                    i2c_rsen=TRUE;
                    plc.t.size=sizeof(txcfg);
                    plc.t.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.t.txc.reg=PLC_OF_RTC;
                    SETACFG(t);
                    plc.t.txc.lga[PLC_OF_WDT]=(uint8_t) cmdp1;
                    plc.t.txc.cmd=PLC_RCD_CFT;
                    WRPLCNACK();

                    //Send command
                    i2c_rsen=FALSE;
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=PLC_RTL_SND;
                    WRPLC();
                    break;
                }
                I2CHKBUS();
            }
            //Restore NVM defaults <addr>
            else if (!(strcmp(cmdb, CmdList[CMDXNVDFL])))
            {
                while (TRUE)
                {
                    i2c_rsen=TRUE;
                    plc.t.size=sizeof(txcfg);
                    plc.t.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.t.txc.reg=PLC_OF_RTC;
                    SETACFG(t);
                    plc.t.txc.lga[PLC_OF_WDT]=(uint8_t) cmdp1;
                    plc.t.txc.cmd=PLC_RCD_RNV;
                    WRPLCNACK();

                    //Send command
                    i2c_rsen=FALSE;
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=PLC_RTL_SND;
                    WRPLC();
                    break;
                }
                I2CHKBUS();
            }
            //Ping PLC <address>
            else if (!(strcmp(cmdb, CmdList[CMDXPGPLC])))
            {
                //Don't allow broadcast or another request if prior still active
                while ((cmdp1!=PLC_AD_BRC) && !(plc_rqt))
                {
                    //Set Tx config, PLC address, remote command
                    i2c_rsen=TRUE;
                    plc.t.size=sizeof(txcfg);
                    plc.t.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.t.txc.reg=PLC_OF_RTC;
                    plc.t.txc.cfg=PLC_CF_RTC;
                    plc.t.txc.lga[PLC_OF_WDT]=(uint8_t) cmdp1;
                    plc.t.txc.cmd=PLC_RCD_GRV;
                    plc_rqa=(uint8_t) cmdp1;        //To validate against resp.
                    WRPLCNACK();

                    //Send request
                    i2c_rsen=FALSE;
                    plc.wr.size=PLC_SZ_RCF;
                    plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
                    plc.wr.reg=PLC_OF_RTL;
                    plc.wr.data[PLC_OF_WDT]=PLC_RTL_SND;
                    WRPLCNACK();

                    plc_rqt=PLC_TM_RQT;
                    break;
                }
                I2CHKBUS();
            }
        }

        usartSetRx(ENABLE);                 //Look for new user input
    }
} //End function main


/* Function cfgPLC. Configures the PLC controller.
 * 
 * Returns:     uint8_t indicating status:
 *                  PLC_RN_NOE=no error,
 *                  PLC_RN_NAK=no response from controller.
 *
 * Parameters:  none.
 *
*/
uint8_t cfgPLC()
{
    uint8_t rc=STS_NAK;

    while (TRUE)
    {
        i2c_rsen=TRUE;                                  //Set I2C bus busy

        //Set node logical address
        plc.wr.i2ca=I2CSETWR(PLC_AD_I2C);
        plc.wr.size=PLC_SZ_RCF;
        plc.wr.reg=PLC_OF_RLL;
        plc.wr.data[PLC_OF_WDT]=ram_cfg.byte[FLASH_AR_PLCADR];
        WRPLCNACK();

        //Set the PLC mode
        plc.wr.reg=PLC_OF_RMD;
        plc.wr.data[PLC_OF_WDT]=PLC_CF_RMD;
        WRPLCNACK();

        //Set the default Tx config
        plc.wr.reg=PLC_OF_RTC;
        plc.wr.data[PLC_OF_WDT]=PLC_CF_RTC;
        WRPLCNACK();

        //Set the threshold noise, modem, Tx & Rx gain registers
        plc.wr.reg=PLC_OF_RTN;
        plc.wr.data[PLC_OF_WDT]=PLC_CF_RTN;
        plc.wr.data[PLC_OF_WDT+1]=PLC_CF_RMC;
        plc.wr.data[PLC_OF_WDT+2]=PLC_CF_RTG;
        plc.wr.data[PLC_OF_WDT+3]=PLC_CF_RRG;
        plc.wr.size=5;                                  //Count data + register
        WRPLCNACK();

        i2c_rsen=FALSE;                                 //Release I2C bus
        SETINTPLC();                                    //Configure interrupts
        WRPLCNACK();

        rc=STS_NOF;
        break;
    }

    I2CHKBUS();
    return(rc);
} //End function cfgPLC


/* Function dspTydata. Sends data from object chg to the console.
 * 
 * Returns:     none.
 *
 * Parameters:  none.
 *
*/
void dspTydata()
{
    usart_nl=FALSE;
    usart_wr_ui16((uint16_t)chg.lga);
    putch(S_CH_SPACE);
    usart_wr_ui16((uint16_t)chg.ty.ccadr);
    putch(S_CH_SPACE);
    usart_wr_ui16(chg.ty.ttime);
    putch(S_CH_SPACE);
    usart_wr_ui16((uint16_t)chg.ty.ctrl);
    putch(S_CH_SPACE);
    usart_wr_ui16((uint16_t)chg.ty.state);
    putch(S_CH_SPACE);
    usart_wr_ui16((uint16_t)chg.ty.sts);
    putch(S_CH_SPACE);
    usart_wr_ui16((uint16_t)chg.ty.cgas);
    putch(S_CH_SPACE);
    usart_wr_ui16((uint16_t)chg.ty.inv);
    putch(S_CH_SPACE);
    usart_wr_ui16((uint16_t)chg.ty.outv);
    putch(S_CH_SPACE);
    usart_wr_ui16((uint16_t)chg.ty.dcp1);
    putch(S_CH_SPACE);
    usart_wr_ui16((uint16_t)chg.ty.dcp2);
    putch(S_CH_SPACE);
    usart_wr_ui16((uint16_t)chg.ty.tp1);
    putch(S_CH_SPACE);
    usart_nl=TRUE;
    usart_wr_ui16((uint16_t)chg.ty.tp2);

    return;
} //End function dspTydata


/*Function InterruptExec.  MCU interrupt handler.  Manages all enabled
 * interrupts which should be ordered according to priority high-low:
 *  + Over-current event capture
 *  + System timer
 *  + I2C
 *  + USART
 *
 * Returns:     none
 *
 * Parameters:  none
 *
*/
void __interrupt() InterruptExec(void)
{
    static uint16_t sc=0;

    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * SYSTEM TIMER. Non-precision timed background service. ~10 mS frequency.
     * Maintains various timing objects:
     *  + i2c_tor - timer for I2C slave acknowledge (ACK).
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    if (S_AL_SYST_IE & S_AL_SYST_IF)
    {
        if (i2c_tor) --i2c_tor;                     //I2C timeout timer
        if (plc_rqt)                                //PLC request timer
        {
            if ((--plc_rqt)==0) flag_rqt=TRUE;
        }
        if (++msc==S_QY_SYST_1S)                  //One second elapsed
        {
            msc=0;
            if (++sc==S_QY_SECHOUR) sc=0;           //One hour elapsed
        }
        S_AL_SYST_IF=FALSE;
    }

    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * I2C READ HANDLER. Reads slave data in background.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    if (SSP1IE & SSP1IF) i2c_m_isrRd();

    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * USART HANDLER. Service RS232 console.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    usartISR();

    return;
} //End function InterruptExec


/*Function Initialize. Called immediately after boot by function main to 
 * complete MCU, port, and on-board peripherals. Initialization of external
 * components should be handled by function main and not in this function.
 *  + Sets system clock speed
 *  + Sets boot fault trap & tests
 *  + Validates NVM & loads to RAM
 *  + Configures pins, peripherals, interrupts
 *  + Starts system timer
 *
 * Returns:     none
 *
 * Parameters:  none
 *
*/
void Initialize(void)
{
    di();                               //No interrupts during initialization    

    //Initialize state flags and start with assumption of clean boot.
    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * INITIALIZE STATE FLAGS & STATUS CODE.
     * For code size efficiency a waterfall method is employed. The FAULT flag
     * is set and the status code cleared. This way, as initialization 
     * progresses, only the status code has to be set when a fault is 
     * encountered. At the end of this function, if the status code is clear,
     * the FAULT flag is cleared and the BOOT flag set.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    state=STE_FAULT;
    sts=STS_NOF;

    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * INITIALIZE SYSTEM CLOCK & WATCHDOG.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    OSCCON=CFG_OSC_C;                   //Configure system clock
    while (!(HFIOFL));                  //Wait for oscillator to stabilize
    WDTCON=CFG_WDT_CON;                 //Watchdog


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * BOOT & NVM PROTECTION AND MANAGEMENT.
     *
     * Recovering from CPU/firmware failures (e.g. watchdog, stack OF/UF, etc.)
     * and NVM corruption is critical for safety and field reliability. While
     * neither should happen unless there's a true failure, statistically it's 
     * possible when operating in environmental extremes or the more likely
     * existence of a firmware bug.
     * 
     * The goal is to maintain safe operation, keep the unit intact, and record
     * the failure state (in that order).
     * 
     * DATA MANAGEMENT
     * A single row (32 bytes) of NVM data is used for the storage of tunable
     * parameters (e.g. current limit, output voltage, etc.). Object nvm_cfg
     * resides in flash (NVM) and is initialized with default values. During
     * normal operation object ram_cfg contains a copy of nvm_cfg and is used
     * to update nvm_cfg. It must also be initialized with the same default
     * values as nvm_cfg.
     * 
     * DATA VALIDATION
     * NVM data is validated with a simple 16-bit checksum stored in the last
     * two bytes of nvm_cfg.
     * 
     * BOOT PROCESS
     * The CPU boot status is always checked first to ensure that any abnormal
     * condition is identified before being cleared by subsequent execution.  
     * The PCON & STATUS registers are examined for reboot due to watchdog, 
     * stack OF/UF, and possibly other conditions. If found the status object
     * sts is updated and it's expected that the main RUN loop halts normal 
     * operation.
     * 
     * The NVM checksum is checked even if a CPU fault condition has been
     * detected. It is read directly from NVM and compared to the NVM's stored
     * checksum. If they match nvm_cfg is copied to ram_cfg. This ensures 
     * that when a CPU fault is recorded it will not overwrite NVM with default
     * parameters.
     * 
     * If there's a mismatch ram_cfg retains it's default values and the status
     * code will be updated if not already set by the CPU checks.
     * 
     * The remainder of the initialization function continues unaltered and
     * when complete the main function enters the RUN loop. The converter
     * should not start if any status code other than BOOT is set. A code block 
     * in the main function loop is responsible for storing error codes, 
     * resulting from the boot process, or otherwise, in NVM.
     * 
     * Since the MCU's architecture requires the entire NVM row to be written
     * it's critical that ram_cfg contains valid data. This is why it's not
     * updated with nvm_cfg when checksum's don't match. By writing the status
     * code it also ensures that if NVM was truly corrupt it will now contain
     * valid data (provided there's not an MCU flash failure) and there's a 
     * chance of the unit returning to operation when power cycled.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    if (((S_AL_REG_PCON & S_LM_PCON_EX) != S_LM_PCON_BT) || \
            ((S_AL_REG_STATUS & S_LM_STAT_EX) != S_LM_STAT_BT))
    {
        sts=STS_MCU;
    }

    if (nvmChksum((unsigned char *) &nvm_cfg) != (unsigned int) \
            nvm_cfg.word[S_AR_FLCS_W])
    {
        SETSTS(STS_NVM);
    }
    else memcpy((void *) &ram_cfg, (void *) &nvm_cfg, S_SZ_FLASHR);


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * INITIALIZE CONTROL DATA:
     * + usart_nl is set TRUE to add new line character to console output.
     * + flag pir is cleared to ensure service of PLC interrupt.
     * + Element state in object chg.ty initialized to indicate no telemetry
     *   data has been received.
     * + The logical address (lga) in objects chg, chgv, and chgf are set to the 
     *   broadcast address to indicate that the objects do not contain valid
     *   data.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    usart_nl=TRUE;
    flag_pir=FALSE;
    flag_rqt=FALSE;
    chg.lga=PLC_AD_BRC;
    chg.ty.state=STE_INVLD;
    chg.ty.sts=STS_NOD;
    chgv.lga=PLC_AD_BRC;
    chgf.lga=PLC_AD_BRC;


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * SET PORT CONFIGURATION:
     * + Set digital latch initial states
     * + Pin in/out-put states
     * + Analog / digital
     * + Open drain (tri-state)
     * + Slew rate
     * + Logic input levels
     * + SSR must be set off since the enable method relies on switching 
     *   between the PWM & latch using PPS. The PPS register is not cleared
     *   (e.g., set to latch) for non power-up resets.
     * 
     * Important: weak pull-ups are not used (OPTION_REG WPUEN is left @ default
     * of disabled) to avoid power-up spurious switching.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    PORTA=S_IN_PORTA;
    PORTB=S_IN_PORTB;
    PORTC=S_IN_PORTC;
    //Port A
    TRISA=IO_PORTA;
    ANSELA=AD_PORTA;
    ODCONA=OD_PORTA;
    SLRCONA=SR_PORTA;
    INLVLA=IL_PORTA;
    //Port B
    TRISB=IO_PORTB;
    ANSELB=AD_PORTB;
    ODCONB=OD_PORTB;
    SLRCONB=SR_PORTB;
    INLVLB=IL_PORTB;
    //Port C
    TRISC=IO_PORTC;
    ANSELC=AD_PORTC;
    ODCONC=OD_PORTC;
    SLRCONC=SR_PORTC;
    INLVLC=IL_PORTC;


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * Initialize USART for command console.
     * 
     * This should be done at end of port / pin config since it configures PPS
     * for the USART. Console echo OFF.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    usartSetRx(ENABLE);
    usartSetEcho(FALSE);


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * Initialize MSSP for I2C communication with PLC controller. Pin and PPS 
     * configuration must be completed separately.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    i2c_m_cfg();


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * CONFIGURE FVR:
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    FVRCON=CFG_FVR_C;


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * CONFIGURE ADC:  enable and connect to first measurement source in RUN 
     * loop. It's expected that all further management of ADC is done thru 
     * function ReadVoltage.
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    ADCON0=CFG_ADC_CON_0;
    ADCON1=CFG_ADC_CON_1;
    ADCON2=CFG_ADC_CON_2;


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * CONFIGURE TIMERS:
     * SYST=system chrono timing
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    S_AL_SYST_CLKCON=CFG_SYST_CLK;
    S_AL_SYST_HLT=CFG_SYST_HLT;
    S_AL_SYST_RST=CFG_SYST_RST;
    S_AL_SYST_PR=CFG_SYST_PR;
    S_AL_SYST_CON=CFG_SYST_CON;


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * MAP PERIPHERAL PINS:
     * + MSSP
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    //MSSP
    PPS_P_I2C_D_O=PPS_M_I2C_D_O;
    PPS_P_I2C_C_O=PPS_M_I2C_C_O;
    SSPDATPPS=PPS_M_I2C_D_I;
    SSPCLKPPS=PPS_M_I2C_C_I;


    /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
     * INITIALIZATION COMPLETE:
     * + Test for fault and if none set the STATE flags accordingly
     * + Enable peripheral & global interrupts
     * + Enable peripheral interrupts serviced by ISR
     * + Start system timer
     *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    if (sts == STS_NOF) state=STE_STOP; //Console always stopped

    PEIE=ENABLE;                        //Peripheral interrupts
    ei();                               //Global interrupts
    S_AL_SYST_IE=TRUE;                  //Enable system timer interrupt
    S_AL_SYST_EN=TRUE;                  //Enable system timer

    return;
} //End function Initialize



//End of file
