/*

DESCRIPTION:    main include file for PVCRC project.

COPYRIGHT:      (c) 2025, Brian Cornell, all rights reserved.

 TARGET MCU:    PIC16F1776

HW REVISION:    241209

VERSION:        3.0

MODIFICATION HISTORY:

     9 Feb 25, BWC, version 3 adapted from charger pvc-231029-fw-v3_0.X


*/



/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * Version constants & definitions.  Build number is defined in constant 
 * VERBUILD and maintained by shell script build.sh and included in build.h.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define VERMAJOR    3
#define VERMINOR    0


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * Necessary includes.  Build.h contains the build number constant which is
 * maintained by the makefile.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#ifndef UINT_MAX
#include <limits.h>
#endif
#include "build.h"


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * Conditional compilation.  Uncomment directive to enable / disable features.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
//#define _OPT_ENA_P1                     //Enable P1 gate drive
//#define _OPT_ENA_P2                     //Enable P2 gate drive

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * System clock & Watchdog Timer
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define CFG_OSC_C               0x78            //16MHz
#define CFG_WDT_CON             0x10            //Watchdog interval=256mS, off


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * System wide constants & definitions.  Convention:  S_<TYPE>_<INFO>
 * Types:
 *  + AD=Address of a network node
 *  + AL=ALias
 *  + AR=ARray address
 *  + BL=boolean value
 *  + BP=bit position within a data object
 *  + CC=Calibration Constant
 *  + CD=command data
 *  + CF=configuration value
 *  + CH=character
 *  + DC=Duty Cycle
 *  + DF=Default value for a configurable parameter
 *  + EN=ENable bit
 *  + IN=INitializing value
 *  + LM=LiMiting operational value
 *  + OF=OFfset in an array, list, etc.
 *  + MK=bit MasK
 *  + MP=Peripheral port bit MaPping
 *  + QY=quantity (count) of something
 *  + RT=RaTio or scaling factor
 *  + RN=function return value
 *  + SL=SeLection value
 *  + SZ=size of an object or data type
 *  + TM=TiMing
 *  + VL=VoLtage
 * 
 * <INFO> is user defined and should be descriptive.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define S_QY_DC_1US             (_XTAL_FREQ / 1000000)  //Count for 1uS
#define S_TM_MOD_F              80              //Master PWM frequency, 200kHz
#define S_DC_MIN                0               //Minimum DC
#define S_DC_SP                 5               //Single pulse DC
#define S_DC_OFF                0               //No DC (effectively off)
#define S_DC_MAX                50              //Maximum allowable duty cycle
#define S_DC_RUN                S_DC_MAX        //Default running duty cycle
#define S_QY_DC_MAX             S_TM_MOD_F      //Maximum count for gate drive period
#define S_TM_DC_RUN             (((S_TM_MOD_F / S_QY_DC_1US)*2)+2)  //Time for DC capture
#define S_TM_DC_SP              (S_TM_MOD_F / S_QY_DC_1US)  //Time for single puluse
#define S_MK_BUCK_LDA           0x03            //PWM LDA bit mask
#define S_TM_DCDELAY            500             //# uS per DC increment
#define S_TM_RTC                50              //# uS to hold PLC reset high
#define S_TM_INT                1400            //Power-up delay, mS, for PLC
#define S_TM_NTMR               12              //Update NVM hours
#define S_TM_SYST               10              //System timer period in mS
#define S_TM_TEMP               5               //# seconds to measure temps
#define S_QY_SYST_1S            (S_QY_MS_S / S_TM_SYST) //System timer periods=1s
#define S_BL_TRIS_IN            1               //Pin configured as input
#define S_BL_TRIS_OUT           0               //Pin configured as output
#define S_EN_PWMLD              0x03            //PWMs to load values via PWMLD

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * VOLTAGE MEASUREMENT RATIOS
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define S_RT_VDD                4               //Vdd measurement ratio
#define S_RT_VIN                25              //Input volts measurement ratio
#define S_RT_VOUT               25              //Output volts measurement ratio
#define S_RT_TEMP               1               //Temperature measurement ratio


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * NVM CONTROL BIT MAPPED VALUES
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define S_BL_EAUTO              0x01                //AUTOSTART enabled
#define S_BL_DAUTO              (0xFF ^ S_BL_EAUTO) //AUTOSTART disabled
#define S_BL_ECLOS              0x02                //CLOSED LOOP enabled
#define S_BL_DCLOS              (0xFF ^ S_BL_ECLOS) //CLOSED LOOP disabled
#define S_BL_ECTRL              0x04                //CONTROLLED mode enabled
#define S_BL_DCTRL              (0xFF ^ S_BL_ECTRL) //STANDALONE mode enabled

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * NVM DEFAULT VALUES & LIMITS
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define S_DF_VSET               560             //56V voltage
//#define S_DF_VSET               540             //54V output
//#define S_DF_VSET               585             //60V output
#define S_DF_VSET_H             (S_DF_VSET >> S_SZ_BYTE)
#define S_DF_VSET_L             (S_DF_VSET & S_MK_U16_L)
#define S_DF_VMPP               0               //MPP disabled
//#define S_DF_VMPP               230               //24V
//#define S_DF_VMPP               240             //Default MPP voltage
#define S_DF_VMPP_H             (S_DF_VMPP >> S_SZ_BYTE)
#define S_DF_VMPP_L             (S_DF_VMPP & S_MK_U16_L)
#define S_DF_CLIM               750             //Default current limit (DAC)
#define S_DF_CLIM_H             (S_DF_CLIM >> S_SZ_BYTE)
#define S_DF_CLIM_L             (S_DF_CLIM & S_MK_U16_L)
#define S_DF_TTIM               0               //Unit total time (hours)
#define S_DF_TTIM_H             (S_DF_TTIM >> S_SZ_BYTE)
#define S_DF_TTIM_L             (S_DF_TTIM & S_MK_U16_L)
#define S_DF_TSTS               0               //Time of last fault code (hours)
#define S_DF_TSTS_H             (S_DF_TSTS >> S_SZ_BYTE)
#define S_DF_TSTS_L             (S_DF_TSTS & S_MK_U16_L)
#define S_DF_REG_STATUS         0               //Default NVM value
#define S_DF_REG_PCON           0               //Default NVM value
#define S_DF_SCOMP              1               //Default slope comp
#define S_DF_LCLADR             0xFE            //Console node address
#define S_DF_CHGADR             0x00            //Charge controller node address
#define S_DF_CONTRL             (S_BL_ECLOS)    //Control bits

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * REGISTER TESTS FOR MCU FAULT
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define S_LM_PCON_EX            0xFA            //PCON bits to examine at boot
#define S_LM_PCON_BT            0x18            //PCON expected value at boot
#define S_LM_STAT_EX            0x18            //STATUS bits to examine at boot
#define S_LM_STAT_BT            0x18            //STATUS expected value at boot

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * FAULT LIMITS, MAXIMUM OPERATING, AND RECOVERY VALUES.
 * These constants define operating limits beyond which the FAULT machine state
 * is set........
 * FLT=
 * MAX=
 * RCV=
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define S_LM_VDD_NVM            3               //Vdd > to allow NVM writes
#define S_LM_VLOW               450             //Lowest allowable voltage set value
#define S_LM_FLT_OCC            50              //Over-current count limit
#define S_LM_MAX_TMP            190             //Maximum temp (VMV)
#define S_LM_RCV_TMP            280             //Recovery temp (VMV)
#define S_LM_FLT_LOV            400             //Low output voltage
#define S_LM_HYS_LOV            50              //Low output voltage hysteresis
#define S_LM_HYS_VSTP           12000           //Charge algorithm hysteresis
#define S_LM_FLT_HOV            65              //High Vout fault (volts)

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * PERIPHERAL ALIASES.
 * 
 * Implementation specific to help with readability and portability.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define S_AL_DC_P1              PWM5DC          //Phase 1 (master) PWM DC register
#define S_AL_DC_P2              PWM6DC          //Phase 2 PWM DC register
#define S_AL_DCTMR_P1           TMR3            //Timer to measure phase 1 DC
#define S_AL_DCTMR_P2           TMR5            //Timer to measure phase 2 DC
#define S_AL_DCTEN_P1           T3GGO_nDONE     //Enable DC timer for P1
#define S_AL_DCTEN_P2           T5GGO_nDONE     //Enable DC timer for P2
#define S_AL_OCIE_P2            C2IE            //P2 over-current comparator IE
#define S_AL_OCIF_P2            C2IF            //P2 over-current comparator IF
#define S_AL_OCIE_P1            C4IE            //P1 over-current comparator IE
#define S_AL_OCIF_P1            C4IF            //P1 over-current comparator IF
#define S_AL_COGAS_P1           COG1ASD0bits.ASE    //P1 COG auto-shutdown bit
#define S_AL_COGAS_P2           COG2ASD0bits.ASE    //P2 COG auto-shutdown bit
#define S_AL_PRGGO_P1           PRG1CON0bits.GO //P1 PRG GO bit
#define S_AL_PRGGO_P2           PRG2CON0bits.GO //P2 PRG GO bit
#define S_AL_ADCCH_VDD          ADC_CH_DAC_3    //Analog channel to measure Vdd

//System chrono timer
#define S_AL_SYST_CLKCON        T8CLKCON
#define S_AL_SYST_HLT           T8HLT
#define S_AL_SYST_RST           T8RST
#define S_AL_SYST_PR            T8PR
#define S_AL_SYST_CON           T8CON
#define S_AL_SYST_IE            TMR8IE
#define S_AL_SYST_IF            TMR8IF
#define S_AL_SYST_EN            T8ON


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * Port/pin configuration: I/O, analog/digital, slew rate, etc.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
//Port I/O: 1=input, 0=output
#define IO_PORTA                0b01100011
#define IO_PORTB                0b10101101
#define IO_PORTC                0b10011011
//Port Analog/Digital: 1=analog, 0=digital
#define AD_PORTA                0b00100011
#define AD_PORTB                0b00101110
#define AD_PORTC                0b11011000
//Port weak pull-up: 1=enabled, 0=disabled
#define WP_PORTA                0b00000000
#define WP_PORTB                0b00000000
#define WP_PORTC                0b00000000
#define WP_PORTE                0b00001000
//Open drain: 1=open drain, 0=push-pull
#define OD_PORTA                0b00000000
#define OD_PORTB                0b00000000
#define OD_PORTC                0b00000000
//Slew rate: 1=limited, 0=maximum
#define SR_PORTA                0b00000000
#define SR_PORTB                0b00000000
#define SR_PORTC                0b00000000
//Input level: 1=ST, 0=TLL
#define IL_PORTA                0b00000000
#define IL_PORTB                0b00000000
#define IL_PORTC                0b00000000
//Initial latch states: all outputs logic low
#define S_IN_PORTA              0b00000000
#define S_IN_PORTB              0b00000000
#define S_IN_PORTC              0b00100000


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * PPS & DIRECT PIN MAPPING.
 * 
 * Naming convention for PPS output pin constants:
 *  + Output pin register assignment - PPS_P_XXX
 *  + Peripheral mapping assignment - PPS_M_XXX
 *      Where XXX=net name, driving function, etc.
 * 
 *      Set PPS_P_XXX to the output PPS register value (ex. RA1PPS).
 *      Set PPS_M_XXX to the PPS output peripheral constant from the 
 *      appropriate picxxxxx-map.h file.
 * 
 * Naming convention for PPS input pin constants: PPS_I_XXX
 *      Where XXX=net name, driving function, etc.
 *      Set PPS_I_XXX to the PPS input pin constant from the appropriate
 *      picxxxxx-map.h file.
 * 
 * For pins accessed directly or assigned in a peripheral's register (like ADC)
 * use the following convention:  PIN_XT_YYYYYY where
 *          X=(I)nput or (O)utput
 *          T=(A)nalog or (D)igital
 *          YYYYYY=user assigned mnemonic describing function
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
//Timing for P1 COG & PRG, master for P2.
#define PPS_P_P5P1              RA3PPS          //Output on pin 5
#define PPS_M_P5P1              PPS_PWM5_OUT    //Drive is PWM5
#define PPS_I_P1P1              PPSRA3          //PRG1 rise/fall event trigger

//Timing for P2 COG & PRG.
#define PPS_P_P6P2              RA4PPS          //Output on pin 6
#define PPS_M_P6P2              PPS_PWM6_OUT    //Drive is PWM6
#define PPS_I_P2P2              PPSRA4          //PRG2 rise/fall event trigger

//P1 gate drive & timing measurement.
#define PPS_P_C1P1              RC2PPS          //Output on pin 13
#define PPS_M_C1P1              PPS_COG1A_OUT   //Drive is COG1 A
#define PPS_I_T3P1              PPSRC2          //P1 DC measurement

//P2 gate drive & timing measurement.
#define PPS_P_CGP2              RB4PPS          //Output on pin 25
#define PPS_M_CGP2              PPS_COG2A_OUT   //Drive is COG2 A
#define PPS_I_T5P2              PPSRB4          //P2 DC measurement

//SSR gate drive
#define PPS_P_SSR               RA2PPS          //Output on pin 4
#define PPS_M_SSR               PPS_PWM9_OUT    //Drive is PWM9 

//I2C pin mapping for PLC controller: bidirectional so in/outputs are mapped
#define PPS_P_I2C_D_O           RC0PPS          //SDA (data out) maps to pin 15
#define PPS_P_I2C_C_O           RC1PPS          //SCL (clock out) maps to pin 16
#define PPS_M_I2C_D_O           0x22            //Output SDA on selected pin
#define PPS_M_I2C_C_O           0x21            //Output SCL on selected pin
#define PPS_M_I2C_D_I           PPSRC0          //I2C data in
#define PPS_M_I2C_C_I           PPSRC1          //I2C clock in

//Direct digital pin mapping
#define PIN_OD_LED              RC5             //Status LED: 1=fault, 0=run
#define PIN_OD_PRT              RA7             //PLC controller reset
#define PIN_ID_PIR              RA6             //PLC controller IRQ

//Analog pin mapping
#define PIN_IA_INV              ANLCAN16        //ADC sense - input voltage
#define PIN_IA_OUTV             ANLCAN0         //ADC sense - output voltage
#define PIN_IA_TFP1             ANLCAN4         //P1 temperature sense
#define PIN_IA_TFP2             ANLCAN13        //P2 temperature sense
#define PIN_IA_PVCS             ANLCAN15        //Current sense from PV array

//TRIS pin mapping for peripheral in/output control
#define TRIS_LED                TRISC5          //TRIS bit for status LED


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * MACROS.
 * + SETSTS sets the object sts to record the fault condition provided it does
 *   not hold an existing code.
 * + DSPSTV sends data from object chgv to the console.
 * + DSPFTD sends data from object chgf to the console.
 * + DSPOPV sends data from object chgo to the console.
 * + DSPCGA sends data from object chga to the console.
 * + DSPCRL sends data from object chgr to the console.
 * + DSPSEP displays the command separator prefix on the console.
 * + DSPPGN sends data from object chgg to the console.
 * + SETACFG assigns the proper configuration to the PLC Tx config register 
 *   based on the address (broadcast or logical).
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define SETSTS(s) if (sts == STS_NOF) sts=s
#define DSPSTV() usart_nl=FALSE; usart_wr_ui16((uint16_t)chgv.lga);\
            putch(S_CH_SPACE); usart_wr_ui16(chgv.sv.out); putch(S_CH_SPACE);\
            usart_nl=TRUE; usart_wr_ui16(chgv.sv.mpp)
#define DSPFTD() usart_nl=FALSE; usart_wr_ui16((uint16_t)chgf.lga);\
            putch(S_CH_SPACE); usart_wr_ui16(chgf.ft.time); putch(S_CH_SPACE);\
            usart_wr_ui16((uint16_t)chgf.ft.stsc); putch(S_CH_SPACE);\
            usart_wr_ui16((uint16_t)chgf.ft.sts); putch(S_CH_SPACE);\
            usart_wr_ui16((uint16_t)chgf.ft.r_sts); putch(S_CH_SPACE);\
            usart_nl=TRUE; usart_wr_ui16((uint16_t)chgf.ft.r_pcon)
#define DSPOPV() usart_nl=FALSE; usart_wr_ui16((uint16_t)chgo.lga);\
            putch(S_CH_SPACE); usart_wr_ui16(chgo.ov.inv); putch(S_CH_SPACE);\
            usart_wr_ui16(chgo.ov.outv); putch(S_CH_SPACE); usart_nl=TRUE;\
            usart_wr_ui16(chgo.ov.cdc)
#define DSPCGA() usart_nl=FALSE; usart_wr_ui16((uint16_t)chgc.lga);\
            putch(S_CH_SPACE); usart_wr_ui16(chgc.ca.vnom); putch(S_CH_SPACE);\
            usart_wr_ui16(chgc.ca.vblk); putch(S_CH_SPACE);\
            usart_wr_ui16(chgc.ca.vabs); putch(S_CH_SPACE);\
            usart_wr_ui16(chgc.ca.vflt); putch(S_CH_SPACE);usart_nl=TRUE;\
            usart_wr_ui16(chgc.ca.tabs)
#define DSPCRL() usart_nl=FALSE; usart_wr_ui16((uint16_t)chgr.lga);\
            putch(S_CH_SPACE); usart_wr_ui16((uint16_t)chgr.rl.vmaj);\
            putch(S_CH_SPACE); usart_wr_ui16((uint16_t)chgr.rl.vmin);\
            putch(S_CH_SPACE); usart_wr_ui16(chgr.rl.vbld); putch(S_CH_SPACE);\
            usart_nl=TRUE; usart_wr_ui16((uint16_t)chgr.rl.hwr)
#define DSPSEP() usart_nl=FALSE; usart_wr_str(cmdb_cpy); usart_wr_str(cmd_sep);\
            usart_nl=TRUE
#define DSPPGN() usart_nl=FALSE; usart_wr_ui16((uint16_t)chgg.lga);\
            putch(S_CH_SPACE); usart_wr_ui16((uint16_t)chgg.pg.txg);\
            putch(S_CH_SPACE);usart_nl=TRUE;usart_wr_ui16((uint16_t)chgg.pg.rxg)
#define SETACFG(co) if (cmdp1==PLC_AD_BRC) plc.co.txc.cfg=PLC_CF_RTB;\
            else plc.co.txc.cfg=PLC_CF_RTC



/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * FVR CONFIGURATION:
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define CFG_FVR_C               0xCF            //Enabled, CDAFVRx4, ADFVRx4


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * ADC & VMV CONFIGURATION.
 * 
 * Reference pic16f1776-map-v2.h for details on setting constants for VMV, 
 * correction, 5V reference.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
//ADC peripheral configuration
#define CFG_ADC_CON_0           (PIN_IA_INV | S_EN_ADC) //Default ch. & enabled
#define CFG_ADC_CON_1           0xD3            //Right, Fosc/16, Vss, FVR
#define CFG_ADC_CON_2           0x00            //No auto conversion


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * OPAMP (OPA) CONFIGURATION:
 * 2=Vout error amp
 * 3=MPP error amp
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define CFG_OPA_2_C             0x80            //Enabled
#define CFG_OPA_2_O             0x00            //Override not used
#define CFG_OPA_2_N             0x00            //OPA2IN0, pin 23
#define CFG_OPA_2_P             0x02            //DAC1_out

#define CFG_OPA_3_C             0x80            //Enabled
#define CFG_OPA_3_O             0x00            //Override not used
#define CFG_OPA_3_N             0x00            //OPA3IN0, pin 18
#define CFG_OPA_3_P             0x02            //DAC5_out


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * COMPARATOR CONFIGURATION:
 * 1=P2 slope compensation
 * 2=P2 current limit
 * 3=P1 slope compensation
 * 4=P1 current limit
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define CFG_CM_1_C0             0xAE            //Enabled, HS, ZLF, HYS, async
#define CFG_CM_1_C1             0x00            //No interrupts
#define CFG_CM_1_N              0x02            //C1IN2-, pin 24
#define CFG_CM_1_P              0x04            //PRG2_out

#define CFG_CM_2_C0             0xAE            //Enabled, HS, ZLF, HYS, async
#define CFG_CM_2_C1             0x01            //Negative edge interrupt
#define CFG_CM_2_N              0x02            //C2IN2-, pin 24
#define CFG_CM_2_P              0x06            //DAC2_out

#define CFG_CM_3_C0             0xAE            //Enabled, HS, ZLF, HYS, async
#define CFG_CM_3_C1             0x00            //No interrupts
#define CFG_CM_3_N              0x04            //C3IN4-, pin 14
#define CFG_CM_3_P              0x03            //PRG1_out

#define CFG_CM_4_C0             0xAE            //Enabled, HS, ZLF, HYS, async
#define CFG_CM_4_C1             0x01            //Negative edge interrupt
#define CFG_CM_4_N              0x04            //C4IN4-, pin 14
#define CFG_CM_4_P              0x06            //DAC2_out


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * TIMER CONFIGURATION:
 * 2=SSR period for PWM9
 * 3=P1 DC measurement
 * 5=P2 DC measurement
 * SYST=system chrono timing
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define CFG_TMR_2_CLK           0x01            //Fosc/4
#define CFG_TMR_2_CON           0x80            //Enabled, 1:1 pre & post-scale
#define CFG_TMR_2_HLT           0x20            //On sync to clk, free run
#define CFG_TMR_2_RST           0x00            //No external reset sources
#define CFG_TMR_2_PR            7               //500kHz

#define CFG_TMR_3_CON           0x41            //Fosc, 1:1 prescale, sync, on
#define CFG_TMR_3_GCON          0xD0            //Gated by pin, single pulse

#define CFG_TMR_5_CON           0x41            //Fosc, 1:1 prescale, sync, on
#define CFG_TMR_5_GCON          0xD0            //Gated by pin, single pulse

#define CFG_SYST_CLK            0x05            //MFINTOSC
#define CFG_SYST_CON            0x19            //Off, 1:2 prescale, 1:10 post
#define CFG_SYST_HLT            0xA0            //Prescale & bit sync, free run
#define CFG_SYST_RST            0x00            //No external reset sources
#define CFG_SYST_PR             250             //10mS period


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * PWM CONFIGURATION:
 * 5=P1
 * 6=P2
 * 9=SSR drive
 * 
 * Duty cycle is loaded from NVM during initialization.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define CFG_PWM_5_CON           0x80            //Enabled, active high standard
#define CFG_PWM_5_INTE          0x00            //No interrupts
#define CFG_PWM_5_CLK           0x00            //No prescale, Fosc
#define CFG_PWM_5_LDCON         0x80            //Load triggering is immediate
#define CFG_PWM_5_OFCON         0x00            //Independent run mode
#define CFG_PWM_5_PHASE         0x0000          //Phase count
#define CFG_PWM_5_PERIOD        S_TM_MOD_F      //Period (frequency)
#define CFG_PWM_5_OFFSET        ((S_TM_MOD_F/2)+(S_TM_MOD_F%2)) //PWM6 phase
#define CFG_PWM_5_TIMER         0x0000          //Starting value for timer

#define CFG_PWM_6_CON           0x80            //Enabled, active high standard
#define CFG_PWM_6_INTE          0x00            //No interrupts
#define CFG_PWM_6_CLK           0x00            //No prescale, Fosc
#define CFG_PWM_6_LDCON         0x80            //Load triggering is immediate
#define CFG_PWM_6_OFCON         0x20            //Slave to PWM5 with sync start
#define CFG_PWM_6_PHASE         0x0000          //Phase count
#define CFG_PWM_6_PERIOD        S_TM_MOD_F      //Period (frequency)
#define CFG_PWM_6_OFFSET        0x0000          //Offset count
#define CFG_PWM_6_TIMER         0x0000          //Starting value for timer

#define CFG_PWM_9_CON           0x80            //Enabled
#define CFG_PWM_9_DC            15              //Duty cycle, ~ 1uS
#define CFG_PWM_9_DCH           (CFG_PWM_9_DC >> 2)
#define CFG_PWM_9_DCL           (CFG_PWM_9_DC << 6)


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * COG CONFIGURATION:
 * 1=P1
 * 2=P2
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define CFG_COG_1_CON0          0xCC            //Enable, load, Fosc, HB
#define CFG_COG_1_CON1          0x00            //Clk delay chain, active high
//Rising events
#define CFG_COG_1_RIS0          0x00            //None
#define CFG_COG_1_RIS1          0x02            //PWM5
#define CFG_COG_1_RSIM0         0x00            //None
#define CFG_COG_1_RSIM1         0x02            //PWM5
//Falling events
#define CFG_COG_1_FIS0_C        0x18            //Closed loop CM3, CM4
#define CFG_COG_1_FIS0_O        0x10            //Open loop CM4
#define CFG_COG_1_FIS1          0x02            //PWM5
#define CFG_COG_1_FSIM0         0x10            //Hi->low for CM4, low for CM3
#define CFG_COG_1_FSIM1         0x02            //Hi>low for PWM5
//Auto shutdown
#define CFG_COG_1_ASD0          0x80            //Shutdown
#define CFG_COG_1_ASD1          0x00            //None
//Delays
#define CFG_COG_1_DBR           0               //Rising dead band
#define CFG_COG_1_DBF           0               //Falling dead band
#define CFG_COG_1_BLKR          0               //Rising edge blanking
#define CFG_COG_1_BLKF          0               //Falling edge blanking
#define CFG_COG_1_PHR           0               //Rising event phase delay
#define CFG_COG_1_PHF           0               //Falling event phase delay

#define CFG_COG_2_CON0          0xCC            //Enable, load, Fosc, HB
#define CFG_COG_2_CON1          0x00            //Clk delay chain, active high
//Rising events
#define CFG_COG_2_RIS0          0x00            //None
#define CFG_COG_2_RIS1          0x04            //PWM6
#define CFG_COG_2_RSIM0         0x00            //None
#define CFG_COG_2_RSIM1         0x04            //PWM6
//Falling events
#define CFG_COG_2_FIS0_C        0x06            //Closed loop CM1, CM2
#define CFG_COG_2_FIS0_O        0x04            //Open loop CM2
#define CFG_COG_2_FIS1          0x04            //PWM6
#define CFG_COG_2_FSIM0         0x04            //Hi->low for CM2, low for CM1
#define CFG_COG_2_FSIM1         0x04            //Hi>low for PWM6
//Auto shutdown
#define CFG_COG_2_ASD0          0x80            //Shutdown
#define CFG_COG_2_ASD1          0x00            //None
//Delays
#define CFG_COG_2_DBR           0               //Rising dead band
#define CFG_COG_2_DBF           0               //Falling dead band
#define CFG_COG_2_BLKR          0               //Rising edge blanking
#define CFG_COG_2_BLKF          0               //Falling edge blanking
#define CFG_COG_2_PHR           0               //Rising event phase delay
#define CFG_COG_2_PHF           0               //Falling event phase delay


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * PRG CONFIGURATION:
 * 1=P1 slope comp
 * 2=P2 slope comp
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define CFG_PRG_1_CON0          0xB0            //Enabled, edge, slope, off
#define CFG_PRG_1_CON1          0x02            //Fall low, rise high
#define CFG_PRG_1_INS           0x00            //OPA1OUT/PRG1IN0, pin 3
#define CFG_PRG_1_CON2          0x03            //Slope=350mV/uS
#define CFG_PRG_1_RTSS          0x07            //PRG1R
#define CFG_PRG_1_FTSS          0x07            //PRG1F

#define CFG_PRG_2_CON0          0xB0            //Enabled, edge, slope, off
#define CFG_PRG_2_CON1          0x02            //Fall low, rise high
#define CFG_PRG_2_INS           0x01            //OPA1OUT/PRG1IN0, pin 3
#define CFG_PRG_2_CON2          0x03            //Slope=350mV/uS
#define CFG_PRG_2_RTSS          0x07            //PRG1R
#define CFG_PRG_2_FTSS          0x07            //PRG1F


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * CLC CONFIGURATION:
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define CFG_CLC_1_CON
#define CFG_CLC_1_POL
#define CFG_CLC_1_SEL_0
#define CFG_CLC_1_SEL_1
#define CFG_CLC_1_SEL_2
#define CFG_CLC_1_SEL_3
#define CFG_CLC_1_GLS_0
#define CFG_CLC_1_GLS_1
#define CFG_CLC_1_GLS_2
#define CFG_CLC_1_GLS_3


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 *                          Print & Format Constants
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * OBJECT DEFINITIONS.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
//Console command array
const char *CmdList[] = {
    "da",                                   //Display charge algorithm values
    "dt",                                   //Display telemetry data
    "do",                                   //Display operating voltages
    "df",                                   //Display NVM fault data
    "dv",                                   //Display set voltages
    "dr",                                   //Display FW/HW release numbers
    "dp",                                   //Display PV power
    "dg",                                   //Display PLC Tx & Rx gain values
    "ga",                                   //Get chg algorithm values <addr>
    "gf",                                   //Get NVM fault data <addr>
    "gg",                                   //Get PLC Tx & Rx gain values
    "gl",                                   //Get PLC register <register #>
    "go",                                   //Get operating voltages <addr>
    "gp",                                   //Get PLC MAC address
    "gr",                                   //Get FW/HW release numbers  <addr>
    "gs",                                   //Get system status
    "gt",                                   //Get telemetry <addr>
    "gv",                                   //Get set voltages <addr>
    "iv",                                   //Incr/decr Vout <addr, 0/1>
    "rz",                                   //Read firmware release
    "sa",                                   //Set chg alg <addr, comma sep val>
    "sc",                                   //Set converter off/on <addr, 0/1>
    "se",                                   //Set operating mode <addr, 0/1/2/4>
    "sg",                                   //Set PLC Tx & Rx gain values
    "sl",                                   //Set chg ctrl addr <addr> <addr>
    "sm",                                   //Set MPP voltage <addr, value>
    "so",                                   //Set output voltage <addr, value>
    "sp",                                   //Set PLC logical addr <MAC, addr>
    "xc",                                   //Reset charger <addr>
    "xd",                                   //Restore NVM defaults <addr>
    "xf",                                   //Clear NVM fault data <addr>
    "xp",                                   //Ping PLC <address>
    "xr",                                   //Reset system
    "xs"                                    //Calibrate current sense
};

#define CMDSIZSTR   2                       //Size of command string

//Array location of commands
#define NUMOFCMDS   sizeof(CmdList) / sizeof(const char *)
#define CMDDSPCGA           0               //Display charge algorithm values
#define CMDDSPTYD           1               //Display telemetry data
#define CMDDSPOPV           2               //Display operating voltages
#define CMDDSPFTD           3               //Display NVM fault data
#define CMDDSPSTV           4               //Display set voltages
#define CMDDSPREL           5               //Display FW/HW release numbers
#define CMDDSPPVP           6               //Display PV power
#define CMDDSPPGN           7               //Display PLC Tx & Rx gain values
#define CMDGETCGA           8               //Get chg algorithm values <addr>
#define CMDGETFTD           9               //Get NVM fault data <addr>
#define CMDGETPGN           10              //Get PLC Tx & Rx gain values
#define CMDGETPLR           11              //Get PLC register <register #>
#define CMDGETOPV           12              //Get operating voltages <addr>
#define CMDGETPMC           13              //Get PLC MAC address
#define CMDGETREL           14              //Get FW/HW release numbers <addr>
#define CMDGETSTS           15              //Get system status
#define CMDGETELM           16              //Get telemetry <addr>
#define CMDGETSTV           17              //Get set voltages <addr>
#define CMDINCVOI           18              //Incr/decr Vout <addr, 0/1>
#define CMDREDREL           19              //Read firmware release
#define CMDSETCGA           20              //Set chg alg <addr, values>
#define CMDSETCNV           21              //Set converter <addr, 0/1>
#define CMDSETMOD           22              //Set operating mode <addr, value>
#define CMDSETPGN           23              //Set PLC Tx & Rx gain values
#define CMDSETCCA           24              //Set chg ctrl addr <addr> <addr>
#define CMDSETMPV           25              //Set MPP voltage <addr, value>
#define CMDSETOPV           26              //Set output voltage <addr, value>
#define CMDSETPLA           27              //Set PLC logical addr <MAC, addr>
#define CMDXRTCHG           28              //Reset charger <addr>
#define CMDXNVDFL           29              //Restore NVM defaults <addr>
#define CMDXFTCLR           30              //Clear NVM fault data <addr>
#define CMDXPGPLC           31              //Ping PLC <addr>
#define CMDXRTMCU           32              //Reset system
#define CMDXPVCCS           33              //Calibrate current sense


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * STATE MANAGEMENT AND STATUS CODES.
 * 
 * STATE MANAGEMENT refers to identifying the operational state of the unit.
 * A bit-mapped byte is used instead of bit flags as a compromise for code size
 * & efficiency for use in bit operations and when multiple status bits must be
 * operated on simultaneously. Only one bit can be set in the machine state 
 * object 'state', and the possible values are defined by the STE_XXXXX 
 * constants below. Constants for combinations of machine state are also defined
 * for use in firmware control.
 * 
 * The invalid telemetry state is stored in the charger telemetry structure to
 * indicate that it does not contain valid data.
 * 
 * The constants STS_XXXX represent the fault status codes used throughout the 
 * control program and recorded in NVM (diagnostics). Status constants for 
 * the charger telemetry object start at the maximum value for the data size and
 * decrement.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define STE_BOOT            0x01            //Initialization & calibration
#define STE_RSTRT           0x02            //Restart converter
#define STE_STOP            0x04            //Converter off: typically console work
#define STE_IDLE            0x08            //Converter off d/t low Vin (MPP only)
#define STE_RUN             0x10            //Converter on 
#define STE_RCVRF           0x20            //Recoverable fault, code in NVM
#define STE_FAULT           0x40            //Unrecoverable fault, code in NVM
#define STE_INVLD           0xFF            //Invalid telemetry data

//Combination state tests.
#define STE_BTRRT           (STE_BOOT | STE_RSTRT)
#define STE_RNFT            (STE_RUN | STE_FAULT)
#define STE_BRS             (STE_BOOT | STE_RSTRT | STE_STOP)
#define STE_RCRIE           (STE_RCVRF | STE_IDLE)
#define STE_RCRFT           (STE_RCVRF | STE_FAULT)
#define STE_BSIR            (STE_BOOT | STE_STOP | STE_IDLE | STE_RUN)
#define STE_BTRRN           (STE_BOOT | STE_RUN)

/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * STATUS CODES.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define STS_NOF             0               //No fault
#define STS_MCU             1               //MCU fault (stack, watchdog, etc.)
#define STS_NVM             2               //Invalid NVM
#define STS_OCL             3               //Over-current limit exceeded
#define STS_HOV             4               //High output voltage
#define STS_NAK             5               //No response from PLC controller
#define STS_TSZ             253             //Received telemetry size mismatch
#define STS_TRA             254             //Telemetry address mismatch
#define STS_NOD             255             //No data requested since boot


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * Flash program memory definitions.  For non-volatile storage of configurable
 * parameters.
 * 
 * + Word values must be stored in little endian (L|H) format.
 * + Row offset address for data must match assignment below.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
#define FLASH_AR_VSET_L     0               //Output volts low byte
#define FLASH_AR_VSET_H     1               //Output volts high byte
#define FLASH_AR_VSET       (FLASH_AR_VSET_L / sizeof(uint16_t))
#define FLASH_AR_VMPP_L     2               //Input volts low byte
#define FLASH_AR_VMPP_H     3               //Input volts high byte
#define FLASH_AR_VMPP       (FLASH_AR_VMPP_L / sizeof(uint16_t))
#define FLASH_AR_CLIM_L     4               //Current limit low byte
#define FLASH_AR_CLIM_H     5               //Current limit high byte
#define FLASH_AR_CLIM       (FLASH_AR_CLIM_L / sizeof(uint16_t))
#define FLASH_AR_TTIM_L     6               //Unit total time low byte
#define FLASH_AR_TTIM_H     7               //Unit total time high byte
#define FLASH_AR_TTIM       (FLASH_AR_TTIM_L / sizeof(uint16_t))
#define FLASH_AR_TSTS_L     8               //Time of last fault code low byte
#define FLASH_AR_TSTS_H     9               //Time of last fault code high byte
#define FLASH_AR_TSTS       (FLASH_AR_TSTS_L / sizeof(uint16_t))
#define FLASH_AR_STS        10              //Last fault code
#define FLASH_AR_STATUS     11              //STATUS register
#define FLASH_AR_PCON       12              //PCON register
#define FLASH_AR_DC         13              //Max or open loop duty cycle
#define FLASH_AR_SCOMP      14              //PRG slope compensation
#define FLASH_AR_PLCADR     15              //Console node address
#define FLASH_AR_CHGADR     16              //Charge controller PLC address
#define FLASH_AR_CONTRL     17              //Control bits (ref. NVM constants)

//Default data to initialize row with. Positions 30 & 31 reserved for checksum.
#define S_DF_FLASH_0        S_DF_VSET_L
#define S_DF_FLASH_1        S_DF_VSET_H
#define S_DF_FLASH_2        S_DF_VMPP_L
#define S_DF_FLASH_3        S_DF_VMPP_H
#define S_DF_FLASH_4        S_DF_CLIM_L
#define S_DF_FLASH_5        S_DF_CLIM_H
#define S_DF_FLASH_6        S_DF_TTIM_L
#define S_DF_FLASH_7        S_DF_TTIM_H
#define S_DF_FLASH_8        S_DF_TSTS_L
#define S_DF_FLASH_9        S_DF_TSTS_H
#define S_DF_FLASH_10       STS_NOF
#define S_DF_FLASH_11       S_DF_REG_STATUS
#define S_DF_FLASH_12       S_DF_REG_PCON
#define S_DF_FLASH_13       S_DC_RUN
#define S_DF_FLASH_14       S_DF_SCOMP
#define S_DF_FLASH_15       S_DF_LCLADR
#define S_DF_FLASH_16       S_DF_CHGADR
#define S_DF_FLASH_17       S_DF_CONTRL
#define S_DF_FLASH_18       0
#define S_DF_FLASH_19       0
#define S_DF_FLASH_20       0
#define S_DF_FLASH_21       0
#define S_DF_FLASH_22       0
#define S_DF_FLASH_23       0
#define S_DF_FLASH_24       0
#define S_DF_FLASH_25       0
#define S_DF_FLASH_26       0
#define S_DF_FLASH_27       0
#define S_DF_FLASH_28       0
#define S_DF_FLASH_29       0

//Calculate default configuration row checksum value
#define S_DF_CFG_CS         (S_DF_FLASH_0+S_DF_FLASH_1+S_DF_FLASH_2+\
                                S_DF_FLASH_3+S_DF_FLASH_4+S_DF_FLASH_5+\
                                S_DF_FLASH_6+S_DF_FLASH_7+S_DF_FLASH_8+\
                                S_DF_FLASH_9+S_DF_FLASH_10+S_DF_FLASH_11+\
                                S_DF_FLASH_12+S_DF_FLASH_13+S_DF_FLASH_14+\
                                S_DF_FLASH_15+S_DF_FLASH_16+S_DF_FLASH_17+\
                                S_DF_FLASH_18+S_DF_FLASH_19+S_DF_FLASH_20+\
                                S_DF_FLASH_21+S_DF_FLASH_22+S_DF_FLASH_23+\
                                S_DF_FLASH_24+S_DF_FLASH_25+S_DF_FLASH_26+\
                                S_DF_FLASH_27+S_DF_FLASH_28+S_DF_FLASH_29)
#if(S_DF_CFG_CS < 256)
#define S_DF_FLASH_30       S_DF_CFG_CS
#define S_DF_FLASH_31       0
#else
#define S_DF_FLASH_30       (S_DF_CFG_CS & 0x00FF)
#define S_DF_FLASH_31       (S_DF_CFG_CS >> 8)
#endif


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * Function declarations.  See source file for explanation of each function.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void main(void);
uint8_t cfgPLC(void);
void dspTydata(void);
void __interrupt() InterruptExec(void);
void Initialize(void);


/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * Preprocessor checks.  Maintain at end of file.
 *++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/



//End of file