
#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/cpufunc.h>
#include <stdlib.h>
#include <string.h>
#include "main.h"
#include "twi.h"
#include "usart.h"
#include "tca9535.h"
#include "tlc59116.h"

#define PLAYER1 1
#define PLAYER2 0
#define MAX_LEDS 16
#define N_POTS 14
#define N_POTS_SIDE 6
#define IDLE_INPUT 127
#define ONE_SECOND 1000
#define TWO_MINUTES 120000
#define DEFAULT_STONES 4

#define AWAKE 0
#define ASLEEP 1

#define IDLE 0x00
#define STANDBY 0x01
#define POWERDOWN 0x02

#define MAX_BRIGHTNESS 0x20
#define MIN_BRIGHTNESS 0x04

#define MANCALA0 13
#define MANCALA1 6

#define SW0 (1<<2)
#define SW1 (1<<3)
#define SW2 (1<<4)
#define SW3 (1<<5)

#define SEED 0xA632;
#define DISABLED 0
#define RANDOM 1

#define DIGITS_RNG 3
#define N_BLINKS 3
#define DEBUGLED (1<<7)

static uint8_t wrongMsg[] = "WRONG TURN!\n\r";
static uint8_t errorMsg[] = "ERROR!\n\r";
static uint8_t emptyMsg[] = "EMPTY!\n\r";
static uint8_t successMsg[] = "OK!\n\r";

volatile uint16_t lfsr = SEED;
uint8_t computerMode;
volatile uint8_t randN = 0x00;

volatile uint8_t usartRingBuffer[MAX_ITEMS];
volatile uint8_t head = 0;
volatile uint8_t tail = 0;
volatile uint8_t items = 0;

FUSES = {
	.WDTCFG = 0x00, // WDTCFG {PERIOD=OFF, WINDOW=OFF}
	.BODCFG = 0x00, // BODCFG {SLEEP=DIS, ACTIVE=DIS, SAMPFREQ=1KHZ, LVL=BODLEVEL0}
	.OSCCFG = 0x7E, // OSCCFG {FREQSEL=20MHZ, OSCLOCK=CLEAR}
	.SYSCFG0 = 0xFA, // SYSCFG0 {EESAVE=CLEAR, RSTPINCFG=RST, TOUTDIS=SET, CRCSRC=NOCRC}
	.SYSCFG1 = 0xFF, // SYSCFG1 {SUT=64MS}
	.APPEND = 0x00, // APPEND {APPEND=User range:  0x0 - 0xFF}
	.BOOTEND = 0x00, // BOOTEND {BOOTEND=User range:  0x0 - 0xFF}
};

LOCKBITS = 0xC5; // {LB=NOLOCK}

uint8_t ledBrightness[MAX_LEDS] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
uint8_t gameTable[N_POTS] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0};
volatile uint8_t player = PLAYER1;
volatile uint8_t selected = IDLE_INPUT;
volatile uint8_t brightnessLevel = 0x0E;
volatile uint8_t sleep = AWAKE;

uint16_t in = IDLE_INPUT;
uint16_t lastInputVector = 0;
uint32_t idleTick = 0;
uint16_t peripheralTick = 0;

static uint8_t convert[10] = {0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39};


uint8_t lfsr_fibonacci(uint16_t *lfsr){
    *lfsr = (*lfsr>>1) | ((((*lfsr) ^ (*lfsr>>2) ^ (*lfsr>>3) ^ (*lfsr>>10))&0b1)<<15);
    return (uint8_t)((*lfsr>>15)&0b1);
}

uint8_t rng(uint16_t *lfsr, uint8_t digits){
    uint8_t rand = 0;
    for(uint8_t i=0; i<digits; i++){
        rand = rand << 1;
        rand |= lfsr_fibonacci(lfsr);
    }
    rand = rand % 6;
    return rand;
}

uint8_t ascii2byte(uint8_t ascii){
    uint8_t byte = 0x10;
    switch(ascii){
        case '0':
            byte = 0;
            break;
        case '1':
            byte = 1;
            break;
        case '2':
            byte = 2;
            break;
        case '3':
            byte = 3;
            break;
        case '4':
            byte = 4;
            break;
        case '5':
            byte = 5;
            break;
        case '6':
            byte = 6;
            break;
        case '7':
            byte = 7;
            break;
        case '8':
            byte = 8;
            break;
        case '9':
            byte = 9;
            break; 
        case 'A':
            byte = 10;
            break; 
        case 'B':
            byte = 11;
            break;
        case 'C':
            byte = 12;
            break; 
        case 'D':
            byte = 13;
            break; 
        case 'E':
            byte = 14;
            break; 
        case 'F':
            byte = 15;
            break;    
    }
    return byte;
}

void byte2ascii(uint8_t byte, uint8_t *ascii){
    uint8_t tensPlace = (byte/10) % 10;
    uint8_t onesPlace = byte%10;
    *ascii = convert[tensPlace];
    *(ascii+1) = convert[onesPlace];
}

void refreshLEDs(uint8_t brightness, uint8_t *arr){
    for(uint8_t i=0; i<N_POTS; i++)
    {
        if(gameTable[i]){
            *(arr+i) = brightness;
        }
        else{
            *(arr+i) = 0;
        }
    }   
    if(player==PLAYER2){
        *(arr+14) = brightnessLevel;
        *(arr+15) = 0;
    }
    else if(player==PLAYER1){
        *(arr+14) = 0;
        *(arr+15) = brightnessLevel;            
    }
    for(uint8_t j=0; j<MAX_LEDS; j++){
        setLED(j,*(arr+j));
    }
}

void initGameState(uint8_t *game, uint8_t *leds, uint8_t bright){
    for(uint8_t i=0; i<N_POTS; i++)
    {
        *(game+i) = DEFAULT_STONES;
    }      
    *(game+MANCALA0) = 0;
    *(game+MANCALA1) = 0;
    for(uint8_t j=0; j<N_POTS; j++)
    {
        if(*(game+j)){
            *(leds+j) = bright;
        }
        else{
            *(leds+j) = 0;
        }
    }
    player = PLAYER1;
    *(leds+14) = bright;
    *(leds+15) = 0;
    refreshLEDs(bright, leds);
}

void wait(uint16_t time_ms, uint16_t *tick){
    *tick = 0;
    while(*tick < time_ms);
    return;
}

uint8_t isEmpty(uint8_t *arr, uint8_t len){
    for(uint8_t i=0; i<len; i++){
        if(*(arr+i)){
            return 0;
        }
    }
    return 1;
}

uint8_t sum(uint8_t *arr, uint8_t len){
    uint8_t sum = 0;
    for(uint8_t i=0; i<len; i++){
        sum = sum + *(arr+i);
    }
    return sum;
}

void gameOver(uint8_t score0, uint8_t score1){
    if(score1 > score0){
        while(1){
            setLEDs(0);
            wait(1000, &peripheralTick);
            setLED(14, brightnessLevel);
            wait(1000, &peripheralTick);
        }
    }
    else if(score0 > score1){
        while(1){
            setLEDs(0);
            wait(1000, &peripheralTick);
            setLED(15, brightnessLevel);
            wait(1000, &peripheralTick);
        }
    }
    else if(score0 == score1){
        while(1){
            setLEDs(0);
            wait(1000, &peripheralTick);
            setLED(14, brightnessLevel);
            setLED(15, brightnessLevel);
            wait(1000, &peripheralTick);
        }
    }
}

void initInterrupts(void){
    SREG |= (1<<7);
    initUSARTInterrupt();
}

void initTimerB0Interrupt1ms(void){
    TCA0.SINGLE.CTRLA = (0x4<<1) | 0x01;
    TCB0.CTRLA = (0x02<<1) | 0x01;
    TCB0.CTRLB = 0x00;
    TCB0.INTCTRL = 0x01;
    TCB0.CCMP = 125;
}

void initI2CDevices(void){
    TWI_Init(TWI_BAUD);
    resetTLC59116();
    initTLC59116();
    initTCA9535();
}

void portInit(void){
    PORTA.DIR = 0b10000000;
    PORTB.DIR = 0b00000101;
    PORTA.PIN1CTRL = 0x05;
}

uint16_t readButtons(uint16_t *last){
    uint16_t buttonVector = readTCA9535Inputs();
    if(*last == buttonVector){
        return IDLE_INPUT;
    }
    *last = buttonVector;
    for(uint8_t i=0; i<MAX_LEDS; i++){
        if(buttonVector&0b1){
            return i;
        }
        buttonVector = buttonVector>>1;
    }
    return IDLE_INPUT;
}

void makeMove(uint8_t start, uint8_t *game){
    uint8_t complimentaryPots1[6] = {12,11,10,9,8,7};
    uint8_t complimentaryPots2[6] = {5,4,3,2,1,0};
    uint8_t skip = 0;
    uint8_t addr = 0;
    for(uint8_t i = (start); i<(start+*(game+start)); i++){
        if(player==PLAYER2){
            if(i%N_POTS == 13){
                skip++;
            }
        }
        else if(player==PLAYER1){   
            if(i%N_POTS == 6){
                skip++;
            }            
        }
        addr = (i+skip)%N_POTS;
        *(game+addr) = *(game+addr) + 1;
        setLEDs(0);
        wait(250, &peripheralTick);
        setLED(addr,brightnessLevel);
        wait(250, &peripheralTick);
    } 
    if(player==PLAYER2){
        if(!(addr == 6)){
            player = player ^ 1;
            if(((*(game+addr)==1)) && ((addr >= 0) && (addr <= 5))){
                *(game+MANCALA1) = *(game+MANCALA1) + *(game+complimentaryPots1[addr]);
                *(game+MANCALA1) = *(game+MANCALA1) + 1;
                *(game+addr) = 0;
                if(*(game+complimentaryPots1[addr])){
                    for(uint8_t i=0; i<3; i++){
                        setLEDs(0);
                        wait(100, &peripheralTick);
                        setLED(addr,brightnessLevel);
                        setLED(complimentaryPots1[addr],brightnessLevel);
                        wait(100, &peripheralTick);
                    }                 
                }
                else{
                    for(uint8_t i=0; i<3; i++){
                        setLEDs(0);
                        wait(100, &peripheralTick);
                        setLED(addr,brightnessLevel);
                        wait(100, &peripheralTick);
                    }                      
                }
                *(game+complimentaryPots1[addr]) = 0;
            }
        }
        else{
             for(uint8_t i=0; i<3; i++){
                setLEDs(0);
                wait(100, &peripheralTick);
                setLED(6, brightnessLevel);
                setLED(14, brightnessLevel);
                wait(100, &peripheralTick);
            }           
        }
    }
    else if(player==PLAYER1){   
        if(!(addr == 13)){
            player = player ^ 1;
            if(((*(game+addr)==1)) && ((addr >= 7) && (addr <= 12))){
                *(game+MANCALA0) = *(game+MANCALA0) + *(game+complimentaryPots2[((addr-1)%6)]);
                *(game+MANCALA0) = *(game+MANCALA0) + 1;
                *(game+addr) = 0;
                if(*(game+complimentaryPots2[(addr-1)%6])){
                    for(uint8_t i=0; i<N_BLINKS; i++){
                        setLEDs(0);
                        wait(100, &peripheralTick);
                        setLED(addr,brightnessLevel);
                        setLED(complimentaryPots2[(addr-1)%6],brightnessLevel);
                        wait(100, &peripheralTick);
                    }                
                }
                else{
                    for(uint8_t i=0; i<N_BLINKS; i++){
                        setLEDs(0);
                        wait(100, &peripheralTick);
                        setLED(addr,brightnessLevel);
                        wait(100, &peripheralTick);
                    }                      
                }
                *(game+complimentaryPots2[(addr-1)%6]) = 0;                
            }            
        }
        else{
            for(uint8_t i=0; i<3; i++){
                setLEDs(0);
                wait(100, &peripheralTick);
                setLED(13,brightnessLevel);
                setLED(15,brightnessLevel);
                wait(100, &peripheralTick);
            }
        }
    }   
    *(game+start) = 0;    
    if(isEmpty(game, N_POTS_SIDE)){
        for(uint8_t i=0; i<6; i++){
            if(*((game+i)+7)){
                setLED(i+7,brightnessLevel);
            }
        }        
        wait(3000, &peripheralTick);
        *(game+MANCALA1) = *(game+MANCALA1) + sum((game+7), N_POTS_SIDE);
        gameOver(*(game+MANCALA0),*(game+MANCALA1));
    }
    else if(isEmpty((game+7), N_POTS_SIDE)){
        for(uint8_t i=0; i<6; i++){
            if(*(game+i)){
                setLED(i,brightnessLevel);
            }
        }
        wait(3000, &peripheralTick);
        *(game+MANCALA0) = *(game+MANCALA0) + sum(game, N_POTS_SIDE);
        gameOver(*(game+MANCALA0),*(game+MANCALA1));
    }    
    wait(1000, &peripheralTick);
    refreshLEDs(brightnessLevel, ledBrightness);
}

void howMany(uint8_t pot, uint8_t *game){
    setLEDs(0);
    for(uint8_t i=0; i<*(game+pot); i++){
        wait(250, &peripheralTick);
        setLED(pot, brightnessLevel);
        wait(250, &peripheralTick);       
        setLEDs(0);
    }
    wait(500, &peripheralTick);        
    refreshLEDs(brightnessLevel, ledBrightness);
}

void brightnessAdjust(uint8_t *brightnessArr, uint8_t brightness){
    for(uint8_t i=0; i<MAX_LEDS; i++){
        if(*(brightnessArr+i))
        {
            *(brightnessArr+i) = brightness;
        }
    }
    refreshLEDs(brightness,brightnessArr);
}

void mancala(void){
    PORTA.PIN1CTRL = 0x00;
    if((computerMode == RANDOM) && (player==PLAYER2)){
        selected = IDLE_INPUT;
        wait(500, &peripheralTick);
        while(!(gameTable[randN]));
        makeMove(randN,gameTable);
    } 
    else{
        idleTick=0;
        PORTA.PIN1CTRL = 0x05;
        while(in == IDLE_INPUT){
            if(sleep == ASLEEP){
                SLPCTRL.CTRLA = (POWERDOWN<<1) | 0x01;
                asm("sleep");
            }
        }
        idleTick=0;
        uint8_t activeButton = in;
        in = IDLE_INPUT;
        PORTA.PIN1CTRL = 0x00;
        if(activeButton == 14){
            if(brightnessLevel<MAX_BRIGHTNESS){
                brightnessLevel = brightnessLevel + 1;
            }
            brightnessAdjust(ledBrightness, brightnessLevel);
        }
        else if(activeButton == 15){
            if(brightnessLevel>MIN_BRIGHTNESS){
                brightnessLevel = brightnessLevel - 1;              
            }          
            brightnessAdjust(ledBrightness, brightnessLevel);
        }
        else if((activeButton == 6) || (activeButton == 13)){
            if(gameTable[activeButton]){
                howMany(activeButton, gameTable);
            }
            selected = IDLE_INPUT;
        }
        else if(activeButton == selected){
            if(((activeButton >= 0) && (activeButton <= 6)) && (player==PLAYER2)){
                makeMove(activeButton, gameTable);
                selected = IDLE_INPUT;
            }
            else if(((activeButton >= 7) && (activeButton <= 12)) && (player==PLAYER1)){
                makeMove(activeButton, gameTable);
                selected = IDLE_INPUT;
            }       
            else{
                if(gameTable[activeButton]){
                    howMany(activeButton, gameTable);
                }                
            }
        }           
        else{
            selected = activeButton;
            if(gameTable[activeButton]){
                howMany(activeButton, gameTable);
            }
        }
    }
}

void lampTest(uint8_t brightness){
    setLEDs(0);
    for(uint8_t i=0; i<MAX_LEDS; i++){
        setLED(i,brightness);
        if(i>0){
            setLED(i-1,0);
        }
        wait(100, &peripheralTick);
    }
    wait(250, &peripheralTick);
    setLEDs(0);
    wait(250, &peripheralTick);
}

int main(void) {
    ccp_write_io((uint8_t *) &(CLKCTRL.MCLKCTRLB), 0x13);
    portInit();
    initRingBuffer(usartRingBuffer);
    initUSARTAsynchronous(BAUDRATE_VALUE);
    initI2CDevices();
    initTimerB0Interrupt1ms();
    initInterrupts();
    lampTest(brightnessLevel);
    sleep = AWAKE;
    if(PORTA.IN & SW0){
        computerMode = RANDOM;
        PORTA.OUT |= DEBUGLED;
    }
    else{
        computerMode = DISABLED;
        PORTA.OUT = 0;
    }
    initGameState(gameTable, ledBrightness, brightnessLevel);
    while(1){
        mancala();
    }
}


ISR(PORTA_PORT_vect){
    if(PORTA.INTFLAGS & (1<<1)){
        if(sleep == ASLEEP){
            if(readButtons(&lastInputVector) == IDLE_INPUT){
                sleep = AWAKE;
                refreshLEDs(brightnessLevel, ledBrightness);
                if(computerMode == RANDOM){
                    PORTA.OUT |= DEBUGLED;
                }
                else{
                    PORTA.OUT = 0;
                }
            }
        }
        else{
            in = readButtons(&lastInputVector);
        }
        PORTA.INTFLAGS |= (1<<1);
        idleTick=0;
    }
}

ISR(USART0_RXC_vect){
    if(USART0.STATUS & (1<<7)){
        uint8_t byteIn = readUSART();
        insert(byteIn, &items, &tail, usartRingBuffer);
        if(sleep == ASLEEP){
            sleep = AWAKE;
            refreshLEDs(brightnessLevel, ledBrightness);
            if(computerMode == RANDOM){
                PORTA.OUT |= DEBUGLED;
            }
            else{
                PORTA.OUT = 0;
            }    
        }
        idleTick=0;
    }
}

ISR(TCB0_INT_vect){
    if(TCB0.INTFLAGS & 0x01){
        peripheralTick++;
        idleTick++;
        randN = 0;
        randN = rng(&lfsr, DIGITS_RNG);
        if(items){
            uint8_t poppedByte = pop(&items,&head,usartRingBuffer);
            if(poppedByte == '?'){
                uint8_t *gameState = (uint8_t*)malloc((N_POTS+1) * sizeof(uint8_t));    
                memcpy(gameState, gameTable, (N_POTS * sizeof(uint8_t)));
                *(gameState+N_POTS) = player;
                sendUSARTbytes(gameState,(N_POTS+1));
                free(gameState);
                sendUSARTbytes(successMsg,sizeof(successMsg));
            }
            else{
                uint8_t cmd = ascii2byte(poppedByte);
                if((cmd>=0) && (cmd<=5)){
                    if(player==PLAYER2){
                        if(*(gameTable+cmd)){
                            selected = cmd;
                            in = selected;
                            sendUSARTbytes(successMsg,sizeof(successMsg));
                        }
                        else{
                            sendUSARTbytes(emptyMsg,sizeof(emptyMsg));
                            in = IDLE_INPUT;
                            selected = IDLE_INPUT;                        
                        }
                    }
                    else{
                        sendUSARTbytes(wrongMsg,sizeof(wrongMsg));
                        in = IDLE_INPUT;
                        selected = IDLE_INPUT;                             
                    }
                }
                else if((cmd>=7) && (cmd<=12)){
                    if(player==PLAYER1){
                        if(*(gameTable+cmd)){
                            selected = cmd;
                            in = selected;
                            sendUSARTbytes(successMsg,sizeof(successMsg));
                        }
                        else{
                            sendUSARTbytes(emptyMsg,sizeof(emptyMsg));
                            in = IDLE_INPUT;
                            selected = IDLE_INPUT;                        
                        }
                    }
                    else{
                        sendUSARTbytes(wrongMsg,sizeof(wrongMsg));
                        in = IDLE_INPUT;
                        selected = IDLE_INPUT;                           
                    }
                }
                else{
                    sendUSARTbytes(errorMsg,sizeof(errorMsg));
                    in = IDLE_INPUT;
                    selected = IDLE_INPUT;
                }
            }
        }
        if(idleTick == TWO_MINUTES){
            idleTick = 0;
            for(uint8_t i=0; i<MAX_LEDS; i++){
                setLEDs(0);
            }
            sleep = ASLEEP;
            selected = IDLE_INPUT;
            PORTA.OUT = 0;
            PORTA.PIN1CTRL = 0x05;
        }
        
        TCB0.INTFLAGS |= 0x01;
    }

}