/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2019 Ha Thach (tinyusb.org)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "bsp/board_api.h"
#include "tusb.h"
#include "usb_descriptors.h"
#include "xgb_capture.h"

#include "ch32v30x.h"

#ifndef MIN
  #define MIN(_a, _b) (((_a) < (_b)) ? (_a) : (_b))
#endif

static uint16_t* puwBufferBase;


__attribute__((optimize("O3"))) void tud_video_prepare_payload_cb(uint_fast8_t ctl_idx, uint_fast8_t stm_idx, tud_video_payload_request_t* request) 
{
  static int bufIdx = FRAME_HEIGHT;
  static size_t iBufPos = 0;
  static uint8_t* aucLineBuffer = NULL;
  int iTransSize;
  (void)ctl_idx;
  (void)stm_idx;

  /* Offset will be zero at the start of a new frame */
  if (!request->offset) 
  { 
    bufIdx = 0;
    iBufPos = 0;
    aucLineBuffer = (uint8_t*)XGB_CAPTURE_GET_LINE(puwBufferBase, 0);
    while (XGB_CAPTURE_GET_LINE_IDX(aucLineBuffer) != 1) continue; // Sync with gameboy
  }

  if (bufIdx < FRAME_HEIGHT)
  {
    do
    {
      iTransSize = MIN((480 - iBufPos), request->length);
      memcpy(request->buf, &aucLineBuffer[iBufPos], iTransSize);
      request->buf += iTransSize;
      request->length -= iTransSize;
      iBufPos += iTransSize;
      if (iBufPos >= 480) 
      {
        iBufPos = 0;
        bufIdx++;
        aucLineBuffer = (uint8_t*)XGB_CAPTURE_GET_LINE(puwBufferBase, bufIdx%XGB_CAPTURE_BUFFERED_LINES_NUM);
        while (XGB_CAPTURE_GET_LINE_IDX(aucLineBuffer) != (bufIdx + 1)) continue; 
      }
    } while (request->length > 0);
  }

}

static uint32_t SysTick_Config(uint32_t ticks) {
  NVIC_EnableIRQ(SysTicK_IRQn);
  SysTick->CTLR = 0;
  SysTick->SR = 0;
  SysTick->CNT = 0;
  SysTick->CMP = ticks - 1;
  SysTick->CTLR = 0xF;
  return 0;
}

static void clk_init(void) { // TODO Used instead of board init, maybe there is a more elegant way

  /* Disable interrupts during init */
  __disable_irq();

  SysTick_Config(SystemCoreClock / 1000);

#ifdef CH32V30x_D8C
  // v305/v307: Highspeed USB
  RCC_USBCLK48MConfig(RCC_USBCLK48MCLKSource_USBPHY);
  RCC_USBHSPLLCLKConfig(RCC_HSBHSPLLCLKSource_HSE);
  RCC_USBHSConfig(RCC_USBPLL_Div2);
  RCC_USBHSPLLCKREFCLKConfig(RCC_USBHSPLLCKREFCLK_4M);
  RCC_USBHSPHYPLLALIVEcmd(ENABLE);
  RCC_AHBPeriphClockCmd(RCC_AHBPeriph_USBHS, ENABLE);
#endif

  // Fullspeed USB
  uint8_t otg_div;
  switch (SystemCoreClock) {
    case 48000000:  otg_div = RCC_OTGFSCLKSource_PLLCLK_Div1; break;
    case 96000000:  otg_div = RCC_OTGFSCLKSource_PLLCLK_Div2; break;
    case 144000000: otg_div = RCC_OTGFSCLKSource_PLLCLK_Div3; break;
    default: TU_ASSERT(0,); break;
  }
  RCC_OTGFSCLKConfig(otg_div);
  RCC_AHBPeriphClockCmd(RCC_AHBPeriph_OTG_FS, ENABLE);

  /* Enable interrupts globally */
  __enable_irq();

  board_delay(2);
}



//--------------------------------------------------------------------+
// Main
//--------------------------------------------------------------------+
int main(void) 
{

  clk_init();

  puwBufferBase = xgb_capture_init();

  // init device stack on configured roothub port
  tusb_rhport_init_t dev_init = 
  {
    .role = TUSB_ROLE_DEVICE,
    .speed = TUSB_SPEED_AUTO
  };
  tusb_init(BOARD_TUD_RHPORT, &dev_init);

  while (1) 
  {
    tud_task(); // tinyusb device task
    tud_video_n_frame_xfer(0, 0, NULL, FRAME_WIDTH * FRAME_HEIGHT * sizeof(uint16_t));
  }

}