/* 
 * SPDX-License-Identifier: GPL-3.0-or-later
 *
 * Copyright (C) 2025 T. Wirtl (embedded-ideas.de)
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

/* There are not many possibilities for the connections to the MCU, therefore the driver uses a fixed scheme 
 * 
 * Exclusively required peripherals
 * - DMA1 (CH5)
 * - TIM4 (CH3)
 * - EXTI3
 * - PORTA
 *
 * RGB555 data lines PA0-PA14
 * - PA0  = R0
 * - PA1  = R1
 * - ...
 * - PA5  = G0
 * - PA6  = G1
 * - ...
 * - PA10 = B0
 * - PA11 = B1
 * - ...
 * Control lines
 * - VSYNC  PB3
 * - PCLK   PB8

 */


#include <stdint.h>
#include "ch32v30x.h"
#include "xgb_capture.h"

#define XGB_CAPTURE_OUTPUT_YUV2 1

#ifndef uint
  typedef unsigned int uint;
#endif
#if XGB_CAPTURE_OUTPUT_YUV2
  #define XGB_CAPTURE_C_
  #include "rgb555_yuv_lut.h"
  #undef XGB_CAPTURE_C_
#endif

static uint16_t aaucBuffer[XGB_CAPTURE_BUFFERED_LINES_NUM][XGB_CAPTURE_WIDTH + 2]; // One additional pixel + line index
static volatile uint uiLineCounter = 0;
static volatile uint uiErr = 0;
static volatile uint uiFramecounter = 0;

void DMA1_Channel5_IRQHandler(void) __attribute__((interrupt));
void EXTI3_IRQHandler(void) __attribute__((interrupt));

void DMA1_Channel5_IRQHandler(void)
{
    
  // Setup the new DMA transfer
  DMA1_Channel5->CFGR &= ~DMA_CFGR1_EN;
  DMA1_Channel5->CNTR = XGB_CAPTURE_WIDTH + 1; // There is one extra pulse (extra pixel) per line
  DMA1_Channel5->MADDR = (uint32_t)aaucBuffer[(uiLineCounter + 1) % XGB_CAPTURE_BUFFERED_LINES_NUM];
  DMA1_Channel5->CFGR |= DMA_CFGR1_EN;
  DMA1->INTFCR = DMA_CTCIF5 | DMA_CGIF5 | DMA_HTIF5;
    
  // Edit data
  for (int i = 0; i < XGB_CAPTURE_WIDTH; i+=sizeof(uint16_t)) 
  {
    uint16_t* uwLine = &aaucBuffer[uiLineCounter % XGB_CAPTURE_BUFFERED_LINES_NUM][1]; // First pixel is extra pulse
    const sYuv* psYuv1 = &rgb555_to_yuv[uwLine[i + 0]];
    const sYuv* psYuv2 = &rgb555_to_yuv[uwLine[i + 1]];
    uwLine[i + 0] = ((psYuv1->ucU+psYuv2->ucU)>>1)<<8;
    uwLine[i + 0] |= psYuv1->ucY;
    uwLine[i + 1] = ((psYuv1->ucV+psYuv2->ucV)>>1)<<8;
    uwLine[i + 1] |= psYuv2->ucY;
  }

  // Release line data for user (indicated by line number)
  aaucBuffer[uiLineCounter % XGB_CAPTURE_BUFFERED_LINES_NUM][241] = uiLineCounter + 1; // store the line number with the data (one based 1-160)
  uiLineCounter++;

}

void EXTI3_IRQHandler(void) // PB3 EXTI handler (VSYNC)
{
  uiFramecounter++;
  if (uiLineCounter != 0xA1) uiErr++;
  uiLineCounter = 0;
  for (int i = 0; i < XGB_CAPTURE_BUFFERED_LINES_NUM; i++) aaucBuffer[i][XGB_CAPTURE_WIDTH + 1] = 0; // Reset line numbers in buffer
  DMA1_Channel5->CFGR &= ~DMA_CFGR1_EN; 
  DMA1_Channel5->CNTR = XGB_CAPTURE_WIDTH + 1; // There is one extra pulse (extra pixel) per line
  DMA1_Channel5->MADDR = (uint32_t)aaucBuffer[0];
  DMA1_Channel5->CFGR |= DMA_CFGR1_EN;
  EXTI->INTFR = EXTI_INTF_INTF3;
}

uint16_t* xgb_capture_init(void)
{
    
  RCC_AHBPeriphClockCmd(RCC_AHBPeriph_DMA1, ENABLE);
  RCC_APB1PeriphClockCmd(RCC_APB1Periph_TIM4, ENABLE);
  RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIOA | RCC_APB2Periph_GPIOB | RCC_APB2Periph_AFIO, ENABLE);

  // DATA IO Config (All floating except A15 --> pull-down)
  AFIO->PCFR1 |= (0x4 << 24);   // 0b100: SWD disabled

  GPIOA->CFGLR =    GPIO_CFGLR_CNF0_0 | GPIO_CFGLR_CNF1_0 | GPIO_CFGLR_CNF2_0 | GPIO_CFGLR_CNF3_0
                  | GPIO_CFGLR_CNF4_0 | GPIO_CFGLR_CNF5_0 | GPIO_CFGLR_CNF6_0 | GPIO_CFGLR_CNF7_0;
    
  GPIOA->CFGHR =    GPIO_CFGHR_CNF8_0  | GPIO_CFGHR_CNF9_0  | GPIO_CFGHR_CNF10_0 | GPIO_CFGHR_CNF11_0
                  | GPIO_CFGHR_CNF12_0 | GPIO_CFGHR_CNF13_0 | GPIO_CFGHR_CNF14_0 | GPIO_CFGHR_CNF15_1;

  GPIOA->OUTDR = 0;; // pull-down if output resistors are enable for the pin


  // TIM4 CH3 Input (PB8)
  GPIOB->CFGLR &= ~(GPIO_CFGHR_MODE8 | GPIO_CFGHR_CNF8);
  GPIOB->CFGLR |= GPIO_CFGHR_CNF8_0;

  TIM4->DMAINTENR |= TIM_CC3DE;           // DMA request enable
  TIM4->CHCTLR2   |= TIM_CC3S_0;          // Capture mode (input) and enable <-- Do before setting TIM_CC1E!
  TIM4->CCER      |= TIM_CC3P | TIM_CC3E; // low active and enable capture
  TIM4->CTLR1 = TIM_CEN; // Enable

  DMA1->INTFR |= DMA_TCIF5 | DMA_GIF5; // DMA Interrupt enable
  NVIC_EnableIRQ(DMA1_Channel5_IRQn);

  DMA1_Channel5->CNTR = XGB_CAPTURE_WIDTH + 1;
  DMA1_Channel5->PADDR = (uint32_t)&GPIOA->INDR;
  DMA1_Channel5->MADDR = (uint32_t)aaucBuffer[0];
  DMA1_Channel5->CFGR = DMA_CFGR1_MINC | DMA_CFGR1_TCIE | DMA_CFGR1_MSIZE_0 | DMA_CFGR1_PSIZE_0;

  AFIO->EXTICR[0] &= ~AFIO_EXTICR1_EXTI3;
  AFIO->EXTICR[0] |= AFIO_EXTICR1_EXTI3_PB;
    
  EXTI->RTENR |= EXTI_RTENR_TR3; // Trigger on rising
  EXTI->INTENR |= EXTI_INTENR_MR3;
  NVIC_EnableIRQ(EXTI3_IRQn);

  return &aaucBuffer[0][0];
}
